/* checkmateIfCapture.t.cc
 */
#include "osl/checkmate/checkmateIfCapture.h"
#include "osl/checkmate/immediateCheckmate.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/move_generator/legalMoves.h"
#include "osl/container/moveVector.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

namespace osl
{
  namespace checkmate
  {
    class CheckmateIfCaptureTest : public CppUnit::TestFixture 
    {
      CPPUNIT_TEST_SUITE(CheckmateIfCaptureTest);
      // ここに(テストのための)メソッドを列挙する．
      CPPUNIT_TEST(testAttack);
      CPPUNIT_TEST(testCandidate);
      CPPUNIT_TEST_SUITE_END();
    public:
      void testAttack();
      void testCandidate();

      static void testCandidatePosition(NumEffectState& state);
    };
  }
} // namespace osl

CPPUNIT_TEST_SUITE_REGISTRATION(osl::checkmate::CheckmateIfCaptureTest);

void osl::checkmate::CheckmateIfCaptureTest::testAttack()
{
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -OU *  * \n"
			   "P2 * -HI *  *  *  *  *  *  * \n"
			   "P3 *  *  *  *  * +TO+TO *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  *  *  *  * +OU *  *  * \n"
			   "P+00GI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m83(Position(8,3), SILVER, BLACK);
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m83, 0));

    const Move m73(Position(7,3), SILVER, BLACK);
    CPPUNIT_ASSERT(! CheckmateIfCapture::effectiveAttack(state, m73, 0));

    // 除きたい?
    const Move m84(Position(8,4), SILVER, BLACK);
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m84, 0));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  * -OU *  * \n"
			   "P2 * -HI *  *  *  *  *  *  * \n"
			   "P3 *  *  *  *  * +TO+TO *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 * +OU *  *  *  *  *  *  * \n"
			   "P+00GI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m83(Position(8,3), SILVER, BLACK);
    // 取ると王手
    CPPUNIT_ASSERT(! CheckmateIfCapture::effectiveAttack(state, m83, 0));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  * -GI-OU-GI * \n"
			   "P2 * -HI *  *  *  *  *  *  * \n"
			   "P3 *  *  *  *  * +TO+TO *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  *  *  *  * +OU *  *  * \n"
			   "P+00GI00GI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m83(Position(8,3), SILVER, BLACK);
    CPPUNIT_ASSERT(! CheckmateIfCapture::effectiveAttack(state, m83, 0));

    // 3手詰
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m83, 2));

    const Move m73(Position(7,3), SILVER, BLACK);
    CPPUNIT_ASSERT(! CheckmateIfCapture::effectiveAttack(state, m73, 0));

    // 除きたい?
    const Move m84(Position(8,4), SILVER, BLACK);
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m84, 2));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  * -KE-KY\n"
			   "P2 *  *  *  *  *  *  * -OU * \n"
			   "P3 *  *  *  *  * +TO-FU-FU * \n"
			   "P4 *  *  *  *  *  *  *  * -FU\n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  *  *  *  *  * \n"
			   "P9 *  *  *  *  * +OU *  *  * \n"
			   "P+00KA00KI\n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m31(Position(3,1), BISHOP, BLACK);
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m31, 0));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  *  *  *  *  *  *  *  * \n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 *  *  *  *  *  *  *  *  * \n"
			   "P4 *  *  *  *  *  *  *  *  * \n"
			   "P5 *  *  *  *  *  *  *  *  * \n"
			   "P6 *  *  *  *  * -OU *  *  * \n"
			   "P7 *  *  *  *  *  *  *  *  * \n"
			   "P8 *  *  *  *  * +OU *  *  * \n"
			   "P9 *  *  *  *  *  *  *  *  * \n"
			   "P-00AL\n"
			   "+\n").getInitialState());
    const Move m47(Position(4,8), Position(4,7), KING, PTYPE_EMPTY, false, BLACK);
    CPPUNIT_ASSERT(! CheckmateIfCapture::effectiveAttack(state, m47, 0));
  }
  {
    NumEffectState state(CsaString(
			   "P1 *  * +HI * -KI-OU * -KE-KY\n"
			   "P2 *  *  *  *  *  *  *  *  * \n"
			   "P3 *  *  * -FU-FU+FU-FU-FU-FU\n"
			   "P4 *  * +KA *  *  *  *  *  * \n"
			   "P5 *  *  *  * +KE *  *  *  * \n"
			   "P6 *  *  *  *  *  *  *  *  * \n"
			   "P7+FU+FU+FU *  *  *  *  *  * \n"
			   "P8+KY+GI+KI *  *  *  *  *  * \n"
			   "P9+OU+KE+KI *  *  *  *  *  * \n"
			   "P+00KI\n"
			   "P-00HI00KA00GI00GI00GI00KE00KY00KY00FU00FU00FU00FU00FU00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    const Move m52(Position(5,2), GOLD, BLACK);
    // 金ではとれない．玉で取ると詰
    CPPUNIT_ASSERT(CheckmateIfCapture::effectiveAttack(state, m52, 0));
  }
}

void osl::checkmate::CheckmateIfCaptureTest::testCandidatePosition(NumEffectState& state)
{
  if (! state.inCheck() 
      && ImmediateCheckmate::hasCheckmateMove(state.getTurn(), state))
    return;
  MoveVector moves;
  LegalMoves::generate(state, moves);
  for (size_t i=0; i<moves.size(); ++i) {
    if (! state.hasEffectBy(alt(state.getTurn()), moves[i].to()))
      continue;			// specification of rating::CheckmateIfCapture
    const bool effective = CheckmateIfCapture::effectiveAttack(state, moves[i], 0);
    const bool predicted = CheckmateIfCapture::effectiveAttackCandidate0(state, moves[i]);
    if (effective && ! predicted)
      std::cerr << state << moves[i] << "\n";
    CPPUNIT_ASSERT(! effective || predicted);
  }
}

void osl::checkmate::CheckmateIfCaptureTest::testCandidate()
{
  {
    NumEffectState state(CsaString(
			   "P1-KY * +KA * +UM *  *  * -KY\n"
			   "P2 *  *  *  * -GI * -KI *  * \n"
			   "P3-FU *  * -GI * -HI-KE * -FU\n"
			   "P4 *  *  *  *  * -KI-FU-FU-OU\n"
			   "P5 * -FU-FU *  *  *  *  * +KE\n"
			   "P6 *  * +FU+FU *  * +FU * +GI\n"
			   "P7+FU+FU+GI-FU * -RY *  * +FU\n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00FU\n"
			   "P-00KI00KE00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state);
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY * +KA *  *  *  *  * -KY\n"
			   "P2 *  *  *  * -GI * -KI *  * \n"
			   "P3-FU *  * -GI * -HI+UM * -FU\n"
			   "P4 *  *  *  *  *  * -FU-FU-OU\n"
			   "P5 * -FU-FU *  *  * -KI * +KE\n"
			   "P6 *  * +FU+FU *  * +FU * +GI\n"
			   "P7+FU+FU+GI-FU * -RY *  * +FU\n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00KE00FU\n"
			   "P-00KI00KE00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state);
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY * +KA *  *  *  *  * +UM\n"
			   "P2 *  *  *  * -GI * -KI *  * \n"
			   "P3-FU *  * -GI * -HI *  * -FU\n"
			   "P4 *  *  *  *  * -KI-FU-FU-OU\n"
			   "P5 * -FU *  *  *  *  *  * +KE\n"
			   "P6 *  * -FU+FU *  * +FU * +GI\n"
			   "P7+FU+FU+GI-FU *  * -RY * +FU\n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00KE00KY00FU\n"
			   "P-00KI00KE00FU00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state);
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY * +KA *  *  *  *  * -KY\n"
			   "P2 *  *  *  * -GI *  *  *  * \n"
			   "P3-FU *  * -GI *  *  * -HI-FU\n"
			   "P4 *  *  *  *  * -KI-FU-FU-OU\n"
			   "P5 * -FU-FU *  *  *  *  * +KE\n"
			   "P6 *  * +FU+FU *  * +FU-KE+GI\n"
			   "P7+FU+FU+GI-FU * -RY *  * +FU\n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00KI00KI00KE00FU\n"
			   "P-00KA00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state);
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY *  *  *  *  *  *  * -KY\n"
			   "P2 *  *  *  * -GI * +UM *  * \n"
			   "P3-FU *  * -GI * -HI * -KE-FU\n"
			   "P4 *  *  *  *  * +UM-FU-FU-OU\n"
			   "P5 * -FU-FU *  *  *  * -KI+KE\n"
			   "P6 *  * +FU+FU *  * -RY * +GI\n"
			   "P7+FU+FU+GI-FU *  *  *  * +FU\n"
			   "P8 * +OU+KI *  *  *  *  *  * \n"
			   "P9+KY+KE *  *  *  *  *  * +KY\n"
			   "P+00KI00KI00KE00FU\n"
			   "P-00FU00FU00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state); // 43um
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY-KE *  * -HI * +UM-KE-KY\n"
			   "P2 *  *  *  *  *  * +KI * -OU\n"
			   "P3 *  * -GI *  * -KI * -KI-FU\n"
			   "P4-FU * -FU * -FU * -FU-GI * \n"
			   "P5 * +FU *  *  * -FU *  *  * \n"
			   "P6 *  * +FU *  *  *  *  *  * \n"
			   "P7+FU *  *  * -UM+FU+FU+FU+FU\n"
			   "P8 *  *  * -RY * -KY+GI+OU * \n"
			   "P9 *  *  *  *  *  * +KI+KE+KY\n"
			   "P+00GI00FU00FU\n"
			   "P-00KE00FU00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state); // 52gi
  }
  {
    NumEffectState state(CsaString(
			   "P1-KY *  *  *  * -KI-KI *  * \n"
			   "P2 * -HI *  *  *  * -OU+GI * \n"
			   "P3 *  *  *  * -GI * +KA-FU+NK\n"
			   "P4-FU * -FU * -FU-FU-GI * -FU\n"
			   "P5 * -FU * -KE *  * -FU *  * \n"
			   "P6+FU * +FU-UM+FU+FU * +FU * \n"
			   "P7 * +FU *  * -TO * +FU+GI * \n"
			   "P8 *  * -RY *  * +KY+KI+OU * \n"
			   "P9+KY *  *  *  *  *  *  *  * \n"
			   "P+00KI\n"
			   "P-00KE00KE00KY00FU00FU\n"
			   "+\n").getInitialState());
    testCandidatePosition(state); // 15ka
  }
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string filename;
  for(int i=0;i<100 && (ifs >> filename) ; i++){
    const Record record=CsaFile(OslConfig::testCsaFile(filename)).getRecord();
    NumEffectState state(record.getInitialState());
    if (i % 32 == 0) 
      std::cerr << '.';
    const vector<Move> moves=record.getMoves();
    for (size_t i=0; i<moves.size(); ++i) {
      testCandidatePosition(state);
      ApplyMoveOfTurn::doMove(state, moves[i]);
    }
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
