/* breakThreatmate.cc
 */
#include "osl/search/breakThreatmate.h"
#include "osl/container/moveLogProbVector.h"
#include "osl/move_generator/addEffectWithEffect.h"
#include "osl/move_generator/capture_.h"
#include "osl/move_generator/pieceOnBoard.h"
#include "osl/move_generator/escape_.h"
#include "osl/move_action/store.h"
#include "osl/checkmate/king8Info.h"
#include "osl/boardTable.h"
#include "osl/ptypeTable.h"
#include "osl/neighboring8.h"
#include <boost/foreach.hpp>

void osl::search::
BreakThreatmate::generateBreakDrop(int limit,
				   const NumEffectState& state, Position to,
				   int default_prob,
				   MoveLogProbVector& out)
{
  const Player Turn = state.getTurn();
  default_prob = std::max(default_prob, 150);
  if (state.hasPieceOnStand<PAWN>(Turn))
  {
    if (! state.isPawnMaskSet(Turn, to.x())
	&& Ptype_Table.canDropTo(Turn, PAWN, to)
	&& limit >= default_prob-100)
      out.push_back(MoveLogProb(Move(to, PAWN, Turn), default_prob-100));
  }
  if (state.hasPieceOnStand<LANCE>(Turn)
      && Ptype_Table.canDropTo(Turn, LANCE, to)
      && limit >= default_prob-50)
    out.push_back(MoveLogProb(Move(to, LANCE, Turn), default_prob-50));
  if (default_prob > limit)
    return;

  if (state.hasPieceOnStand<KNIGHT>(Turn)
      && Ptype_Table.canDropTo(Turn, KNIGHT, to))
    out.push_back(MoveLogProb(Move(to, KNIGHT, Turn), default_prob));
  if (state.hasPieceOnStand<SILVER>(Turn))
    out.push_back(MoveLogProb(Move(to, SILVER, Turn), default_prob));
  if (state.hasPieceOnStand<GOLD>(Turn))
    out.push_back(MoveLogProb(Move(to, GOLD, Turn), default_prob));
  if (state.hasPieceOnStand<BISHOP>(Turn))
    out.push_back(MoveLogProb(Move(to, BISHOP, Turn), default_prob));
  if (state.hasPieceOnStand<ROOK>(Turn))
    out.push_back(MoveLogProb(Move(to, ROOK, Turn), default_prob));
}

void osl::search::BreakThreatmate::findBlockLong(const NumEffectState& state, Move threatmate_move,
						 MoveVector& out)
{
  const Player Turn = alt(threatmate_move.player());
  const Piece my_king = state.getKingPiece(Turn);
  const Position target = threatmate_move.to();
  out.clear();
  move_action::Store store(out);
  // block additional
  if (! threatmate_move.isDrop()
      && threatmate_move.oldPtype() != KNIGHT)
  {
    const Offset step = Board_Table.getShortOffsetNotKnight(Offset32(threatmate_move.from(), target));
    Position p=threatmate_move.from()+step;
    Piece piece=state.getPieceAt(p);
    for (; piece.isEmpty(); p+=step, piece=state.getPieceAt(p))
      ;
    if (piece.isPiece() && piece.owner() == alt(Turn)
	&& state.hasEffectByPiece(piece, threatmate_move.from())) {
      if (Turn == BLACK)
	move_generator::Escape<move_action::Store>::generateBlocking<BLACK,true>
	  (state, my_king, threatmate_move.from(), piece.position(), store);
      else
	move_generator::Escape<move_action::Store>::generateBlocking<WHITE,true>
	  (state, my_king, threatmate_move.from(), piece.position(), store);
    }
  }
  // block long
  {
    mask_t attack = state.selectLong(target, alt(Turn));
    while (attack.any()) {
      BOOST_STATIC_ASSERT(PtypeFuns<LANCE>::indexNum == PtypeFuns<BISHOP>::indexNum);
      BOOST_STATIC_ASSERT(PtypeFuns<LANCE>::indexNum == PtypeFuns<ROOK>::indexNum);
      const int num = (attack.takeOneBit()+((PtypeFuns<LANCE>::indexNum)<<5));
      if (Turn == BLACK)
	move_generator::Escape<move_action::Store>::generateBlocking<BLACK,true>
	  (state, my_king, target, state.getPieceOf(num).position(), store);
      else
	move_generator::Escape<move_action::Store>::generateBlocking<WHITE,true>
	  (state, my_king, target, state.getPieceOf(num).position(), store);
    }
  }
}

void osl::search::BreakThreatmate::
generateAddEffect(int limit, const NumEffectState& state, Position target,
		  const MoveVector& all_moves, MoveLogProbVector& out)
{
  const Player Turn = state.getTurn();
  const int max_prob = state.king8Info(state.getTurn()).liberty() ? limit : 100;
  BOOST_FOREACH(Move move, all_moves)
  {
    const Ptype ptype = move.ptype();
    if (ptype == KING)
      continue;			// KING_WALK will generate
    if (! move.isDrop())
    {
      if (isMajor(ptype)
	  && state.hasEffectByPiece(state.getPieceOnBoard(move.from()), target))
	continue;			// already have effect
    }
    const Position to = move.to();
    const int me = state.countEffect(Turn, to) + (move.isDrop() ? 1 : 0);
    const int op = state.countEffect(alt(Turn), to);
    int prob = (move.isDrop() ? 100 : 100); // delay drop
    if (move.capturePtype() != PTYPE_EMPTY)
    {
      prob -= 50;
    }
    else
    { 
      if (isMajor(ptype)
	  || ((ptype == GOLD || ptype == SILVER)
	      && (to.x() == 1 || to.x() == 9)))
      {
	prob += 50;
      }
      if (! ((me >= 2) || (op == 0)))
      {
	prob += 300;
      }
    }
    prob = std::min(prob, max_prob);
    if (prob <= limit)
      out.push_back(MoveLogProb(move, prob));
  }
}

// yoshiki's suggestion 駒取り 50, 普通 100, 駒捨て 400 
void osl::search::BreakThreatmate::
generate(int limit, const NumEffectState& state, Move threatmate_move,
	 MoveLogProbVector& out)
{
  assert(threatmate_move.isNormal());
  const Player Turn = state.getTurn();
  
  MoveVector all_moves;
  assert(threatmate_move.isNormal());
  const Position target = threatmate_move.to();
  move_generator::GenerateAddEffectWithEffect::generate<false>
      (Turn, state, target, all_moves);
  generateAddEffect(limit, state, target, all_moves, out);

  if (threatmate_move.isDrop())
  {
    const int drop_prob = (state.hasEffectBy(alt(Turn), target) ? 400 : 100);
    generateBreakDrop(limit, state, target, drop_prob, out);
  }
  else
  {
    // not drop
    const Position from = threatmate_move.from();
    const Offset offset
      = Board_Table.getShortOffsetNotKnight(Offset32(target, from));
    if (! offset.zero())
    {
      for (Position to = from + offset; to != target; to += offset) 
      {
	assert(to.isOnBoard());
	assert(state.getPieceOnBoard(to) == Piece::EMPTY());
	const int drop_prob = (state.hasEffectBy(Turn, to) ? 100 : 400);
	generateBreakDrop(limit, state, to, drop_prob, out);

	const int move_prob = (state.hasMultipleEffectBy(Turn, to) ? 100 : 400);
	if (move_prob > limit)
	  continue;
	all_moves.clear();
	move_generator::GenerateCapture::generate
	  (Turn, state, to, all_moves);
 	BOOST_FOREACH(Move move, all_moves)
	{
	  out.push_back(MoveLogProb(move, move_prob));
	}
      }
    }
  }
  const Piece my_king = state.getKingPiece(Turn);
  if (my_king.position()
      != target+Board_Table.getShortOffset(Offset32(my_king.position(),target)))
  {
    const checkmate::King8Info king8info = state.king8Info(Turn);
    unsigned int drop_candidate = king8info.dropCandidate();
    if (drop_candidate) {
      const int d = misc::BitOp::bsf(drop_candidate);
      const Position to = my_king.position()
	+ Board_Table.getOffset(Turn, static_cast<Direction>(d));
      if (to != target) {
	all_moves.clear();
	move_generator::GenerateAddEffectWithEffect::generate<false>
	  (Turn, state, to, all_moves);
	generateAddEffect(limit, state, to, all_moves, out);
      }
    }
  }
  // king walk
  const int king_prob = 100;
  if (king_prob <= limit)
  {
    all_moves.clear();
    {
      GeneratePieceOnBoard::generate(Turn, state, my_king, all_moves);
    }
    BOOST_FOREACH(Move move, all_moves)
    {
      if (state.hasEffectBy(alt(Turn), move.to()))
	continue;
      out.push_back(MoveLogProb(move, king_prob));
    }
  }
  
  // open king road
  const Position center = my_king.position();
  generateOpenRoad(limit, state, center + DirectionTraits<U>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<UL>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<UR>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<L>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<R>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<D>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<DL>::blackOffset(), out);
  generateOpenRoad(limit, state, center + DirectionTraits<DR>::blackOffset(), out);

  // block
  findBlockLong(state, threatmate_move, all_moves);
  if (! all_moves.empty()) 
  {
    Ptype cheapest = PTYPE_EMPTY;
    if (state.hasPieceOnStand<PAWN>(Turn)) cheapest = PAWN;
    else if (state.hasPieceOnStand<LANCE>(Turn)) cheapest = LANCE;
    else if (state.hasPieceOnStand<KNIGHT>(Turn)) cheapest = KNIGHT;
    int added = 0;
    Move chuai_reserve;
    BOOST_FOREACH(Move m, all_moves) {
      const int d = state.countEffect(Turn, m.to()) + m.isDrop();
      const int a = state.countEffect(alt(Turn), m.to());
      if (d == 1
	  || (m.ptype() != cheapest && cheapest != PTYPE_EMPTY
	      && Ptype_Table.canDropTo(Turn, cheapest, m.to())
	      && (cheapest != PAWN || ! state.isPawnMaskSet(m.player(), m.to().x()))))
	continue;
      if (a >= d) {
	if (! chuai_reserve.isNormal()) 
	  chuai_reserve = m;
	continue;
      }
      const int prob = 150+added*50;
      if (prob > limit)
	break;
      out.push_back(MoveLogProb(m, prob));
    }
    if (added == 0 && chuai_reserve.isNormal() && limit >= 250) {
      out.push_back(MoveLogProb(chuai_reserve, 250));
      if (chuai_reserve.isDrop() && chuai_reserve.ptype() == KNIGHT && 300 <= limit) {
	if (state.hasPieceOnStand<SILVER>(Turn))
	  out.push_back(MoveLogProb(Move(chuai_reserve.to(),SILVER,Turn), 300));
	else if (state.hasPieceOnStand<GOLD>(Turn))
	  out.push_back(MoveLogProb(Move(chuai_reserve.to(),GOLD,Turn), 300));
	else if (state.hasPieceOnStand<BISHOP>(Turn))
	  out.push_back(MoveLogProb(Move(chuai_reserve.to(),BISHOP,Turn), 300));
	else if (state.hasPieceOnStand<ROOK>(Turn))
	  out.push_back(MoveLogProb(Move(chuai_reserve.to(),ROOK,Turn), 300));
      }
    }
  }
}

void osl::search::BreakThreatmate::
generateOpenRoad(int limit, const NumEffectState& state, 
		 Position from, MoveLogProbVector& out)
{
  const Piece target = state.getPieceAt(from);
  if (! target.isPiece())
    return;
  const Player Turn = state.getTurn();
  if (target.owner() != Turn)
    return;

  const int capture_prob = 50;
  const int default_prob = 100;
  const int sacrifice_prob = 400;
  if (limit < capture_prob)
    return;

  MoveVector moves;
  GeneratePieceOnBoard::generate(Turn, state, target, moves);

  BOOST_FOREACH(Move move, moves)
  {
    const bool capture = (move.capturePtype() != PTYPE_EMPTY);
    const bool sacrifice = state.hasEffectBy(alt(Turn), move.to());
    const int prob = capture ? capture_prob
      : (sacrifice ? sacrifice_prob : default_prob);
    if (prob <= limit)
      out.push_back(MoveLogProb(move, prob));
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
