#ifndef OSL_MOVE_GENERATOR_ADD_EFFECT_H
#define OSL_MOVE_GENERATOR_ADD_EFFECT_H
#include "osl/move_generator/drop.h"
#include "osl/move_generator/addEffectTable.h"
#include "osl/move_action/store.h"
#include "osl/container/moveVector.h"
#include <boost/static_assert.hpp>

namespace osl
{
  namespace move_generator
  {
    /**
     * AddEffect の部品 足の短い駒のケース.
     */
    template<Player P,Ptype T,bool isAttackToKing=false>
    struct AddEffectShort
    {
      BOOST_STATIC_ASSERT((PtypeTraits<T>::isBasic));
      BOOST_STATIC_ASSERT((PtypeTraits<T>::canPromote));
      /**
       * 足の短い駒による利きの生成
       * @param P(template) - 攻撃側のプレイヤー
       * @param T(template) - Ptype T およびpromoteした駒による利きの生成
       * @param state - 盤面
       * @param target - 利きをつけられるマス
       * @param action - 手がある時のcallback関数
       * @param nearMask - targetの10近傍のマスク
       */
      template<class Action>
      static void generate(const NumEffectState& state,Position target,Action& action,NearMask nearMask);
    };
    
    /**
     * AddEffect の部品 足の長い駒のケース.
     */
    template<Player P,Ptype T,bool isAttackToKing=false>
    struct AddEffectLong
    {
      BOOST_STATIC_ASSERT((PtypeTraits<T>::isBasic));
      /**
       * 長い利きを持つもの
       */
      template<class Action>
      static void generate(const NumEffectState& state,Position target,Action& action,NearMask nearMask);
    };
    

    /**
     * 利きがないstateで使えるAddEffect.
     * isAttackToKingがtrueの場合はKINGによる王手は自殺手なので生成しない
     * 自分の手が自殺手かどうかのチェックをするか？
     * isAttackToKing がfalseの時は，影利き，追加利きも生成する
     * 扱う影利き，追加利きは間に駒が1個まで
     * isAttackToKingがfalseの時は，影利き，追加利きかつopen attackを重複生成してしまうことがある．
     * アルゴリズム:
     * \li 最初に利きをつけたいマスの10近傍(8近傍+桂馬)の状態(空白，敵駒)を調べる
     * \li 40個すべての駒に関して，盤上の自分の駒の場合は
     *   利きをつけたいマスとの位置の差分(Offset32)を計算し，テーブルを参照する
     * \li マスの10近傍の状態と比較することにより，その駒を使った利きを付ける手
     *   が絞られる
     * 特徴:
     * \li 平均的には利きのあるstateで使ってもAddEffectWithEffectより速い
     * \li 相手玉の自由度がほとんどない場合などはAddEffectWithEffectの方が速いこともある．
     */
    template<Player P,bool isAttackToKing=false>
    class AddEffect
    {
      // BOOST_CLASS_REQUIRE(Action,osl::move_action,Concept);
    public:
      /**
       * promoteとは関係ないやつ
       */
      template<class Action>
      static void generateKing(const NumEffectState& state,Position target,Action& action,NearMask nearMask);
      template<class Action>
      static void generateGold(const NumEffectState& state,Position target,Action& action,NearMask nearMask);
      /**
       * 対象とするマスに利きを付ける手を生成する
       * @param P(template) - 手番のプレイヤー
       * @param state - 対象とする局面
       * @param target - 利きをつけたいマス
       * @param action - 手が存在した時に呼び出すコールバック関数
       */
      template<class Action>
      static void generate(const NumEffectState& state,Position target,Action& action);
      static void generate(const NumEffectState& state,Position target,MoveVector& out)
      {
	move_action::Store store(out);
	generate(state, target, store);
      }
    };

    template<bool isAttackToKing=false>
    struct GenerateAddEffect
    {
      /**
       * 対象とするマスに利きを付ける手を生成する.
       * 内部でSafeFilter をかける．
       * Note: Action がPlayerで型付けされていると，無駄があるのでStore限定に
       */
      static void generate(Player player, const NumEffectState& state, Position target, 
			   move_action::Store& store);
      static void generate(Player player, const NumEffectState& state, Position target, 
			   MoveVector& out)
      {
	move_action::Store store(out);
	generate(player, state, target, store);
      }
    };
  } // namespace move_generator
} // namespace osl
#endif /* OSL_MOVE_GENERATOR_ADD_EFFECT_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
