#ifndef _MASK_H
#define _MASK_H
#include "osl/misc/bitOp.h"
#include "osl/misc/cstdint.h"
#include "osl/config.h"
#include <cassert>
#include <iosfwd>

namespace osl
{
  namespace misc
  {
    template <class Integer>
    class GeneralMask
    {
      Integer mask;
    private:
      GeneralMask(Integer value) : mask(value) {}
    public:
      GeneralMask() : mask(0) {}
      static const GeneralMask makeDirect(Integer value) { return GeneralMask(value); }
      GeneralMask& operator&=(const GeneralMask& r)
      {
	mask &= r.mask;
	return *this;
      }
      GeneralMask& operator|=(const GeneralMask& r)
      {
	mask |= r.mask;
	return *this;
      }
      GeneralMask& operator^=(const GeneralMask& r)
      {
	mask ^= r.mask;
	return *this;
      }
      GeneralMask& operator-=(const GeneralMask& r)
      {
	mask -= r.mask;
	return *this;
      }
      GeneralMask& operator+=(const GeneralMask& r)
      {
	mask += r.mask;
	return *this;
      }
      GeneralMask& operator<<=(int shift)
      {
	mask <<= shift;
	return *this;
      }
      GeneralMask& operator>>=(int shift)
      {
	mask >>= shift;
	return *this;
      }
      const GeneralMask operator~() const { return GeneralMask(~mask); }

      int bsf() const { return BitOp::bsf(mask); }
      int bsr() const { return BitOp::bsr(mask); }
      /**
       * non-zeroのmaskのsetされているビットをLSBから探し，その番号を返す
       * 副作用としてmaskの対応するビットをクリアする
       * @param mask - 対象とするデータ(non-zero)
       * @return - どのビットか
       */
      int takeOneBit() { return BitOp::takeOneBit(mask); }
  
      /**
       * non-zeroのmaskが複数ビットセットされているかどうかを返す.
       * @param mask - 対象とするデータ(non-zero)
       * @return - 複数ビットがセットされているか？
       */
      bool hasMultipleBit() const { return BitOp::hasMultipleBit(mask); }
      /**
       * non-zeroのmaskにセットされているビットの数を2まで数える．
       * @param mask - 対象とするデータ(non-zero)
       * @return 1,2 (2の場合は2以上)
       */
      int countBit2() const {
	assert(mask);
	return (mask & (mask-1)) ? 2 : 1;
      }
      /**
       * mask にセットされているビットの数を数える．
       * あまり速くない．
       */
      int countBit() const { return BitOp::countBit(mask); }
      /**
       * non-zeroのmaskのsetされているビットをLSBから探し，そのビットだけがsetされたmaskを返す.
       * @param mask - 対象とするデータ(non-zero)
       * @return - そのビットだけがsetされたmask
       */
      GeneralMask lowestBit() const { return BitOp::lowestBit(mask); }
      bool none() const { return mask == 0; }
      bool any() const { return ! none(); }
      Integer value() const { return mask; }
    };

    template <class Integer> inline
    bool operator==(const GeneralMask<Integer>& l, const GeneralMask<Integer>& r)
    {
      return l.value() == r.value();
    }
    template <class Integer> inline
    bool operator!=(const GeneralMask<Integer>& l, const GeneralMask<Integer>& r)
    {
      return ! (l == r);
    }
    template <class Integer> inline
    bool operator<(const GeneralMask<Integer>& l, const GeneralMask<Integer>& r)
    {
      return l.value() < r.value();
    }

    template <class Integer> inline
    const GeneralMask<Integer> operator&(GeneralMask<Integer> l,
					GeneralMask<Integer> r) {
      GeneralMask<Integer> result = l;
      return result &= r;
    }
    template <class Integer> inline
    const GeneralMask<Integer> operator|(GeneralMask<Integer> l,
					GeneralMask<Integer> r) {
      GeneralMask<Integer> result = l;
      return result |= r;
    }
    template <class Integer> inline
    const GeneralMask<Integer> operator^(GeneralMask<Integer> l,
					GeneralMask<Integer> r) {
      GeneralMask<Integer> result = l;
      return result ^= r;
    }
    template <class Integer> inline
    const GeneralMask<Integer> operator<<(GeneralMask<Integer> m, int shift) {
      GeneralMask<Integer> result = m;
      return result <<= shift;
    }
    template <class Integer> inline
    const GeneralMask<Integer> operator>>(GeneralMask<Integer> m, int shift) {
      GeneralMask<Integer> result = m;
      return result >>= shift;
    }
  
    typedef GeneralMask<unsigned long long> Mask64;
    typedef GeneralMask<unsigned int> Mask32;


#if OSL_WORDSIZE == 64
    typedef unsigned long long mask_int_t;
#elif OSL_WORDSIZE == 32
    typedef unsigned int mask_int_t;
#endif
    typedef GeneralMask<mask_int_t> mask_t;

    std::ostream& operator<<(std::ostream&, const mask_t&);
  } // namespace misc
  using misc::mask_int_t;
  using misc::mask_t;
} // namespace osl

#endif
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
