/* carray.h
 */
#ifndef _CARRAY_H
#define _CARRAY_H
#include "osl/player.h"
#include "osl/config.h"
#include <cstddef>
#include <cassert>
#include <algorithm>
#include <iterator>
#include <boost/type_traits.hpp>

#define CONSERVATIVE_PLAYER_ACCESS

namespace osl
{
  namespace misc
  {
  /**
   * CArray の iterator.
   * 単純にT*を使うと， if (iter == end()) と書くべきところで
   * ポインタのつもりで if (iter) と書いてしまっても気付けないため．
   * TODO: boost にない?
   */
  template <typename T>
  struct CArrayIterator
  {
    typedef std::random_access_iterator_tag iterator_category;
    typedef T   value_type;
    typedef int difference_type;
    typedef T*  pointer;
    typedef T&  reference;

    T *ptr;
    CArrayIterator(T *p) : ptr(p) {}
    CArrayIterator(const CArrayIterator<typename boost::remove_cv<T>::type>& src) : ptr(src.ptr)
    {
    }
    T& operator*() const { return *ptr; }
    T* operator->() const { return ptr; }
    CArrayIterator& operator+=(int diff)
    {
      ptr += diff;
      return *this;
    }
    CArrayIterator& operator-=(int diff) { return operator+=(-diff); }
    CArrayIterator& operator++() { return operator+=(1); }
    CArrayIterator operator++(int)
    {
      const CArrayIterator result = *this;
      operator++();
      return result;
    }
    CArrayIterator& operator--() { return operator+=(-1); }
    CArrayIterator operator--(int)
    {
      const CArrayIterator result = *this;
      operator--();
      return result;
    }
  private:
#ifndef _MSC_VER
    operator bool();		// not implemented
#endif
  };
  template <class T> inline
  const CArrayIterator<T> operator+(const CArrayIterator<T>& iter, int diff) 
  {
    CArrayIterator<T> result(iter);
    result += diff;
    return result;
  }
  template <class T> inline
  const CArrayIterator<T> operator-(const CArrayIterator<T>& iter, int diff) { 
    return iter + (-diff); 
  }
  // T と const T の違いを吸収，それ以外はcompile error になるはず
  template <class T, class T2>
  inline int operator-(CArrayIterator<T> l, CArrayIterator<T2> r)
  {
    return l.ptr - r.ptr;
  }
  template <class T, class T2>
  inline bool operator==(CArrayIterator<T> l, CArrayIterator<T2> r)
  {
    return l.ptr == r.ptr;
  }
  template <class T, class T2>
  inline bool operator!=(CArrayIterator<T> l, CArrayIterator<T2> r)
  {
    return l.ptr != r.ptr;
  }
  template <class T, class T2>
  inline bool operator<(CArrayIterator<T> l, CArrayIterator<T2> r)
  {
    return l.ptr < r.ptr;
  }
  template <class T, class T2>
  inline bool operator>(CArrayIterator<T> l, CArrayIterator<T2> r)
  {
    return l.ptr > r.ptr;
  }

  /**
   * boost::array のまね。operator[] に assert をいれたかったので。
   */
  template <typename T, size_t Capacity>
  class CArray
  {
  public:
    /** {} による初期化を許すために public にしておく */
    T elements[Capacity];
    typedef typename boost::remove_cv<T>::type T_simple;
  public:
    typedef T value_type;
    typedef CArrayIterator<T> iterator;
    iterator begin() { return &elements[0]; }
    iterator end() { return &elements[Capacity]; }

    void fill(T_simple value=T_simple())
    {
      std::fill(begin(), end(), value);
    }
    T& operator[] (size_t i) 
    {
      assert(i < Capacity);
      return elements[i];
    }

    static size_t size() { return Capacity; }
    
    T const& operator[] (size_t i) const
    {
      assert(i < Capacity);
      return elements[i];
    }

    typedef CArrayIterator<const T> const_iterator;
    const_iterator begin() const { return &elements[0]; }
    const_iterator end()   const { return &elements[Capacity]; }

    bool operator==(const CArray& other) const
    {
      return std::equal(begin(), end(), other.begin());
    }

    T& operator[] (Player p) 
    {
#ifndef CONSERVATIVE_PLAYER_ACCESS
      // equivalent to operator[](playerToIndex(p))
      assert(1 < Capacity);
      return *((T*)((char *)&elements[0] + 
		    (p & ((char *)&elements[1]-(char *)&elements[0]))));
#else
      return operator[](playerToIndex(p));
#endif
    }
    const T& operator[] (Player p) const
    {
#ifndef CONSERVATIVE_PLAYER_ACCESS
      assert(1 < Capacity);
      return *((T*)((char *)&elements[0] + 
		    (p & ((char *)&elements[1]-(char *)&elements[0]))));
#else
      return operator[](playerToIndex(p));
#endif
    }
  };
  } // namespace misc
  using misc::CArray;
} // namespace osl


#endif /* _FIXED_CAPACITY_VECTOR_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
