/***************************************************************************
 *   Copyright (C) 2004-2008 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#include <stdlib.h>
#include <netinet/in.h>

#include <qstring.h>
#include <qstringlist.h>

#include <klocale.h>
#include <kdebug.h>

#include "portnumbernamelist.h"
#include "ip-protocol.h"
#include "icmp-protocol.h"
#include "tcp-protocol.h"
#include "udp-protocol.h"
#include "../packet.h"
#include "../packetstructures.h"

IpProtocol::IpProtocol(Packet *packet)
{
  init(packet);
}


IpProtocol::IpProtocol(Packet *packet, int length)
{
  init(packet, length);
}


IpProtocol::IpProtocol(ptrPacketType packet)
{
  init((Packet *)packet);
}


IpProtocol::IpProtocol(ptrPacketType packet, int length)
{
  init((Packet *)packet, length);
}


void IpProtocol::init(Packet *packet, int length)
{
  m_packet = packet;
  m_length = length != -1 ? length : packet->frameHeaderLength();
  m_ip = (struct IpHdr*) ( (ptrPacketType)m_packet + m_length );
  m_PortNumberNameList = NULL;
}


u_int8_t IpProtocol::version() const
{
  return ( (u_int8_t) m_ip->version );
}


int IpProtocol::headerLength() const
{
  return (m_ip->ihl << 2);
}


u_int16_t IpProtocol::totalLength() const
{
  return ntohs(m_ip->tot_len);
}


u_int16_t IpProtocol::id() const
{
  return ntohs(m_ip->id);
}


QString IpProtocol::flagR() const
{
  u_int16_t frag_off = ntohs(m_ip->frag_off);
  QString ret;

  if ( frag_off & (1 << 13) )
    ret = i18n("Set");
  else
    ret = i18n("Not set");

  return ret;
}


QString IpProtocol::flagDF() const
{
  u_int16_t frag_off = ntohs(m_ip->frag_off);
  QString ret;

  if ( frag_off & (1 << 14) )
    ret = i18n("Set");
  else
    ret = i18n("Not set");

  return ret;
}


QString IpProtocol::flagMF() const
{
  u_int16_t frag_off = ntohs(m_ip->frag_off);
  QString ret;

  if ( frag_off & (1 << 15) )
    ret = i18n("Set");
  else
    ret = i18n("Not set");

  return ret;
}


u_int16_t IpProtocol::fragmentOffset() const
{
  return ( ntohs(m_ip->frag_off) & 8191 );  // 8191 is 00011111 11111111 in binary
}


u_int8_t IpProtocol::ttl() const
{
  return ( (u_int8_t) m_ip->ttl );
}


u_int8_t IpProtocol::protocol() const
{
  return ( (u_int8_t) m_ip->protocol );
}


QString IpProtocol::protocolName() const
{
  QString proto;

  switch ( (u_int8_t) m_ip->protocol )
  {
    // this case has to be removed. We have to parse: /etc/protocols
    case 1:
      proto = "ICMP";
      break;
    case 2:
      proto = "IGMP";
      break;
    case 6:
      proto = "TCP";
      break;
    case 17:
      proto = "UDP";
      break;
    case 88:
      proto = "EIGRP";
      break;
  }

  return proto;
}


QString IpProtocol::applicationProtocolName() const
{
  QString result;
  TcpProtocol *tcp;
  UdpProtocol *udp;

  switch ( (u_int8_t) m_ip->protocol )
  {
    // this case has to be removed. We have to parse: /etc/protocols
    case 1:
      result = "ICMP";
      break;
    case 2:
      result = "IGMP";
      break;
    case 6:
      tcp = new TcpProtocol;
      tcp->setPacket((Packet*)((ptrPacketType)m_packet + m_length + headerLength()));
      if (m_PortNumberNameList == NULL)
        result = "TCP";
      else
      {
        // IANA: The Dynamic and/or Private Ports are those from 49152 through 65535
        if (tcp->sourcePort() < 30000)
          result = m_PortNumberNameList->detectPortNumberName( tcp->sourcePort() ).upper();
        else
          result = m_PortNumberNameList->detectPortNumberName( tcp->destinationPort() ).upper();
        if (result.isEmpty())
          result = "TCP";
      }
      break;
    case 17:
      udp = new UdpProtocol;
      udp->setPacket((Packet*)((ptrPacketType)m_packet + m_length + headerLength()));
      if (m_PortNumberNameList == NULL)
        result = "UDP";
      else
      {
        // IANA: The Dynamic and/or Private Ports are those from 49152 through 65535
        if (udp->sourcePort() < 30000)
          result = m_PortNumberNameList->detectPortNumberName( udp->sourcePort() ).upper();
        else
          result = m_PortNumberNameList->detectPortNumberName( udp->destinationPort() ).upper();
        if (result.isEmpty())
          result = "UDP";
      }
      break;
    case 88:
      result = "EIGRP";
      break;
    default:
      result = "-";
      break;
  }

  return result;
}


u_int16_t IpProtocol::checksum() const
{
  return ntohs(m_ip->check);
}


QString IpProtocol::sourceAddress() const
{
  unsigned char *x = (unsigned char *) (&m_ip->saddr);

  return QString("%1.%2.%3.%4").arg(x[0]).arg(x[1]).arg(x[2]).arg(x[3]);
}


QString IpProtocol::destinationAddress() const
{
  unsigned char *x = (unsigned char *) (&m_ip->daddr);

  return QString("%1.%2.%3.%4").arg(x[0]).arg(x[1]).arg(x[2]).arg(x[3]);
}


QString IpProtocol::strInfo() const
{
  QString result;
  IcmpProtocol icmp;
  TcpProtocol tcp;
  UdpProtocol udp;

  switch ( protocol() )
  {
    // this case has to be removed. We have to parse: /etc/protocols
    case 1:
      icmp.setPacket((Packet*)((ptrPacketType)m_packet + m_length + headerLength()));
      result = icmp.strInfo();
      break;
    case 2:
      result = "IGMP";
      break;
    case 6:
      tcp.setPacket((Packet*)((ptrPacketType)m_packet + m_length + headerLength()));
      result = tcp.strInfo();
      break;
    case 17:
      udp.setPacket((Packet*)((ptrPacketType)m_packet + m_length + headerLength()));
      result = udp.strInfo();
      break;
    case 88:
      result = "EIGRP";
      break;
    default:
      result = "-";
      break;
  }

  return result;
}


void IpProtocol::setPortNumberNameList( PortNumberNameList *pnnl )
{
  m_PortNumberNameList = pnnl;
}


QStringList IpProtocol::headerLines() const
{
  QStringList list;

  list << i18n("IP protocol field", "Version: %1").arg( version() );
  list << i18n("IP protocol field", "Header length: %1").arg( headerLength() );
  list << i18n("IP protocol field", "Total length: %1").arg( totalLength() );
  list << i18n("IP protocol field", "Identification: 0x%1 (%2)").arg( id(), 0, 16 ).arg( id() );
  list << "*open";
  list << i18n("IP protocol field: flags", "Flags:");
  list << i18n("IP protocol field: flags (R)", "Reserved bit: %1").arg( flagR() );
  list << i18n("IP protocol field: flags (DF)", "Don't fragment: %1").arg( flagDF() );
  list << i18n("IP protocol field: flags (MF)", "More fragments: %1").arg( flagMF() );
  list << "*close";
  list << i18n("IP protocol field", "Fragment offset: %1").arg( fragmentOffset() );
  list << i18n("IP protocol field", "Time to live: %1").arg( ttl() );
  list << i18n("IP protocol field", "Protocol: %1 (0x%2)").arg( protocolName() ).arg( protocol(), 0, 16 );
  list << i18n("IP protocol field", "Header checksum: 0x%1").arg( checksum(), 0, 16 );
  list << i18n("IP protocol field", "Source address: %1").arg( sourceAddress() );
  list << i18n("IP protocol field", "Destination address: %1").arg( destinationAddress() );

  return list;
}
