/***************************************************************************
 *   Copyright (C) 2004-2008 by Giovanni Venturi                           *
 *   giovanni@ksniffer.org                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02110-1301, USA.          *
 ***************************************************************************/

#ifndef KSNIFFERMAINWIDGET_H
#define KSNIFFERMAINWIDGET_H

#define WIDTH_COL_1 70
#define WIDTH_COL_2 100
#define WIDTH_COL_3 100
#define WIDTH_COL_4 80
#define WIDTH_COL_5 550

#define TIME_WAITING_UPDATE 1000

#include <qwidget.h>
#include <qmutex.h>

/**
 * This class contains the stuff to create and manage the QListView containing the
 * captured packet listed
 *
 * @author Giovanni Venturi
 */

class QListView;
class QListViewItem;
class QString;
class QVBoxLayout;
class QPoint;
class QTextEdit;
class QTimer;

class KListView;
class KURL;
class KListViewSearchLine;
class KSnifferViewItem;
class KInfoIP;

class PortNumberNameList;
class PacketManager;
class CaptureOptions;

class KSnifferMainWidget : public QWidget
{
  Q_OBJECT

public:
  KSnifferMainWidget( QWidget* parent = 0, const char* name = 0 );
  ~KSnifferMainWidget();

  /**
   * Random 'get' function
   */
  QString currentURL();

  /**
   * Random 'set' function accessed by DCOP
   */
  virtual void openURL(const QString& url);

  /**
   * Random 'set' function
   */
  virtual void openURL(const KURL& url);

  /**
   * Print this view to any medium -- paper or not
   */
  void print(QPainter *, int height, int width);

  /**
   * Delete all the element from the views
   */
  void clearView();

  /**
   * Let sort or not from extern class
   */
  void setSortList(bool value);

  void setOptions( CaptureOptions *options );

  void setPortNumberNameList( PortNumberNameList *pnnl );

  void loadFromFile( bool load = true );

  /**
   * Set the policy to view or not packets.
   */
  void displayPacket( bool toView );

  /**
   * @return true is the current policy is: display packets.
   */
  bool displayingPacket();

  /**
   * Update the filter list
   */
  void updateFilter();

  /**
   * Say yo the view the sniffing is starting
   */
  void startSniffing();

  /**
   * Say to the view the sniffing is stopping
   */
  void stopSniffing();

public slots:
  /**
   * Say to upgrade the view: you need it when you loaded file and you should apply a view filter
   */
  void updateView();

  /**
   * Say to change the raw form area where it's displayed the raw captured packet
   */
  void changeRawFormPosition( bool rightPosition );

signals:
  /**
   * Use this signal to change the content of the statusbar
   */
  void signalChangeStatusbar(const QString& text);

  /**
   * Use this signal to change the content of the caption
   */
  void signalChangeCaption(const QString& text);

private:
  KListView* m_list;
  QListView* m_detail;
  KListViewSearchLine *m_listSearch;
  QTextEdit *m_rawPacketArea;
  QTextEdit *m_rawPacketAreaRight;
  QTextEdit *m_rawPacketAreaUnder;
  QTimer *m_timer;
  QMutex m_mutex;

  bool m_needToUpdateSearch;     // true if there is text in the search line
  bool m_loadFromFile;     // true if you gets packets from file
  bool m_bExpanding0;      // true if the detail line 0 has to be expanded
  bool m_bExpanding1;      // true if the detail line 1 has to be expanded

  KInfoIP *m_info;
  KPopupMenu *m_popupMenu;
  bool m_firstInsert;              // true if we have to insert the first element into the list
  KSnifferViewItem *m_lastElem;    // point to the last element inserted into the list
  PacketManager *m_packetManager;  // needed to get and display all packet details
  CaptureOptions *m_options;
  PortNumberNameList *m_portNumberNameList;

  bool m_displayingPacket;  // true = captured packet has to be displayed into the view
                            // false = captured packet has NOT to be displayed into the view

protected:
  QVBoxLayout* m_mainViewLayout;

private slots:
  void slotOnURL(const QString& url);
  void slotSetTitle(const QString& title);

  /**
   * Use this slot to set to expande state the the m_detail list
   */
  void setExpand( QListViewItem* i );

  /**
   * Use this slot to set to collapsed state the the m_detail list
   */
  void setCollapse( QListViewItem* i );

  void applyNewSearchFilter( const QString & text );

protected slots:
  /**
   * Use this slot to view captured packets
   */
  void displayPacket(long ord, PacketManager *packetManager);

  /**
   * Use this slot to show details about a captured packet
   */
  void showDetails( QListViewItem * item );

  /**
   * Use this slot to get right mouse clicking event on the list items (the m_list widget)
   */
  void clickedMouseButton(QListViewItem* item, const QPoint& pos, int column);

private:
  void viewSubtree(QStringList::Iterator &it, QListViewItem **itemFrame, QListViewItem **subitemFrame);
  void showRawPacketDetails();

};

#endif // KSNIFFERMAINWIDGET_H
