/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "kz-root-bookmark.h"

#include <glib/gi18n.h>
#include "kz-bookmark-file.h"
#include "kz-session.h"

static void     dispose     (GObject *object);

G_DEFINE_TYPE(KzRootBookmark, kz_root_bookmark, KZ_TYPE_BOOKMARK_FOLDER)

static void
kz_root_bookmark_class_init (KzRootBookmarkClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS(klass);

	object_class->dispose     = dispose;
}


static void
kz_root_bookmark_init (KzRootBookmark *root)
{
	root->bookmark_bars   = kz_bookmark_folder_new("Bookmark Bars");
	root->tab_groups      = kz_bookmark_folder_new("Tab Groups");
	root->smarts          = NULL;
	root->current_session = NULL;
}


static void
dispose (GObject *object)
{
	KzRootBookmark *root;

	root = KZ_ROOT_BOOKMARK(object);

	if (root->menu)
	{
		g_object_unref(root->menu);
		root->menu = NULL;
	}
	if (root->clip)
	{
		g_object_unref(root->clip);
		root->clip = NULL;
	}
	if (root->bookmark_bars)
	{
		g_object_unref(root->bookmark_bars);
		root->bookmark_bars = NULL;
	}
	if (root->history)
	{
		g_object_unref(root->history);
		root->history = NULL;
	}
	if (root->current_session)
	{
		g_object_unref(root->current_session);
		root->current_session = NULL;
	}
	if (root->tab_groups)
	{
		g_object_unref(root->tab_groups);
		root->tab_groups = NULL;
	}
	if (root->smarts)
	{
		g_object_unref(root->smarts);
		root->smarts = NULL;
	}

	if (G_OBJECT_CLASS (kz_root_bookmark_parent_class)->dispose)
		G_OBJECT_CLASS (kz_root_bookmark_parent_class)->dispose(object);
}


KzRootBookmark *
kz_root_bookmark_new (const gchar *menu_file,  const gchar *sys_menu_file,
		      const gchar *clip_file, const gchar *sys_clip_file)
{
	KzRootBookmark *root;

	root = g_object_new(KZ_TYPE_ROOT_BOOKMARK,
			    "title", _("Root"),
			    NULL);

	if (root->menu)
		kz_bookmark_folder_append(KZ_BOOKMARK_FOLDER(root), root->menu);
	if (root->clip)
		kz_bookmark_folder_append(KZ_BOOKMARK_FOLDER(root), root->clip);
	if (root->bookmark_bars)
		kz_bookmark_folder_append(KZ_BOOKMARK_FOLDER(root), root->bookmark_bars);

	kz_root_bookmark_set_menu_file(root, menu_file, sys_menu_file);
	kz_root_bookmark_set_clip_file(root, clip_file, sys_clip_file);

	return root;
}


static KzBookmark *
ensure_bookmark_file (const gchar *file, const gchar *subfile,
		      const gchar *title, const gchar *type)
{
	KzBookmarkFile *bookmark;

	g_return_val_if_fail(file && *file, NULL);
	g_return_val_if_fail(type && *type, NULL);

	if (g_file_test(file, G_FILE_TEST_EXISTS))
	{
		bookmark = kz_bookmark_file_new(file, title, type);
		kz_bookmark_file_load_start(bookmark);
	}
	else if (subfile && *subfile &&
		 g_file_test(subfile, G_FILE_TEST_EXISTS))
	{
		bookmark = kz_bookmark_file_new(subfile, title, type);
		kz_bookmark_file_load_start(bookmark);
	}
	else
	{
		bookmark = kz_bookmark_file_create_new(file, title, type);
	}

	kz_bookmark_file_set_location(bookmark, file);

	return KZ_BOOKMARK(bookmark);
}


void
kz_root_bookmark_set_menu_file (KzRootBookmark *root,
				const gchar *file, const gchar *subfile)
{
	KzBookmark *menu;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */

	menu = ensure_bookmark_file(file, subfile, _("Bookmarks"), "XBEL");

	if (root->menu)
	{
		kz_bookmark_folder_remove(KZ_BOOKMARK_FOLDER(root), root->menu);
		g_object_unref(root->menu);
		root->menu = NULL;
	}

	if (menu)
	{
		root->menu = menu;
		kz_bookmark_folder_insert_before(KZ_BOOKMARK_FOLDER(root),
					  menu, root->clip);
	}
}


void
kz_root_bookmark_set_clip_file (KzRootBookmark *root,
				 const gchar *file, const gchar *subfile)
{
	KzBookmark *clip;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */
	clip = ensure_bookmark_file(file, subfile, _("Clip"), "XBEL");

	if (root->clip)
	{
		kz_bookmark_folder_remove(KZ_BOOKMARK_FOLDER(root), root->clip);
		g_object_unref(root->clip);
		root->clip = NULL;
	}

	if (clip)
	{
		root->clip = clip;
		kz_bookmark_folder_insert_before(KZ_BOOKMARK_FOLDER(root),
					         clip, root->smarts);
	}
}


void
kz_root_bookmark_add_smartbookmark_file (KzRootBookmark *root,
			 	 	 const gchar *file,
				 	 const gchar *subfile)
{
	KzBookmark *smart;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	/* FIXME! Check permission */

	smart = ensure_bookmark_file(file, subfile, _("SmartBookmarks"), "XBEL");

	if (root->smarts)
	{
		kz_bookmark_folder_remove(KZ_BOOKMARK_FOLDER(root), root->smarts);
		g_object_unref(root->smarts);
		root->smarts = NULL;
	}

	if (smart)
	{
		root->smarts = smart;
		kz_bookmark_folder_insert_before(KZ_BOOKMARK_FOLDER(root),
					         smart, root->bookmark_bars);
	}
}


void
kz_root_bookmark_add_bookmark_bar_file (KzRootBookmark *root,
					const gchar *file, const gchar *subfile)
{
	KzBookmarkFile *bar;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);
	g_return_if_fail(root->bookmark_bars);

	/* FIXME! Check permission */

	if (g_file_test(file, G_FILE_TEST_EXISTS))
	{
		bar = kz_bookmark_file_new(file,
					   _("Bookmark Bar"),
					   "XBEL");
	}
	else if (subfile && *subfile &&
		 g_file_test(subfile, G_FILE_TEST_EXISTS))
	{
		bar = kz_bookmark_file_new(subfile,
					   _("Bookmark Bar"),
					   "XBEL");
	}
	else
	{
		bar = kz_bookmark_file_create_new(file,
						  _("Bookmark Bar"),
						  "XBEL");
	}

	kz_bookmark_file_load_start(bar);
	kz_bookmark_file_set_location(bar, file);

	kz_bookmark_folder_append(KZ_BOOKMARK_FOLDER(root->bookmark_bars), KZ_BOOKMARK(bar));
	g_object_unref(bar);
}


void
kz_root_bookmark_add_current_session_file (KzRootBookmark *root,
					   const gchar *file)
{
	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));
	g_return_if_fail(file && *file);

	if (root->current_session)
	{
		g_object_unref(root->current_session);
		root->current_session = NULL;
	}

	root->current_session = kz_session_new(file, _("Current Session"));
}


void
kz_root_bookmark_save_all (KzRootBookmark *root)
{
	GList *node;

	g_return_if_fail(KZ_IS_ROOT_BOOKMARK(root));

	if (root->menu)
		kz_bookmark_file_save(KZ_BOOKMARK_FILE(root->menu));
	if (root->clip)
		kz_bookmark_file_save(KZ_BOOKMARK_FILE(root->clip));

	if (root->bookmark_bars)
	{
		GList *children;

		children = kz_bookmark_folder_get_children(KZ_BOOKMARK_FOLDER(root->bookmark_bars));
		for (node = children; node; node = g_list_next(node))
		{
			KzBookmark *bar = node->data;

			if (!KZ_IS_BOOKMARK(bar) || !KZ_IS_BOOKMARK_FILE(bar))
				g_warning("Invalid bookmark bar file!");
			kz_bookmark_file_save(KZ_BOOKMARK_FILE(bar));
		}
		g_list_free(children);
	}
	if (root->smarts)
		kz_bookmark_file_save(KZ_BOOKMARK_FILE(root->smarts));
}
