/*
    Copyright (C) 2008 Fons Adriaensen <fons@kokkinizita.net>
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/


#include <stdlib.h>
#include <stdio.h>
#include <clthreads.h>
#include <sys/mman.h>
#include <signal.h>
#include "styles.h"
#include "mainwin.h"


#define NOPTS 6
#define CP (char *)


XrmOptionDescRec options [NOPTS] =
{
    {CP"-help",      CP".help",     XrmoptionNoArg,   CP"true"    },
    {CP"-type",      CP".type",     XrmoptionSepArg,  0           },
    {CP"-gain",      CP".gain",     XrmoptionSepArg,  0           },
    {CP"-name",      CP".name",     XrmoptionSepArg,  CP"jmeters" },
    {CP"-rate",      CP".rate",     XrmoptionSepArg,  CP"20"      },
    {CP"-columns",   CP".columns",  XrmoptionSepArg,  0           }
};


static Jclient  *jclient = 0;
static Mainwin  *mainwin = 0;
static JMconf    config;


static void help (void)
{
    fprintf (stderr, "\nJmeters-%s\n", VERSION);
    fprintf (stderr, "(C) 2008 Fons Adriaensen  <fons@kokkinizita.net>\n");
    fprintf (stderr, "Syntax: jmeters <options> <jack port>+\n");
    fprintf (stderr, "Options:\n");
    fprintf (stderr, "  -h         Display this text\n");
    fprintf (stderr, "  -t <type>  Meter type (vu, svu, ppm, sppm, ppm2, sppm2)\n");
    fprintf (stderr, "  -g <v>     Gain (0 dB)\n");
    fprintf (stderr, "  -n <name>  Name to use as Jack client (jmeters)\n");
    fprintf (stderr, "  -r <n>     Update rate (20)\n");
    fprintf (stderr, "  -c <n>     Columns (4)\n");
    exit (1);
}


static void sigint_handler (int)
{
    signal (SIGINT, SIG_IGN);
    mainwin->stop ();
}


int main (int ac, char *av [])
{
    X_resman      xresman;
    X_display     *display;
    X_handler     *handler;
    X_rootwin     *rootwin;
    const char    *p;
    int           i, e;
    

    xresman.init (&ac, av, CP"jmeters", options, NOPTS);
    if (xresman.getb (".help", 0)) help ();
            
    display = new X_display (xresman.get (".display", 0));
    if (display->dpy () == 0)
    {
	fprintf (stderr, "Can't open display.\n");
        delete display;
	exit (1);
    }

    p = xresman.get (".type", "vu");
    if      (! strcmp (p, "vu"))    config._type = Meterwin::MVUM;
    else if (! strcmp (p, "svu"))   config._type = Meterwin::SVUM;
    else if (! strcmp (p, "ppm"))   config._type = Meterwin::MPPM;
    else if (! strcmp (p, "sppm"))  config._type = Meterwin::SPPM;
    else if (! strcmp (p, "ppm2"))  config._type = Meterwin::MPPM2;
    else if (! strcmp (p, "sppm2")) config._type = Meterwin::SPPM2;
    else help ();
    if (--ac < 1) help ();
    config._numb = ac;
    config._ncol = atoi (xresman.get (".columns", "4"));
    config._gain = atof (xresman.get (".gain", "0"));
    for (i = 0; i < ac; i++)
    {
	p = *++av;
	if (*p == '-') help ();
        config._mdata [i]._port = p;			 
    }

    styles_init (display, &xresman);
    rootwin = new X_rootwin (display);
    jclient = new Jclient (xresman.rname (), &config);
    mainwin = new Mainwin (rootwin, &xresman, jclient, &config);
    handler = new X_handler (display, mainwin, EV_X11);
    handler->next_event ();
    XFlush (display->dpy ());

    if (mlockall (MCL_CURRENT | MCL_FUTURE)) fprintf (stderr, "Warning: memory lock failed.\n");
    signal (SIGINT, sigint_handler); 

    do
    {
	e = mainwin->process ();
	if (e == EV_X11)
	{
	    rootwin->handle_event ();
	    handler->next_event ();
	}
    }
    while (e != EV_EXIT);	

    delete jclient;
    delete handler;
    delete rootwin;
    delete display;
   
    return 0;
}



