/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2011 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "isurfacepipeline.h"


#include "iboundedplanesource.h"
#include "iboundedspheresource.h"
#include "icontourfilter.h"
#include "ierror.h"
#include "iflipnormalsfilter.h"
#include "iidentityfilter.h"
#include "ioptimizepolydatafilter.h"
#include "iprobefilter.h"
#include "ireducepolydatafilter.h"
#include "ireducepolydatafilter2.h"
#include "ireplicatedpolydata.h"
#include "ishiftpolydatafilter.h"
#include "ismoothpolydatafilter.h"
#include "isurfaceviewsubject.h"

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"
#include "iviewsubjectpipelinetemplate.h"


using namespace iParameter;


//
// iSurfacePipeline class
//
iSurfacePipeline::iSurfacePipeline(iSurfaceViewSubject *s) : iViewSubjectPipeline(s,1)
{
	int i;

	mSpecialVar = 0;
	mParent = s;

	vtkPolyData *tmp = vtkPolyData::New(); IERROR_ASSERT(tmp);
	this->AddOutput(tmp);
	tmp->Delete();

	//
	//  Do VTK stuff
	//	
	mGateway = this->CreateFilter<iIdentityFilter>();
	mIsoSurfaceExitGateway = this->CreateFilter<iIdentityFilter>();
	mIsoSurfaceEntryGateway = this->CreateFilter<iIdentityFilter>();
	mIsoSurfaceSource = this->CreateFilter<iContourFilter>();
	mShiftFilter = this->CreateFilter<iShiftPolyDataFilter>();
	mSmoothFilter = this->CreateFilter<iSmoothPolyDataFilter>();
	mReduceFilter = this->CreateFilter<iReducePolyDataFilter>();
	mReduce2Filter = this->CreateFilter<iReducePolyDataFilter2>();
	mOptimizeFilter = this->CreateFilter<iOptimizePolyDataFilter>();

	mSphereSource = this->CreateFilter<iBoundedSphereSource>();
	mPlaneSource = this->CreateFilter<iBoundedPlaneSource>();

	for(i=0; i<2; i++)
	{
		mProbeFilter[i] = this->CreateFilter<iProbeFilter>();
		mDataReplicated[i] = this->CreateFilter<iReplicatedPolyData>();
		mFlipNormalsFilter[i] = this->CreateFilter<iFlipNormalsFilter>();
	}

	//
	//  A dummy gateway filter needs to have its input specified first, so do not request its output for now
	//
	for(i=0; i<2; i++)
	{
		mProbeFilter[i]->SetInput(mSphereSource->GetOutput());
		mFlipNormalsFilter[i]->SetInput(mProbeFilter[i]->GetPolyDataOutput());
		mFlipNormalsFilter[i]->ReverseCellsOff();
	}

	mSmoothFilter->BoundarySmoothingOff();
	mSmoothFilter->NonManifoldSmoothingOn();
	mSmoothFilter->FeatureEdgeSmoothingOn();

	mReduceFilter->PreserveTopologyOn();
	mReduceFilter->BoundaryVertexDeletionOff(); 
	
	mIsoSurfaceEntryGateway->SetInput(mIsoSurfaceSource->GetOutput());

	mReduceFilter->SetInput(mIsoSurfaceEntryGateway->GetPolyDataOutput());
	mReduce2Filter->SetInput(mIsoSurfaceEntryGateway->GetPolyDataOutput());
	mShiftFilter->SetInput(mFlipNormalsFilter[1]->GetOutput());

	mDataReplicated[0]->SetInput(mFlipNormalsFilter[0]->GetOutput());
	mDataReplicated[1]->SetInput(mShiftFilter->GetOutput());

	this->UpdateContents(_All);
}


iSurfacePipeline::~iSurfacePipeline()
{
}


void iSurfacePipeline::ProduceOutput()
{
	int i;

	vtkDataSet *input = this->GetInput();

	if(mGateway->GetInput() != input)
	{
		mGateway->SetInput(input);
		mIsoSurfaceSource->SetInput(mGateway->GetOutput());
		for(i=0; i<2; i++)
		{
			mProbeFilter[i]->SetSource(mGateway->GetOutput());
		}
	}
	
	//
	//  Check whether fixed probe surfaces need to be adjusted to correspond to the input resolution
	//
	bool redo = false;
	if(mSphereSource->GetMTime() < input->GetMTime())
	{
		this->UpdateSphereResolution();
		redo = true;
	}
	if(mPlaneSource->GetMTime() < input->GetMTime())
	{
		this->UpdatePlaneResolution();
		redo = true;
	}

	if(redo)
	{
		double b[6];
		input->GetBounds(b);
		mPlaneSource->SetBounds(b[0],b[1],b[2],b[3],b[4],b[5]);
		mSphereSource->SetBounds(b[0],b[1],b[2],b[3],b[4],b[5]);
	}

	//
	//  Update the pipeline
	//
	mDataReplicated[0]->Update();
	this->GetOutput(0)->ShallowCopy(mDataReplicated[0]->GetOutput());

	if(mParent->GetPaintVar(1) > 0)
	{
		mDataReplicated[1]->Update();
		this->GetOutput(1)->ShallowCopy(mDataReplicated[1]->GetOutput());
	}
	else this->GetOutput(1)->Initialize();
}


float iSurfacePipeline::GetContentsMemorySize() const
{
	int i;
	float s;

	s = mIsoSurfaceSource->GetMemorySize();

	for(i=0; i<2; i++)
	{
		s += mProbeFilter[i]->GetMemorySize();
		s += mDataReplicated[i]->GetMemorySize();
		s += mFlipNormalsFilter[i]->GetMemorySize();
	}

	s += mSmoothFilter->GetMemorySize();
	s += mReduceFilter->GetMemorySize();
	s += mReduce2Filter->GetMemorySize();
	s += mOptimizeFilter->GetMemorySize();

	s += mSphereSource->GetMemorySize();
	s += mPlaneSource->GetMemorySize();

	return s;
}


void iSurfacePipeline::UpdateContents(int n, int info)
{
	//
	//  Update the pipeline
	//
	if(n==_All || n==_FixedSurfacePosition) this->UpdateFixedSurfacePosition();
	if(n==_All || n==_FixedSurfaceSize) this->UpdateFixedSurfaceSize();
	if(n==_All || n==_IsoSurfaceLevel) this->UpdateIsoSurfaceLevel();
	if(n==_All || n==_IsoSurfaceMethod) this->UpdateIsoSurfaceMethod();
	if(n==_All || n==_IsoSurfacePipeline) this->UpdateIsoSurfacePipeline(0);
	if(n==_All || n==_IsoSurfaceVar) this->UpdateIsoSurfaceVar();
	if(n==_All || n==_Method) this->UpdateMethod();
	if(n==_All || n==_NormalsFlipped) this->UpdateNormalsFlipped();
	if(n==_All || n==_PlaneDirection) this->UpdatePlaneDirection();
	if(n==_All || n==_PlaneResolution) this->UpdatePlaneResolution();
	if(n==_All || n==_Replicas) this->UpdateReplicas();
	if(n==_All || n==_SphereResolution) this->UpdateSphereResolution();
	if(n == _All)
	{
		this->UpdateVarPaint(0);
		this->UpdateVarPaint(1);
	}
	if(n == _VarPaint) this->UpdateVarPaint(info);
}


void iSurfacePipeline::UpdateIsoSurfacePipeline(int force)
{ 
	if(force==0 && mParent->GetMethod()!=SurfaceMethod::IsoSurface) return;

	switch(mParent->GetIsoSurfaceReduction())
	{
	case 1:
		{
			mReduceFilter->SetTargetReduction(0.75);
			mReduce2Filter->SetTargetReduction(0.75);
			break;
		}
	case 2:
		{
			mReduceFilter->SetTargetReduction(0.90);
			mReduce2Filter->SetTargetReduction(0.90);
			break;
		}
	case 3:
		{
			mReduceFilter->SetTargetReduction(0.99);
			mReduce2Filter->SetTargetReduction(0.99);
			break;
		}
	}

	mSmoothFilter->SetNumberOfIterations(2<<(2*mParent->GetIsoSurfaceSmoothing()));

	//
	//  Create the pipeline
	//
	vtkPolyData *step1, *step2, *step3;

	if(mParent->GetIsoSurfaceReduction() > 0) 
	{
		step1 = mParent->GetAlternativeIsoSurfaceReductionMethod() ? mReduce2Filter->GetOutput() : mReduceFilter->GetOutput();
	}
	else
	{
		step1 = mIsoSurfaceEntryGateway->GetPolyDataOutput();
	}

	mSmoothFilter->SetInput(step1);

	if(mParent->GetIsoSurfaceSmoothing() > 0)
	{
		step2 = mSmoothFilter->GetOutput();
	}
	else
	{
		step2 = step1;
	}

	mOptimizeFilter->SetInput(step2);

	if(mParent->GetIsoSurfaceOptimization())
	{
		step3 = mOptimizeFilter->GetOutput();
	}
	else
	{
		step3 = step2;
	}

	mIsoSurfaceExitGateway->SetInput(step3);

	this->Modified();
}


void iSurfacePipeline::UpdateIsoSurfaceVar()
{
	mIsoSurfaceSource->SetCurrentVar(mParent->GetIsoSurfaceVar());
	this->Modified();
}


void iSurfacePipeline::UpdateIsoSurfaceLevel()
{ 
	mIsoSurfaceSource->SetLevel(mParent->GetIsoSurfaceLevel());
	this->Modified();
}


void iSurfacePipeline::UpdateIsoSurfaceMethod()
{ 
	mIsoSurfaceSource->SetMethod(mParent->GetIsoSurfaceMethod());
	this->Modified();
}


void iSurfacePipeline::UpdateVarPaint(int)
{
	//
	//  We carry all scalars, do nothing here
	//
}


void iSurfacePipeline::UpdateNormalsFlipped()
{ 
	int var = mParent->GetIsoSurfaceVar();
	bool s = mParent->GetNormalsFlipped();

	if(mParent->GetMethod() != SurfaceMethod::IsoSurface) var = -1;

	if((s && var!=mSpecialVar) || (!s && var==mSpecialVar)) 
	{
		mFlipNormalsFilter[0]->ReverseNormalsOn(); 
		mFlipNormalsFilter[1]->ReverseNormalsOff();
		mShiftFilter->SetShift(fabs(mShiftFilter->GetShift()));
	} 
	else 
	{
		mFlipNormalsFilter[0]->ReverseNormalsOff();
		mFlipNormalsFilter[1]->ReverseNormalsOn();
		mShiftFilter->SetShift(-fabs(mShiftFilter->GetShift()));
	}
	this->Modified();
}


void iSurfacePipeline::UpdateReplicas()
{
	mDataReplicated[0]->Modified();
	mDataReplicated[1]->Modified();
	this->Modified();
}


void iSurfacePipeline::UpdateMethod()
{
	switch(mParent->GetMethod())
	{
	case SurfaceMethod::IsoSurface:
		{
			this->UpdateIsoSurfacePipeline(0);
			mProbeFilter[0]->SetInput(mIsoSurfaceExitGateway->GetOutput());
			mProbeFilter[1]->SetInput(mIsoSurfaceExitGateway->GetOutput());
			this->Modified();
			break;
		}
	case SurfaceMethod::Sphere:
		{
			mProbeFilter[0]->SetInput(mSphereSource->GetOutput());
			mProbeFilter[1]->SetInput(mSphereSource->GetOutput());
			this->Modified();
			break;
		}
	case SurfaceMethod::Plane:
		{
			mProbeFilter[0]->SetInput(mPlaneSource->GetOutput());
			mProbeFilter[1]->SetInput(mPlaneSource->GetOutput());
			this->Modified();
			break;
		}
	}
	this->UpdateNormalsFlipped();
}


void iSurfacePipeline::UpdateFixedSurfaceSize()
{
	mSphereSource->SetRadius(mParent->GetSize());
	this->UpdateSphereResolution();
	this->Modified();
}


void iSurfacePipeline::UpdateFixedSurfacePosition()
{
	mSphereSource->SetCenter(mParent->GetPosition());
	mPlaneSource->SetCenter(mParent->GetPosition());
	this->Modified();
}


void iSurfacePipeline::UpdatePlaneDirection()
{
	const float *n = mParent->GetPlaneDirection();
	mPlaneSource->SetNormal(n);
	this->Modified();
}


void iSurfacePipeline::UpdateSphereResolution()
{
	double dx;

	vtkDataSet *input = this->GetInput();
	if(input == 0) return;

	if(input->IsA("vtkImageData"))
	{
		double spa[3];
		iRequiredCast<vtkImageData>(INFO,input)->GetSpacing(spa);
		dx = spa[0];
		if(dx < spa[1]) dx = spa[1];
		if(dx < spa[2]) dx = spa[2];
		if(dx < 1.0e-3) dx = 1.0e-3;
	}
	else dx = 0.01;

	float s = mSphereSource->GetRadius();
	int n = round(3.14*s/dx);
	if(n < 6) n = 6;
	mSphereSource->SetResolution(n);
	this->Modified();
}


void iSurfacePipeline::UpdatePlaneResolution()
{
	double dx;

	vtkDataSet *input = this->GetInput();
	if(input == 0) return;

	if(input->IsA("vtkImageData"))
	{
		double spa[3];
		iRequiredCast<vtkImageData>(INFO,input)->GetSpacing(spa);
		dx = spa[0];
		if(dx < spa[1]) dx = spa[1];
		if(dx < spa[2]) dx = spa[2];
		if(dx < 1.0e-3) dx = 1.0e-3;
	}
	else dx = 0.01;

	float s = 7.0;
	int n = round(s/dx);
	mPlaneSource->SetResolution(n);
	this->Modified();
}
