#include <glib.h>
#include <glib/gi18n.h>

#include "hrn-marshal.h"
#include "hrn-sidebar-item.h"
#include "hrn-sidebar-subitem.h"
#include "hrn-spinner.h"

enum {
    PROP_0,
    PROP_PIN_MANAGER,
    PROP_SOURCE,
    PROP_NAME,
};

enum {
    SOURCE_CHANGED,
    LAST_SIGNAL
};

struct _HrnSidebarItemPrivate {
    ClutterActor *group;
    NbtkWidget *title;
    ClutterActor *spinner;
    NbtkWidget *child_grid;

    HrnSource *source;
    HrnPinManager *pin_manager;
    gboolean initialised; /* NbtkListView seems to like setting properties
                             on items frequently, so only do stuff if we've
                             been not already been set up */
    GList *items_pinned;
};

#define SIDEBAR_WIDTH 170
#define GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), HRN_TYPE_SIDEBAR_ITEM, HrnSidebarItemPrivate))
G_DEFINE_TYPE (HrnSidebarItem, hrn_sidebar_item, NBTK_TYPE_TABLE);
static guint32 signals[LAST_SIGNAL] = {0, };

static void
hrn_sidebar_item_finalize (GObject *object)
{
    G_OBJECT_CLASS (hrn_sidebar_item_parent_class)->finalize (object);
}

static void
hrn_sidebar_item_dispose (GObject *object)
{
    HrnSidebarItem *self = (HrnSidebarItem *) object;
    HrnSidebarItemPrivate *priv = self->priv;

    if (priv->source) {
        g_object_unref (priv->source);
        priv->source = NULL;
    }

    if (priv->pin_manager) {
        g_object_unref (priv->pin_manager);
        priv->pin_manager = NULL;
    }

    G_OBJECT_CLASS (hrn_sidebar_item_parent_class)->dispose (object);
}

static void
emit_source_changed (HrnSidebarItem *item,
                     int             filter,
                     const char     *query_name)
{
    HrnSidebarItemPrivate *priv = item->priv;

    g_signal_emit (item, signals[SOURCE_CHANGED], 0,
                   priv->source, filter, query_name);
}

static void
select_subitem_cb (ClutterActor *actor,
                   gpointer      data)
{
    nbtk_button_set_checked ((NbtkButton *) actor,
                             actor == (ClutterActor *) data);
}

static void
select_subitem (HrnSidebarItem    *item,
                HrnSidebarSubitem *subitem)
{
    HrnSidebarItemPrivate *priv = item->priv;

    clutter_container_foreach (CLUTTER_CONTAINER (priv->child_grid),
                               select_subitem_cb, subitem);
}

static void
change_source_query (HrnSidebarSubitem *subitem,
                     HrnSidebarItem    *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    const char *group;
    HrnPin *pin;

    select_subitem (item, subitem);

    group = hrn_sidebar_subitem_get_qname (subitem);
    pin = hrn_pin_manager_get_pin (priv->pin_manager, group);
    if (pin) {
        emit_source_changed (item, pin->filter, group);
    }
}

static void
item_pinned (HrnSidebarSubitem *subitem,
             gboolean           pinned,
             HrnSidebarItem    *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    const char *group;
    HrnPin *pin;
    GList *p;

    /* This shouldn't happen */
    if (G_UNLIKELY (pinned)) {
        return;
    }

    group = hrn_sidebar_subitem_get_qname (subitem);
    pin = hrn_pin_manager_get_pin (priv->pin_manager, group);

    if (G_UNLIKELY (pin == NULL)) {
        return;
    }

    for (p = priv->items_pinned; p; p = p->next) {
        if (p->data == subitem) {
            priv->items_pinned = g_list_remove_link (priv->items_pinned, p);
            g_list_free (p);
            break;
        }
    }

    hrn_pin_manager_remove (priv->pin_manager, group);
    clutter_actor_destroy ((ClutterActor *) subitem);
}

static void
item_renamed (HrnSidebarSubitem *subitem,
              const char        *old_name,
              const char        *new_name,
              HrnSidebarItem    *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    HrnPin *pin;
    const gchar *group;

    group = hrn_sidebar_subitem_get_qname (subitem);

    pin = hrn_pin_manager_get_pin (priv->pin_manager, group);
    if (pin == NULL) {
        g_warning ("No pin for %s", group);
        return;
    }

    hrn_pin_manager_rename_pin (priv->pin_manager, pin, new_name);
}

static HrnSidebarSubitem *
create_subitem (HrnSidebarItem *item,
                HrnPin         *pin)
{
    HrnSidebarSubitem *subitem;
    int filter = pin->filter;
    const char *iname;

    switch (filter) {
    case BKL_ITEM_TYPE_AUDIO:
        iname = "icon-audio";
        break;

    case BKL_ITEM_TYPE_IMAGE:
        iname = "icon-images";
        break;

    case BKL_ITEM_TYPE_VIDEO:
        iname = "icon-video";
        break;

    default:
        iname = "icon-all";
    }

    subitem = hrn_sidebar_subitem_new (pin->group,
                                       pin->name ? pin->name : pin->group,
                                       iname, TRUE, TRUE);
    return subitem;
}

static void
setup_source (HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    const char *source_path;
    GList *pins, *p;

    if (priv->initialised == TRUE) {
        return;
    }

    priv->initialised = TRUE;

    source_path = hrn_source_get_object_path (priv->source);
    pins = hrn_pin_manager_get_pins (priv->pin_manager);

    for (p = pins; p; p = p->next) {
        HrnPin *pin = p->data;
        HrnSidebarSubitem *subitem;

        if (g_str_equal (pin->source, source_path) == FALSE) {
            continue;
        }

        subitem = create_subitem (item, pin);
        g_signal_connect (subitem, "clicked",
                          G_CALLBACK (change_source_query), item);

        g_signal_connect (subitem, "pinned",
                          G_CALLBACK (item_pinned), item);
        g_signal_connect (subitem, "renamed",
                          G_CALLBACK (item_renamed), item);

        priv->items_pinned = g_list_prepend (priv->items_pinned, subitem);
        clutter_container_add_actor (CLUTTER_CONTAINER (priv->child_grid),
                                     CLUTTER_ACTOR (subitem));
        nbtk_bin_set_alignment (NBTK_BIN (subitem),
                                NBTK_ALIGN_START,
                                NBTK_ALIGN_START);
    }
    g_list_free (pins);
}

static void
pin_added_cb (HrnPinManager  *pin_manager,
              HrnPin         *pin,
              HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    HrnSidebarSubitem *subitem;
    const char *source_path;

    source_path = hrn_source_get_object_path (priv->source);

    if (g_str_equal (pin->source, source_path) == FALSE) {
        return;
    }

    subitem = create_subitem (item, pin);
    g_signal_connect (subitem, "clicked",
                      G_CALLBACK (change_source_query), item);

    g_signal_connect (subitem, "pinned",
                      G_CALLBACK (item_pinned), item);
    g_signal_connect (subitem, "renamed",
                      G_CALLBACK (item_renamed), item);

    priv->items_pinned = g_list_prepend (priv->items_pinned, subitem);
    clutter_container_add_actor (CLUTTER_CONTAINER (priv->child_grid),
                                 CLUTTER_ACTOR (subitem));
    nbtk_bin_set_alignment (NBTK_BIN (subitem),
                            NBTK_ALIGN_START,
                            NBTK_ALIGN_START);
}

static void
pin_removed_cb (HrnPinManager  *pin_manager,
                HrnPin         *pin,
                HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    GList *p;

    for (p = priv->items_pinned; p; p = p->next) {
        HrnSidebarSubitem *subitem = (HrnSidebarSubitem *) p->data;
        const char *group_name;

        group_name = hrn_sidebar_subitem_get_qname (subitem);
        if (g_str_equal (group_name, pin->group)) {
            priv->items_pinned = g_list_remove_link (priv->items_pinned, p);
            clutter_actor_destroy ((ClutterActor *) subitem);
            g_list_free (p);
            break;
        }
    }
}

static void
index_started_cb (HrnSource      *source,
                  HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;

    clutter_actor_show (priv->spinner);
    hrn_spinner_start ((HrnSpinner *) priv->spinner);
}

static void
index_finished_cb (HrnSource      *source,
                   HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;

    hrn_spinner_stop ((HrnSpinner *) priv->spinner);
    clutter_actor_hide (priv->spinner);
}

static void
hrn_sidebar_item_set_property (GObject      *object,
                               guint         prop_id,
                               const GValue *value,
                               GParamSpec   *pspec)
{
    HrnSidebarItem *item = (HrnSidebarItem *) object;
    HrnSidebarItemPrivate *priv = item->priv;
    HrnSource *source;

    switch (prop_id) {
    case PROP_PIN_MANAGER:
        if (priv->pin_manager) {
            g_object_unref (priv->pin_manager);
            priv->pin_manager= NULL;
        }
        priv->pin_manager = g_value_dup_object (value);
        g_signal_connect (priv->pin_manager, "pin-added",
                          G_CALLBACK (pin_added_cb), item);
        g_signal_connect (priv->pin_manager, "pin-removed",
                          G_CALLBACK (pin_removed_cb), item);

        if (priv->pin_manager && priv->source) {
            setup_source (item);
        }
        break;

    case PROP_SOURCE:
        source = g_value_get_object (value);
        if (priv->source && priv->source != source) {
            g_object_unref (priv->source);
            priv->source = NULL;
        }
        priv->source = g_object_ref (source);
        g_signal_connect (priv->source, "index-started",
                          G_CALLBACK (index_started_cb), item);
        g_signal_connect (priv->source, "index-finished",
                          G_CALLBACK (index_finished_cb), item);

        if (priv->source && priv->pin_manager) {
            setup_source (item);
        }
        break;

    case PROP_NAME:
        nbtk_label_set_text ((NbtkLabel *) priv->title,
                             g_value_get_string (value));
        break;

    default:
        break;
    }
}

static void
hrn_sidebar_item_get_property (GObject    *object,
                               guint       prop_id,
                               GValue     *value,
                               GParamSpec *pspec)
{
    switch (prop_id) {

    default:
        break;
    }
}

static void
hrn_sidebar_item_class_init (HrnSidebarItemClass *klass)
{
    GObjectClass *o_class = (GObjectClass *) klass;

    o_class->dispose = hrn_sidebar_item_dispose;
    o_class->finalize = hrn_sidebar_item_finalize;
    o_class->set_property = hrn_sidebar_item_set_property;
    o_class->get_property = hrn_sidebar_item_get_property;

    g_type_class_add_private (klass, sizeof (HrnSidebarItemPrivate));

    signals[SOURCE_CHANGED] = g_signal_new ("source-changed",
                                            G_TYPE_FROM_CLASS (klass),
                                            G_SIGNAL_NO_RECURSE |
                                            G_SIGNAL_RUN_FIRST, 0, NULL, NULL,
                                            hrn_marshal_VOID__OBJECT_INT_STRING,
                                            G_TYPE_NONE, 3,
                                            HRN_TYPE_SOURCE, G_TYPE_INT,
                                            G_TYPE_STRING);

    g_object_class_install_property (o_class, PROP_SOURCE,
                                     g_param_spec_object ("source", "", "",
                                                          HRN_TYPE_SOURCE,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_PIN_MANAGER,
                                     g_param_spec_object ("pin-manager", "", "",
                                                          HRN_TYPE_PIN_MANAGER,
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
    g_object_class_install_property (o_class, PROP_NAME,
                                     g_param_spec_string ("name", "", "", "",
                                                          G_PARAM_WRITABLE |
                                                          G_PARAM_STATIC_STRINGS));
}

static void
audio_item_clicked (NbtkWidget     *button,
                    HrnSidebarItem *item)
{
    select_subitem (item, HRN_SIDEBAR_SUBITEM (button));
    emit_source_changed (item, BKL_ITEM_TYPE_AUDIO, "");
}

static void
image_item_clicked (NbtkWidget     *button,
                    HrnSidebarItem *item)
{
    select_subitem (item, HRN_SIDEBAR_SUBITEM (button));
    emit_source_changed (item, BKL_ITEM_TYPE_IMAGE, "");
}

static void
video_item_clicked (NbtkWidget     *button,
                    HrnSidebarItem *item)
{
    select_subitem (item, HRN_SIDEBAR_SUBITEM (button));
    emit_source_changed (item, BKL_ITEM_TYPE_VIDEO, "");
}

struct _DefaultEntry {
    char *name;
    char *icon;
    GCallback callback;
};

static void
hrn_sidebar_item_init (HrnSidebarItem *self)
{
    HrnSidebarItemPrivate *priv = GET_PRIVATE (self);
    NbtkWidget *title_table;
    int i;
    struct _DefaultEntry default_entries[] = {
        { N_("Music"), "icon-audio", G_CALLBACK (audio_item_clicked) },
        { N_("Pictures"), "icon-images", G_CALLBACK (image_item_clicked) },
        { N_("Videos"), "icon-video", G_CALLBACK (video_item_clicked) },
        { NULL, NULL, NULL }
    };

    self->priv = priv;

    nbtk_widget_set_style_class_name (NBTK_WIDGET (self), "HrnSidebarItem");
    clutter_actor_set_reactive (CLUTTER_ACTOR (self), TRUE);

    title_table = nbtk_table_new ();
    nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                          (ClutterActor *) title_table, 0, 0,
                                          "x-align", 0.0,
                                          "y-align", 0.0,
                                          "y-fill", FALSE,
                                          "y-expand", FALSE,
                                          NULL);

    priv->title = nbtk_label_new ("");
    nbtk_widget_set_style_class_name (NBTK_WIDGET (priv->title),
                                      "HrnSidebarItemTitle");
    clutter_actor_set_width (CLUTTER_ACTOR (priv->title), SIDEBAR_WIDTH - 20);
    nbtk_table_add_actor (NBTK_TABLE (title_table),
                          (ClutterActor *) priv->title, 0, 0);

    priv->spinner = (ClutterActor *) hrn_spinner_new (PKGDATADIR "/hrn-spinner.png");
    clutter_actor_set_size (priv->spinner, 20, 20);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (title_table),
                                          priv->spinner,
                                          0, 1,
                                          "x-expand", FALSE,
                                          "x-fill", FALSE,
                                          "y-expand", FALSE,
                                          "y-fill", FALSE,
                                          NULL);
    clutter_actor_hide (priv->spinner);

    priv->child_grid = nbtk_grid_new ();
    nbtk_grid_set_max_stride (NBTK_GRID (priv->child_grid), 1);
    clutter_actor_set_width (CLUTTER_ACTOR (priv->child_grid),
                             SIDEBAR_WIDTH - 60);
    nbtk_table_add_actor_with_properties (NBTK_TABLE (self),
                                          (ClutterActor *) priv->child_grid,
                                          1, 0,
                                          "x-align", 0.0,
                                          "y-align", 0.0,
                                          "y-fill", FALSE,
                                          "y-expand", FALSE,
                                          NULL);

    /* These predefined audio/video/image searches should be created on demand
     * when there is audio images or video present in the library
     */
    for (i = 0; default_entries[i].name; i++) {
        HrnSidebarSubitem *subitem;

        subitem = hrn_sidebar_subitem_new ("predefined",
                                           _(default_entries[i].name),
                                           default_entries[i].icon,
                                           FALSE, FALSE);

        clutter_container_add_actor (CLUTTER_CONTAINER (priv->child_grid),
                                     CLUTTER_ACTOR (subitem));
        nbtk_bin_set_alignment (NBTK_BIN (subitem), NBTK_ALIGN_START,
                                NBTK_ALIGN_START);
        g_signal_connect (subitem, "clicked",
                          G_CALLBACK (default_entries[i].callback), self);
    }
}


HrnSidebarItem *
hrn_sidebar_item_new (const char    *name,
                      HrnSource     *source,
                      HrnPinManager *pin_manager)
{
    HrnSidebarItem *item;
    HrnSidebarItemPrivate *priv;

    item = g_object_new (HRN_TYPE_SIDEBAR_ITEM,
                         "pin-manager", pin_manager,
                         "source", source,
                         "name", name,
                         NULL);
    priv = item->priv;

    return item;
}

static void
unselect_subitems (ClutterActor *actor,
                   gpointer      data)
{
    nbtk_button_set_checked ((NbtkButton *) actor, FALSE);
}

void
hrn_sidebar_item_set_selected (HrnSidebarItem *item,
                               gboolean        selected)
{
    HrnSidebarItemPrivate *priv = item->priv;

    if (selected == FALSE) {
        clutter_container_foreach ((ClutterContainer *) priv->child_grid,
                                   unselect_subitems, item);
    }
}

void
hrn_sidebar_item_select_initial (HrnSidebarItem *item)
{
    HrnSidebarItemPrivate *priv = item->priv;
    GList *children;

    /* FIXME: Getting a whole list of the children, just to use the first item
       seems a bit wasteful, but our list should be short enough that its
       not *that* much extra work */
    children = clutter_container_get_children ((ClutterContainer *) priv->child_grid);
    if (children == NULL) {
        return;
    }

    nbtk_button_set_checked ((NbtkButton *) children->data, TRUE);
}
