/*
 * Hornsey - Moblin Media Player.
 * Copyright © 2009 Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <config.h>

#include <glib.h>

#include <glib/gi18n.h>
#include <gio/gio.h>
#include <stdlib.h>
#include <string.h>

#include <clutter/clutter.h>

#include <dbus/dbus-protocol.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <bickley/bkl-utils.h>
#include <bickley/bkl-source-client.h>
#include <bickley/bkl-source-manager-client.h>

#include <bognor/br-iface-player.h>
#include <bognor/br-queue.h>

#define HRN_SN_SUPPORT 1
#ifdef HRN_SN_SUPPORT
#define SN_API_NOT_YET_FROZEN
#include <libsn/sn.h>
#endif

#include <clutter/x11/clutter-x11.h>

#include <clutter-gtk/clutter-gtk.h>

#include "hrn.h"
#include "hrn-content-area.h"
#include "hrn-source-manager.h"
#include "hrn-state-manager.h"
#include "hrn-pin-manager.h"
#include "hrn-ui-controls.h"
#include "hrn-window.h"
#include "hrn-theatre-ui.h"

#include "math.h"

#include <unique/unique.h>

#define HORNSEY_DBUS_SERVICE    "org.moblin.Hornsey"
#define HORNSEY_DBUS_PATH       "/org/moblin/Hornsey"

#define DEFAULT_WIDTH 1024
#define DEFAULT_HEIGHT 600

HrnWindow *window;

ClutterActor *hrn_content_area;
ClutterActor *hrn_theatre_ui;

GtkRecentManager *recent_manager;

HrnQueue *master_queue; /* FIXME: Need queue manager */
ClutterActor *master_controls;
HrnPinManager *pin_manager;

HrnSourceManager *source_manager;
BrQueue *local_queue = NULL;

HrnStateManager *state_manager;

static gchar *load_path = NULL;

static void
set_source (HrnSource *source)
{
    const char *source_path;

    if (source) {
        hrn_content_area_set_source ((HrnContentArea *) hrn_content_area,
                                     source);
        return;
    }

    source_path = hrn_state_manager_get_source (state_manager);
    if (source_path == NULL || *source_path == '\0') {
        source = hrn_source_manager_get_local_source (source_manager);
    } else {
        source = hrn_source_manager_get_source_for_path (source_manager,
                                                         source_path);
        if (source == NULL) {
            source = hrn_source_manager_get_local_source (source_manager);
        }
    }

    hrn_content_area_set_source ((HrnContentArea *) hrn_content_area,
                                 source);
}

static UniqueResponse
hrn_message_received_cb (UniqueApp *app, gint command,
                         UniqueMessageData *message_data, guint time_,
                         gpointer user_data)
{
  char *uri;

  switch (command)
    {
      case UNIQUE_ACTIVATE:
        hrn_window_present (window);
        return UNIQUE_RESPONSE_OK;

      case UNIQUE_OPEN:
        uri = unique_message_data_get_text (message_data);
        if (uri && *uri != '\0') {
            BklItem *item;
            HrnSource *source;

            item = hrn_source_manager_get_item_for_uri (source_manager, uri,
                                                        &source);
            set_source (source);

            hrn_play_uri_now (uri);
        }

        g_free (uri);
        hrn_window_present (window);

        return UNIQUE_RESPONSE_OK;

      default:
        g_print ("uh?! %i\n", command);
        break;
    }
  return UNIQUE_RESPONSE_PASSTHROUGH;
}

static void
theatre_shown_cb (HrnTheatreUi *ui,
                  gboolean      shown,
                  gpointer      userdata)
{
    /* clutter_actor_hide (hrn_view); */
}

static void
theatre_hidden_cb (HrnTheatreUi *ui,
                   gboolean      shown,
                   gpointer      userdata)
{
    clutter_actor_hide ((ClutterActor *) ui);
    clutter_actor_lower_bottom ((ClutterActor *) ui);
}

static void
show_theatre (void)
{
    clutter_actor_show ((ClutterActor *) hrn_theatre_ui);
    clutter_actor_raise_top ((ClutterActor *) hrn_theatre_ui);

    hrn_theatre_ui_show ((HrnTheatreUi *) hrn_theatre_ui,
                         theatre_shown_cb, NULL);
    hrn_content_area_hide ((HrnContentArea *) hrn_content_area);
}

static void
activate_theatre_cb (HrnContentArea *area,
                     gboolean        activate,
                     gpointer        userdata)
{
  if (activate) {
      show_theatre ();
  } else {
      hrn_theatre_ui_hide ((HrnTheatreUi *) hrn_theatre_ui,
                           theatre_hidden_cb, NULL);

      hrn_content_area_show ((HrnContentArea *) hrn_content_area);
  }
}

static void
leave_theatre_cb (HrnTheatreUi *ui,
                  gpointer      userdata)
{
    hrn_theatre_ui_hide ((HrnTheatreUi *) hrn_theatre_ui,
                         theatre_hidden_cb, NULL);

    hrn_content_area_show ((HrnContentArea *) hrn_content_area);
}

#if 0
static void
zoom_changed_cb (HrnContentArea *area,
                 double          zl,
                 gpointer        userdata)
{
    hrn_state_manager_set_zoom (state_manager, zl);
}

void
hrn_loc_slider_update ()
{
  hrn_ui_controls_set_position ((HrnUiControls *) hrn_ui_controls,
                                hrn_view_get_loc ());
}
#endif

static void
query_changed_cb (HrnContentArea *area,
                  const char     *query,
                  gpointer        userdata)
{
  hrn_state_manager_set_query (state_manager, query);
}

static void
source_changed_cb (HrnContentArea *area,
                   HrnSource      *source,
                   gpointer        userdata)
{
    const char *source_path;

    source_path = hrn_source_get_object_path (source);
    hrn_state_manager_set_source (state_manager, source_path);
}

static void
filter_changed_cb (HrnContentArea *area,
                   int             filter,
                   gpointer        userdata)
{
    hrn_state_manager_set_filter (state_manager, filter);
}

#define SIDEBAR_WIDTH 180.0

static void
hrn_build_ui (void)
{
  ClutterActor *window_child = clutter_group_new ();
  int width, height;

  gtk_window_get_size (GTK_WINDOW (window), &width, &height);

  hrn_theatre_ui = g_object_new (HRN_TYPE_THEATRE_UI,
                                 "state-manager", state_manager,
                                 "local-queue", local_queue,
                                 NULL);
  g_signal_connect (hrn_theatre_ui, "leave-theatre",
                    G_CALLBACK (leave_theatre_cb), NULL);
  clutter_container_add_actor (CLUTTER_CONTAINER (window_child),
                               hrn_theatre_ui);
  clutter_actor_hide (CLUTTER_ACTOR (hrn_theatre_ui));
  clutter_actor_set_size (hrn_theatre_ui, width, height);

  hrn_content_area = g_object_new (HRN_TYPE_CONTENT_AREA,
                                   "pin-manager", pin_manager,
                                   "local-queue", local_queue,
                                   "source-manager", source_manager,
                                   "theatre-ui", hrn_theatre_ui,
                                   NULL);
  clutter_actor_set_size (hrn_content_area, width, height);
  clutter_actor_set_position (hrn_content_area, 0, 0);
  clutter_container_add_actor (CLUTTER_CONTAINER (window_child),
                               hrn_content_area);

  g_signal_connect (hrn_content_area, "filter-changed",
                    G_CALLBACK (filter_changed_cb), NULL);
  g_signal_connect (hrn_content_area, "source-changed",
                    G_CALLBACK (source_changed_cb), NULL);
  g_signal_connect (hrn_content_area, "query-changed",
                    G_CALLBACK (query_changed_cb), NULL);
  g_signal_connect (hrn_content_area, "activate-theatre",
                    G_CALLBACK (activate_theatre_cb), NULL);

  hrn_window_set_internal_actor (window, window_child);
}

void
hrn_play_uri_now (const char *uri)
{
    BklItem *item;
    HrnSource *source;

    item = hrn_source_manager_get_item_for_uri (source_manager, uri,
                                                &source);
    if (item) {
        if (g_str_has_prefix (bkl_item_get_mimetype (item), "audio/")) {
            hrn_queue_play_now (master_queue, item);
        } else {
            HrnClusterTree *tree;
            HrnClusterNode *node;

            tree = hrn_source_get_cluster_tree (source);
            node = g_hash_table_lookup (tree->uri_to_item,
                                        bkl_item_get_uri (item));
            hrn_theatre_ui_show_node ((HrnTheatreUi *) hrn_theatre_ui, node);
            show_theatre ();

            if (g_str_has_prefix (bkl_item_get_mimetype (item), "video/")) {
                hrn_theatre_ui_set_playing ((HrnTheatreUi *) hrn_theatre_ui,
                                            TRUE);
            }
        }
    } else {
        char *mimetype = hrn_resolve_mimetype (uri);

        if (mimetype == NULL) {
            g_warning ("Unknown mimetype for %s", uri);
            return;
        }

        if (g_str_has_prefix (mimetype, "audio/")) {
            hrn_queue_play_uri_now (master_queue, uri);
        } else if (g_str_has_prefix (mimetype, "image/")) {
            hrn_theatre_ui_show_uri ((HrnTheatreUi *) hrn_theatre_ui,
                                     uri, mimetype);
            show_theatre ();
        } else if (g_str_has_prefix (mimetype, "video/")) {
            hrn_theatre_ui_show_uri ((HrnTheatreUi *) hrn_theatre_ui,
                                     uri, mimetype);
            show_theatre ();
            hrn_theatre_ui_set_playing ((HrnTheatreUi *) hrn_theatre_ui, TRUE);
        }

        g_free (mimetype);
    }
}

static char *
make_uri (const char *filename)
{
    char *absolute_path, *current_dir;
    char *ret;

    if (strstr (filename, "://")) {
        return g_strdup (filename);
    }

    if (g_path_is_absolute (filename)) {
        return g_filename_to_uri (filename, NULL, NULL);
    }

    current_dir = g_get_current_dir ();
    absolute_path = g_build_filename (current_dir, filename, NULL);
    g_free (current_dir);

    ret = g_filename_to_uri (absolute_path, NULL, NULL);
    g_free (absolute_path);

    return ret;
}

static void
source_manager_ready_cb (HrnSourceManager *manager,
                         gpointer          userdata)
{
    HrnSource *source;

    hrn_state_manager_ready (state_manager);

    if (hrn_state_manager_get_force_visual_mode (state_manager)) {
        show_theatre ();
    }

    if (load_path) {
        char *uri = make_uri (load_path);
        BklItem *item;

        item = hrn_source_manager_get_item_for_uri (source_manager, uri,
                                                    &source);

        set_source (source);

        hrn_play_uri_now (uri);
        g_free (uri);
    } else if (!hrn_state_manager_get_force_visual_mode (state_manager)) {
        /* FIXME: Should this not be done by hrn_content_area_restore_state? */
        set_source (NULL);

#if 0
        /* Restore state */
        hrn_content_area_restore_state ((HrnContentArea *) hrn_content_area,
                                        state_manager);
#endif
    }

#ifdef HRN_SN_SUPPORT
    {
        SnLauncheeContext *context;

        context = g_object_get_data (G_OBJECT (window), "sn-context");
        if (context) {
            sn_launchee_context_complete (context);
            sn_launchee_context_unref (context);
        }
    }
#endif
}

#if 0
static void
remove_stage_decorations (ClutterStage *stage)
{
  struct
  {
    unsigned long flags;
    unsigned long functions;
    unsigned long decorations;
    long          inputMode;
    unsigned long status;
  }        MWMHints = { 2, 0, 0, 0, 0 };

  Display *xdisplay = clutter_x11_get_default_display ();
  Atom     wm_hints = XInternAtom (xdisplay, "_MOTIF_WM_HINTS", True);
  Window   xwindow  = clutter_x11_get_stage_window (CLUTTER_STAGE (stage));

  if (wm_hints != None)
    {
      XChangeProperty (xdisplay, xwindow, wm_hints, wm_hints, 32,
                       PropModeReplace, (guchar*) &MWMHints,
                       sizeof (MWMHints) / sizeof (long));
    }
}
#endif

static void
dbus_init (HrnStateManager *sm)
{
  DBusGConnection *connection;
  GError          *error = NULL;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL)
    {
      g_warning ("Error getting bus: %s", error->message);
      g_error_free (error);
      return;
    }

  /* We dont need to request the horney name as libunique has already
     done that for us */

  dbus_g_connection_register_g_object (connection, HORNSEY_DBUS_PATH,
                                       (GObject *) sm);
}

static GdkFilterReturn
gdk_to_clutter_event_forward (GdkXEvent *xevent,
                              GdkEvent  *event,
                              gpointer   data)
{
  switch (clutter_x11_handle_event ((XEvent *) xevent)) {
  default:
  case CLUTTER_X11_FILTER_CONTINUE:
    return GDK_FILTER_CONTINUE;
  case CLUTTER_X11_FILTER_TRANSLATE:
    return GDK_FILTER_TRANSLATE;
  case CLUTTER_X11_FILTER_REMOVE:
    return GDK_FILTER_REMOVE;
  }

  return GDK_FILTER_CONTINUE;
}

static void
window_resized (GtkWidget     *widget,
                GtkAllocation *allocation,
                gpointer       data)
{
  int width, height;

  gtk_window_get_size (GTK_WINDOW (window), &width, &height);
  if (hrn_theatre_ui) {
      clutter_actor_set_size (hrn_theatre_ui, width, height);
  }

  if (hrn_content_area) {
      clutter_actor_set_size (hrn_content_area, width, height);
  }
}

static gboolean
window_focus_in (GtkWidget     *widget,
                 GdkEventFocus *event,
                 gpointer       userdata)
{
    hrn_theatre_ui_set_focused (HRN_THEATRE_UI (hrn_theatre_ui), TRUE);
    return FALSE;
}

static gboolean
window_focus_out (GtkWidget     *widget,
                  GdkEventFocus *event,
                  gpointer       userdata)
{
    hrn_theatre_ui_set_focused (HRN_THEATRE_UI (hrn_theatre_ui), FALSE);
    return FALSE;
}

gint
main (gint argc, gchar **argv)
{
  UniqueApp   *app;

  gfloat       stage_width      = 0;
  gfloat       stage_height     = 0;
#ifdef HAVE_WINDOWMODE
  gboolean     stage_fullscreen = FALSE;
#else
  gboolean     stage_fullscreen = TRUE;
#endif

  GError *error = NULL;
  static gboolean window_mode = FALSE;
  static char *geometry = NULL;
  static char **files = NULL;

#ifdef HRN_SN_SUPPORT
  SnLauncheeContext *context;
  SnDisplay *display;
  Display *xdisplay;
  int screen;
#endif

  static GOptionEntry hrn_options[] = {
    { "window", 'w', 0, G_OPTION_ARG_NONE, &window_mode,
      N_("Run in a window"), NULL },
    { "size", 's', 0, G_OPTION_ARG_STRING, &geometry,
      N_("Size of the window"), "widthXheight" },
    { G_OPTION_REMAINING, 0, 0, G_OPTION_ARG_FILENAME_ARRAY, &files,
      NULL, N_("[FILE...]") },
    { NULL },
  };

  bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);

  g_thread_init (NULL);

  if (gtk_clutter_init_with_args (&argc, &argv, "Hornsey media player",
                                  hrn_options, NULL, &error) <= 0)
  {
      g_print ("Error: %s\n", error->message);
      g_error_free (error);
      return 1;
  }
  gdk_window_add_filter (NULL, gdk_to_clutter_event_forward, NULL);

#ifdef USE_HELIX
  clutter_helix_init (&argc, &argv);
#else
  clutter_gst_init (&argc, &argv);
#endif

  bkl_init ();
  notify_init ("Hornsey Media Player");

  clutter_set_font_flags (CLUTTER_FONT_HINTING);

  g_set_application_name ("Hornsey Media Player");

#ifdef HRN_SN_SUPPORT
  xdisplay = clutter_x11_get_default_display ();

  if (g_getenv ("LIBSN_SYNC") != NULL)
    XSynchronize (xdisplay, True);

  display = sn_display_new (xdisplay, NULL, NULL);
  screen  = clutter_x11_get_default_screen ();
  context = sn_launchee_context_new_from_environment (display, screen);
#endif

  app = unique_app_new ("org.moblin.Hornsey", NULL);

  if (unique_app_is_running (app))
    {
      if (files == NULL)
        {
          unique_app_send_message (app, UNIQUE_ACTIVATE, NULL);
        }
      else
        {
          UniqueMessageData *data = unique_message_data_new ();
          char *uri = make_uri (files[0]);

          /* FIXME: Send all the uris */
          unique_message_data_set_text (data, uri, -1);
          unique_app_send_message (app, UNIQUE_OPEN, data);
          unique_message_data_free (data);
          g_free (uri);
        }
      return 0;
    }

  recent_manager = gtk_recent_manager_get_default ();

  state_manager = g_object_new (HRN_TYPE_STATE_MANAGER, NULL);
  dbus_init (state_manager);

  local_queue = br_queue_new_local ();

  /* Setup the Bickley Source manager */
  source_manager = g_object_new (HRN_TYPE_SOURCE_MANAGER, NULL);
  g_signal_connect (source_manager, "ready",
                    G_CALLBACK (source_manager_ready_cb), NULL);

  nbtk_style_load_from_file (nbtk_style_get_default (),
                             PKGDATADIR "hornsey.css", NULL);

  if (window_mode == TRUE) {
    stage_fullscreen = FALSE;
  }

  if (geometry != NULL) {
    char *xspot;

    stage_width = atoi (geometry);

    xspot = strchr (geometry, 'x');
    if (xspot == NULL) {
      xspot = strchr (geometry, 'X');
    }

    if (xspot) {
      stage_height = atoi (xspot + 1);
    }
  }

  if (files != NULL) {
    /* FIXME: Handle all the uris */
    load_path = g_strdup (files[0]);
  }

  window = hrn_window_new (state_manager);
  gtk_window_set_resizable (GTK_WINDOW (window), TRUE);
  g_signal_connect (window, "size-allocate",
                    G_CALLBACK (window_resized), NULL);
  g_signal_connect (window, "focus-in-event",
                    G_CALLBACK (window_focus_in), NULL);
  g_signal_connect (window, "focus-out-event",
                    G_CALLBACK (window_focus_out), NULL);
  g_signal_connect (window, "destroy",
                    G_CALLBACK (gtk_main_quit), NULL);

  if (!stage_fullscreen)
    {
      stage_width  = DEFAULT_WIDTH;
      stage_height = DEFAULT_HEIGHT;
    }
  else if (stage_width == 0 &&
           stage_height == 0)
    {
      /* This causes us to read back the dimensions of the X desktop
       * using only Clutter APIs
       */
      stage_width  = gdk_screen_get_width (gdk_screen_get_default ());
      stage_height = gdk_screen_get_height (gdk_screen_get_default ());
    }

  gtk_window_resize (GTK_WINDOW (window), stage_width, stage_height);
  if (stage_fullscreen) {
    hrn_window_fullscreen (HRN_WINDOW (window));
  }

#ifdef HRN_SN_SUPPORT
  if (context) {
    g_object_set_data (G_OBJECT (window), "sn-context", context);
  }
#endif

  {
    gchar *search_path = hrn_make_config_file ("queries");

    /* Check for the old skool version as well */
    if (g_file_test (search_path, G_FILE_TEST_EXISTS) == FALSE) {
      g_free (search_path);

      g_warning ("Loading old deprecated query file");
      search_path = g_strdup_printf ("%s/.hrn-queries", g_get_home_dir ());
    }

    pin_manager = hrn_pin_manager_new (search_path);
    g_free (search_path);
  }

  hrn_build_ui ();

  gtk_window_set_title (GTK_WINDOW (window), _ ("Media Player"));

  g_signal_connect (app, "message-received",
                    G_CALLBACK (hrn_message_received_cb), NULL);

  gtk_widget_show_all (GTK_WIDGET (window));
  gtk_main ();

  {
    char *query_path = hrn_make_config_file ("queries");

    hrn_pin_manager_write_to_file (pin_manager, query_path);
    g_free (query_path);
  }

  {
    char *state_path = hrn_make_config_file ("saved-state");

    hrn_state_manager_write_to_file (state_manager, state_path);
    g_free (state_path);
  }

  bkl_shutdown ();

  return 0;
}

char *
hrn_make_config_file (const char *filename)
{
  const char *base;
  char *path, *full;

  base = g_getenv ("XDG_CONFIG_HOME");
  if (base) {
    path = g_strdup_printf ("%s/hornsey", base);
    full = g_strdup_printf ("%s/hornsey/%s", base, filename);
  } else {
    path = g_strdup_printf ("%s/.config/hornsey", g_get_home_dir ());
    full = g_strdup_printf ("%s/.config/hornsey/%s", g_get_home_dir (),
                            filename);
  }

  /* Make sure the directory exists */
  if (g_file_test (path, G_FILE_TEST_EXISTS) == FALSE) {
    g_mkdir_with_parents (path, 0700);
  }
  g_free (path);

  return full;
}

void
hrn_quit (void)
{
  gtk_main_quit ();
}

static const char *
hrn_guess_mimetype (const char *uri)
{
  GFile *file;
  GFileInfo *info;
  const char *mimetype = "unknown/unknown";

  file = g_file_new_for_commandline_arg (uri);
  info = g_file_query_info (file, "standard::content-type",
                            G_FILE_QUERY_INFO_NONE,
                            NULL, NULL);
  if (info) {
    mimetype = g_file_info_get_content_type (info);
  }

  if (!info ||
      !(g_str_has_prefix (mimetype, "image") ||
        g_str_has_prefix (mimetype, "video") ||
        g_str_has_prefix (mimetype, "audio")))
    {
      if (g_str_has_suffix (uri, "png")||
          g_str_has_suffix (uri, "PNG")||
          g_str_has_suffix (uri, "jpg")||
          g_str_has_suffix (uri, "JPG"))
        {
          mimetype = "image/foo";
        }
      else if (g_str_has_suffix (uri, "ogg")||
               g_str_has_suffix (uri, "OGG")||
               g_str_has_suffix (uri, "mp3")||
               g_str_has_suffix (uri, "MP3"))
        {
          mimetype = "audio/foo";
        }
      else if (g_str_has_suffix (uri, "mpg")||
               g_str_has_suffix (uri, "MPG")||
               g_str_has_suffix (uri, "asf")||
               g_str_has_suffix (uri, "avi")||
               g_str_has_suffix (uri, "AVI")||
               g_str_has_suffix (uri, "ogv")||
               g_str_has_suffix (uri, "OGV")||
               g_str_has_suffix (uri, "MP4")||
               g_str_has_suffix (uri, "rm")||
               g_str_has_suffix (uri, "mp4"))
        {
          mimetype = "video/foo";
        }
    }

  mimetype = g_strdup (mimetype);

  if (info)
    g_object_unref (info);
  g_object_unref (file);

  return mimetype;
}

/* do a best effort to retrieve a mimetype for uri,
 * either through bickley or gio,
 * ultimately falling back to extension based detection.
 */
char *
hrn_resolve_mimetype (const char *uri)
{
  BklItem *item;
  char *mimetype;

  item = hrn_source_manager_get_item_for_uri (source_manager, uri, NULL);
  if (item) {
    mimetype = g_strdup (bkl_item_get_mimetype (item));
  } else {
    mimetype = g_strdup (hrn_guess_mimetype (uri));
  }

  return mimetype;
}

void
hrn_add_recent_uri (const char *uri,
                    const char *mimetype)
{
    GtkRecentData data;
    gboolean ret;

    data.display_name = NULL;
    data.description = NULL;
    data.mime_type = g_strdup (mimetype);
    data.app_name = "Hornsey";
    data.app_exec= "hornset %u";
    data.groups = NULL;
    data.is_private = FALSE;

    ret = gtk_recent_manager_add_full (recent_manager, uri, &data);
    if (ret == FALSE) {
        g_warning ("Error registering recent use of %s", uri);
    }
}
