/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-sound.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

static gboolean 
get_provider_matches (GdmDevice *device)
{
        return (gdm_device_test_capability (device, "oss") || 
                gdm_device_test_capability (device, "alsa"));
}

static char *
get_icon_name (GdmDevice *device)
{
        const char *str;
        str = "gnome-stock-mic";
        return g_strdup (str);
}

static char *
get_short_name (GdmDevice *device)
{
        const char *type;
        const char *str;

        if (gdm_device_test_capability (device, "alsa")) {
                type = gdm_device_get_property_string (device, "alsa.type");

                if (strcmp (type, "control") == 0) {
                        str = _("ALSA Control");
                } else if (strcmp (type, "capture") == 0) {
                        str = _("ALSA Capture");
                } else if (strcmp (type, "midi") == 0) {
                        str = _("ALSA MIDI");
                } else if (strcmp (type, "playback") == 0) {
                        str = _("ALSA Playback");
                } else if (strcmp (type, "hw_specific") == 0) {
                        str = _("ALSA HW Specific");
                } else if (strcmp (type, "timer") == 0) {
                        str = _("ALSA Timer");
                } else if (strcmp (type, "sequencer") == 0) {
                        str = _("ALSA Sequencer");
                } else {
                        str = _("ALSA Unknown");
                }

        } else if (gdm_device_test_capability (device, "oss")) {
                type = gdm_device_get_property_string (device, "oss.type");

                if (strcmp (type, "mixer") == 0) {
                        str = _("OSS Mixer");
                } else if (strcmp (type, "pcm") == 0) {
                        str = _("OSS PCM");
                } else if (strcmp (type, "midi") == 0) {
                        str = _("OSS MIDI");
                } else if (strcmp (type, "sequencer") == 0) {
                        str = _("OSS Sequencer");
                } else {
                        str = _("OSS Unknown");
                }
        } else {
                str = _("Unknown Sound Device");
        }

        return g_strdup (str);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_short_name (device);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup ("");
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "input.product"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if (value != NULL) {                                                 \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, value);                               \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        GSList *p = NULL;

        if (gdm_device_test_capability (device, "oss")) {
                const char *card_id;
                const char *device_file;

                card_id = gdm_device_get_property_string (device, "oss.card_id");
                device_file = gdm_device_get_property_string (device, "oss.device_file");

                if (card_id != NULL)
                        ADD_SUM (p, _("Sound Card"), g_strdup (card_id));
                ADD_SUM (p, _("Device File"), g_strdup (device_file));
                
        } else if (gdm_device_test_capability (device, "alsa")) {
                const char *card_id;
                const char *device_id;
                const char *device_file;

                card_id = gdm_device_get_property_string (device, "alsa.card_id");
                device_file = gdm_device_get_property_string (device, "alsa.device_file");
                device_id = gdm_device_get_property_string (device, "alsa.device_id");

                if (card_id != NULL)
                        ADD_SUM (p, _("Sound Card"), g_strdup (card_id));
                if (device_id != NULL)
                        ADD_SUM (p, _("Sound Device"), g_strdup (device_id));
                ADD_SUM (p, _("Device File"), g_strdup (device_file));
        }


        return p;
}

GdmInfoProviderIface gdm_info_provider_sound =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
};

