/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-info-provider-pci.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-info-provider.h"

/* can't make this const due to i18n */
struct pci_class_id {
        int class_id;
        int subclass_id;
        int protocol_id;
        char *short_name;
        char *long_name;
} static pci_class_names[] = {
        {0x01,   -1,   -1, N_("Storage Controller"), N_("Mass Storage Controller")},
        {0x01, 0x00,   -1, N_("SCSI Controller"), N_("SCSI Storage Controller")},
        {0x01, 0x01,   -1, N_("IDE Controller"), N_("IDE Storage Controller")},
        {0x01, 0x02,   -1, N_("Floppy Controller"), N_("Floppy Disk Storage Controller")},
        {0x01, 0x03,   -1, N_("IPI Controller"), N_("IPI Bus Storage Controller")},
        {0x01, 0x04,   -1, N_("RAID Controller"), N_("Raid Bus Storage Controller")},
        {0x01, 0x05,   -1, N_("ATA Controller"), N_("ATA Storage Controller")},
        {0x01, 0x06,   -1, N_("SATA Controller"), N_("SATA Storage Controller")},
        {0x01, 0x06, 0x01, N_("AHCI Controller"), N_("SATA ACHI 1.0 Storage Controller")},
        {0x01, 0x07,   -1, N_("Serial SCSI Controller"), N_("Serial SCSI Storage Controller")},

        {0x02,   -1,   -1, N_("Network Controller"), N_("Network Controller")},
        {0x02, 0x00,   -1, N_("Ethernet Controller"), N_("Ethernet Network Controller")},
        {0x02, 0x01,   -1, N_("Token Ring Controller"), N_("Token Ring Network Controller")},
        {0x02, 0x02,   -1, N_("FDDI Controller"), N_("FDDI Network Controller")},
        {0x02, 0x03,   -1, N_("ATM Controller"), N_("ATM Network Controller")},
        {0x02, 0x03,   -1, N_("ISDN Controller"), N_("ISDN Network Controller")},

        {0x03,   -1,   -1, N_("Display Controller"), N_("Display Controller")},
        {0x03, 0x00,   -1, N_("VGA Controller"), N_("VGA Controller")},
        {0x03, 0x01,   -1, N_("XGA Controller"), N_("XGA Controller")},
        {0x03, 0x02,   -1, N_("3D Controller"), N_("3D Controller")},

        {0x04,   -1,   -1, N_("Multimedia Controller"), N_("Multimedia Controller")},
        {0x04, 0x00,   -1, N_("Video Controller"), N_("Video Controller")},
        {0x04, 0x01,   -1, N_("Audio Controller"), N_("Audio Controller")},
        {0x04, 0x02,   -1, N_("Telephony Controller"), N_("Telephony Controller")},
        {0x04, 0x03,   -1, N_("Audio Device"), N_("Audio Device")},

        {0x05,   -1,   -1, N_("Memory Controller"), N_("Memory Controller")},
        {0x05, 0x00,   -1, N_("RAM Controller"), N_("RAM Memory Controller")},
        {0x05, 0x01,   -1, N_("FLASH Controller"), N_("FLASH Memory Controller")},

        {0x06,   -1,   -1, N_("Bridge"), N_("Bridge")},
        {0x06, 0x00,   -1, N_("Host Bridge"), N_("Host Bridge")},
        {0x06, 0x01,   -1, N_("ISA Bridge"), N_("ISA Bridge")},
        {0x06, 0x02,   -1, N_("EISA Bridge"), N_("EISA Bridge")},
        {0x06, 0x03,   -1, N_("MicroChannel Bridge"), N_("MicroChannel Bridge")},
        {0x06, 0x04,   -1, N_("PCI Bridge"), N_("PCI Bridge")},
        {0x06, 0x04, 0x00, N_("PCI Bridge"), N_("PCI Bridge (Normal decode)")},
        {0x06, 0x04, 0x01, N_("PCI Bridge"), N_("PCI Bridge (Subtractive decode)")},
        {0x06, 0x05,   -1, N_("PCMCIA Bridge"), N_("PCMCIA Bridge")},
        {0x06, 0x06,   -1, N_("NuBus Bridge"), N_("NuBus Bridge")},
        {0x06, 0x07,   -1, N_("CardBus Bridge"), N_("CardBus Bridge")},
        {0x06, 0x08,   -1, N_("RACEway Bridge"), N_("RACEway Bridge")},
        {0x06, 0x09,   -1, N_("Semitrans. PCI-to-PCI Bridge"), N_("Semitransparent PCI-to-PCI Bridge")},
        {0x06, 0x0a,   -1, N_("InfiniBand to PCI Host Bridge"), N_("InfiniBand to PCI Host Bridge")},

        {0x07,   -1,   -1, N_("Communications Controller"), N_("Communications Controller")},
        {0x07, 0x00,   -1, N_("Serial Controller"), N_("Serial Controller")},
        {0x07, 0x00, 0x00, N_("8250 Serial Controller"), N_("8540 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16450 Serial Controller"), N_("16450 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16550 Serial Controller"), N_("16550 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16650 Serial Controller"), N_("16650 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16750 Serial Controller"), N_("16750 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16850 Serial Controller"), N_("16850 Serial Controller")},
        {0x07, 0x00, 0x00, N_("16950 Serial Controller"), N_("16950 Serial Controller")},
        {0x07, 0x01,   -1, N_("Parallel Controller"), N_("Parallel Controller")},
        {0x07, 0x01, 0x00, N_("SPP Parallel Controller"), N_("SPP Parallel Controller")},
        {0x07, 0x01, 0x01, N_("Bidir Parallel Controller"), N_("Bidirectional Parallel Controller")},
        {0x07, 0x01, 0x02, N_("ECP Parallel Controller"), N_("ECP Parallel Controller")},
        {0x07, 0x01, 0x03, N_("IEEE 1284 Parallel Controller"), N_("IEEE 1284 Parallel Controller")},
        {0x07, 0x01, 0xfe, N_("IEEE 1284 Target Parallel Controller"), N_("IEEE 1284 Target Parallel Controller")},
        {0x07, 0x02,   -1, N_("Multiport Serial Controller"), N_("Multiport Serial Controller")},
        {0x07, 0x03,   -1, N_("Modem"), N_("Modem")},
        {0x07, 0x03, 0x00, N_("Generic Modem"), N_("Generic Modem")},
        {0x07, 0x03, 0x01, N_("Hayes/16450 Modem"), N_("Hayes/16450 Compatiable Modem")},
        {0x07, 0x03, 0x02, N_("Hayes/16550 Modem"), N_("Hayes/16550 Compatiable Modem")},
        {0x07, 0x03, 0x03, N_("Hayes/16650 Modem"), N_("Hayes/16650 Compatiable Modem")},
        {0x07, 0x03, 0x04, N_("Hayes/16750 Modem"), N_("Hayes/16750 Compatiable Modem")},

        {0x08,   -1,   -1, N_("System Peripheral"), N_("Generic System Peripheral")},
        {0x08, 0x00,   -1, N_("PIC"), N_("PIC System Peripheral")},
        {0x08, 0x00, 0x00, N_("8259 PIC"), N_("8259 PIC System Peripheral")},
        {0x08, 0x00, 0x01, N_("ISA PIC"), N_("ISA PIC System Peripheral")},
        {0x08, 0x00, 0x02, N_("EISA PIC"), N_("EISA PIC System Peripheral")},
        {0x08, 0x00, 0x10, N_("IO-APIC"), N_("IO-APIC System Peripheral")},
        {0x08, 0x00, 0x20, N_("IO(X)-APIC"), N_("IO(X)-APIC System Peripheral")},
        {0x08, 0x01,   -1, N_("DMA Controller"), N_("DMA Controller")},
        {0x08, 0x01, 0x00, N_("8237 DMA Controller"), N_("DMA Controller")},
        {0x08, 0x01, 0x01, N_("ISA DMA Controller"), N_("ISA DMA Controller")},
        {0x08, 0x01, 0x02, N_("EISA DMA Controller"), N_("EISA DMA Controller")},
        {0x08, 0x02,   -1, N_("Timer Controller"), N_("Timer Controller")},
        {0x08, 0x02, 0x00, N_("8254 Timer Controller"), N_("8254 Timer Controller")},
        {0x08, 0x02, 0x01, N_("ISA Timer Controller"), N_("ISA Timer Controller")},
        {0x08, 0x02, 0x02, N_("EISA Timer Controller"), N_("EISA Timer Controller")},
        {0x08, 0x03,   -1, N_("Real-Time Clock"), N_("Real-Time Clock")},
        {0x08, 0x03, 0x00, N_("Generic Real-Time Clock"), N_("Generic Real-Time Clock")},
        {0x08, 0x03, 0x01, N_("ISA Real-Time Clock"), N_("ISA Real-Time Clock")},
        {0x08, 0x04,   -1, N_("PCI Hot-plug Controller"), N_("PCI Hot-plug Controller")},

        {0x09,   -1,   -1, N_("Input Controller"), N_("Input Device Controller")},
        {0x09, 0x00,   -1, N_("Keyboard Controller"), N_("Keyboard Controller")},
        {0x09, 0x01,   -1, N_("Digitizer Pen Controller"), N_("Digitizer Pen Controller")},
        {0x09, 0x02,   -1, N_("Mouse Controller"), N_("Mouse Controller")},
        {0x09, 0x03,   -1, N_("Scanner Controller"), N_("Scanner Controller")},
        {0x09, 0x04,   -1, N_("Gameport Controller"), N_("Gameport Controller")},
        {0x09, 0x04, 0x00, N_("Gameport Controller"), N_("Generic Gameport Controller")},
        {0x09, 0x04, 0x10, N_("Gameport Controller"), N_("Extended Gameport Controller")},

        {0x0a,   -1,   -1, N_("Docking Station"), N_("Docking Station")},
        {0x0a, 0x00,   -1, N_("Docking Station"), N_("Generic Docking Station")},

        {0x0b,   -1,   -1, N_("Processor"), N_("Processor")},
        {0x0b, 0x00,   -1, N_("386 Processor"), N_("386 Processor")},
        {0x0b, 0x01,   -1, N_("486 Processor"), N_("486 Processor")},
        {0x0b, 0x02,   -1, N_("Pentium Processor"), N_("Pentium Processor")},
        {0x0b, 0x10,   -1, N_("Alpha Processor"), N_("Alpha Processor")},
        {0x0b, 0x20,   -1, N_("Power PC Processor"), N_("Power PC Processor")},
        {0x0b, 0x30,   -1, N_("MIPS Processor"), N_("MIPS Processor")},
        {0x0b, 0x40,   -1, N_("Co-processor"), N_("Co-processor")},

        {0x0c,   -1,   -1, N_("Serial Bus Controller"), N_("Serial Bus Controller")},
        {0x0c, 0x00,   -1, N_("IEEE 1394 Controller"), N_("IEEE 1394 Controller")},
        {0x0c, 0x00, 0x10, N_("IEEE 1394 OHCI Controller"), N_("IEEE 1394 OHCI Controller")},
        {0x0c, 0x01,   -1, N_("ACCESS Bus Controller"), N_("ACCESS Bus Controller")},
        {0x0c, 0x02,   -1, N_("SSA Controller"), N_("SSA Controller")},
        {0x0c, 0x03,   -1, N_("USB Controller"), N_("USB Controller")},
        {0x0c, 0x03, 0x00, N_("USB UHCI Controller"), N_("USB UHCI Controller")},
        {0x0c, 0x03, 0x10, N_("USB OHCI Controller"), N_("USB OHCI Controller")},
        {0x0c, 0x03, 0x20, N_("USB EHCI Controller"), N_("USB EHCI Controller")},
        {0x0c, 0x04,   -1, N_("Fibre Channel Controller"), N_("Fibre Channel Controller")},
        {0x0c, 0x05,   -1, N_("SMBus Controller"), N_("SMBus Controller")},
        {0x0c, 0x06,   -1, N_("InfiniBand Controller"), N_("InfiniBand Controller")},

        {0x0d,   -1,   -1, N_("Wireless Controller"), N_("Wireless Controller")},
        {0x0d, 0x00,   -1, N_("IRDA Controller"), N_("IRDA Wireless Controller")},
        {0x0d, 0x01,   -1, N_("Consumer IR Controller"), N_("Consumer IR Wireless Controller")},
        {0x0d, 0x10,   -1, N_("RF Controller"), N_("RF Wireless Controller")},

        {0x0e,   -1,   -1, N_("Intelligent Controller"), N_("Intelligent Controller")},
        {0x0e,   -1,   -1, N_("I20 Controller"), N_("I20 Intelligent Controller")},

        {0x0f,   -1,   -1, N_("Satellite Comm. Controller"), N_("Satellite Communications Controller")},
        {0x0f, 0x00,   -1, N_("Satellite TV Controller"), N_("Satellite TV Communications Controller")},
        {0x0f, 0x01,   -1, N_("Satellite Audio Controller"), N_("Satellite Audio Communications Controller")},
        {0x0f, 0x03,   -1, N_("Satellite Voice Controller"), N_("Satellite Voice Communications Controller")},
        {0x0f, 0x04,   -1, N_("Satellite Data Controller"), N_("Satellite Data Communications Controller")},

        {0x10,   -1,   -1, N_("Encryption Controller"), N_("Encryption Controller")},
        {0x10, 0x00,   -1, N_("Network Encryption Device"), N_("Network and Computing Encryption Device")},

        {0x11,   -1,   -1, N_("Signal Processing Controller"), N_("Signal Processing Controller")},
        {0x11, 0x00,   -1, N_("DPIO Module"), N_("DPIO Module Signal Processing Controller")},
        {0x11, 0x01,   -1, N_("Performance Counters"), N_("Performance Counters")},
};

static const char *
get_class_name (int dc, int ds, int dp, gboolean get_short_name)
{
        int i;
        for (i = sizeof (pci_class_names) / sizeof (struct pci_class_id) - 1; i >= 0; i--) {
                struct pci_class_id *cid = pci_class_names + i;
                if ((cid->class_id == -1    || cid->class_id == dc) &&
                    (cid->subclass_id == -1 || cid->subclass_id == ds) &&
                    (cid->protocol_id == -1 || cid->protocol_id == dp)) {
                        if (get_short_name)
                                return cid->short_name;
                        else
                                return cid->long_name;
                }
        }
        return NULL;
}

static char *
get_name (GdmDevice *device, gboolean get_short_name)
{
        const char *s;
        char *name;
        int dc, ds, dp;

        dc = gdm_device_get_property_int (device, "pci.device_class");
        ds = gdm_device_get_property_int (device, "pci.device_subclass");
        dp = gdm_device_get_property_int (device, "pci.device_protocol");
        s = get_class_name (dc, ds, dp, get_short_name);
        if (s == NULL)
                name = g_strdup (_("PCI Device"));
        else
                name = g_strdup (s);

        return name;
}

static gboolean 
get_provider_matches (GdmDevice *device)
{
        const char *subsys;

        subsys = gdm_device_get_property_string (device, "info.subsystem");
        if (subsys != NULL && g_ascii_strcasecmp (subsys, "pci") == 0)
                return TRUE;

        return FALSE;
}

static char *
get_icon_name (GdmDevice *device)
{
        return g_strdup ("gnome-device-manager-device-pci");
}

static char *
get_short_name (GdmDevice *device)
{
        return get_name (device, TRUE);
}

static char *
get_long_name (GdmDevice *device)
{
        return get_name (device, FALSE);
}

static char *
get_vendor (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "pci.vendor"));
}

static char *
get_product (GdmDevice *device)
{
        return g_strdup (gdm_device_get_property_string (device, "pci.product"));
}

#define ADD_SUM(p, key, value)                                                       \
        do {                                                                         \
                if (value != NULL) {                                                 \
                        p = g_slist_append (p, g_strdup (key));                      \
                        p = g_slist_append (p, value);                               \
                }                                                                    \
        } while (FALSE)

static GSList *
get_summary (GdmDevice *device)
{
        const char *s;
        const char *s2;
        GSList *p = NULL;

        s = gdm_device_get_property_string (device, "pci.product");
        s2 = gdm_device_get_property_string (device, "pci.subsys_product");
        if (s != NULL) {
                if (s2 == NULL || g_ascii_strcasecmp (s, s2) == 0) {
                        ADD_SUM (p, _("Model"), g_strdup (s));
                } else {
                        ADD_SUM (p, _("Model"), g_strdup_printf ("%s (%s)", s, s2));
                }
        }
        
        s = gdm_device_get_property_string (device, "pci.vendor");
        s2 = gdm_device_get_property_string (device, "pci.subsys_vendor");
        if (s != NULL) {
                if (s2 == NULL || g_ascii_strcasecmp (s, s2) == 0) {
                        ADD_SUM (p, _("Vendor"), g_strdup (s));
                } else {
                        ADD_SUM (p, _("Vendor"), g_strdup_printf ("%s (%s)", s, s2));
                }
        }
        ADD_SUM (p, _("Connection"), g_strdup (_("PCI (Peripheral Component Interconnect)")));

        return p;
}

GdmInfoProviderIface gdm_info_provider_pci =
{
        .get_provider_matches = get_provider_matches,
        .get_icon_name        = get_icon_name,
        .get_short_name       = get_short_name,
        .get_long_name        = get_long_name,
        .get_vendor           = get_vendor,
        .get_product          = get_product,
        .get_summary          = get_summary,
};

