% FUNCTION [...] = gf_mesh_get(mesh M, [operation [, args]])
% 
%   General mesh inquiry function. All these functions accept also a
%   mesh_fem argument instead of a mesh M (in that case, the mesh_fem
%   linked mesh will be used). Note that when your mesh is recognized as
%   a Matlab object , you can simply use "get(M, 'dim')" instead of
%   "gf_mesh_get(M, 'dim')".
% 
%   * d = gf_mesh_get(M,'dim')
%   Get the dimension of the mesh (2 for a 2D mesh, etc).
% 
%   * P = gf_mesh_get(M,'pts'[, ivec PIDs])
%   Return the list of point coordinates of the mesh.  Each column of the returned
%   matrix contains the coordinates of one point. If the optional argument `PIDs`
%   was given, only the points whose #id is listed in this vector are returned.
%   Otherwise, the returned matrix will have gf_mesh_get(M,'max_pid') columns,
%   which might be greater than gf_mesh_get(M,'nbpts') (if some points of the mesh
%   have been destroyed and no call to gf_mesh_set(M,'optimize_structure') have
%   been issued). The columns corresponding to deleted points will be filled with
%   NaN. You can use gf_mesh_get(M,'pid') to filter such invalid points.
% 
%   * np = gf_mesh_get(M,'nbpts')
%   Get the number of points of the mesh.
% 
%   * nc = gf_mesh_get(M,'nbcvs')
%   Get the number of convexes of the mesh.
% 
%   * Pid = gf_mesh_get(M,'pid')
%   Return the list of points #id of the mesh.  Note that their numbering is not
%   supposed to be contiguous from 1 to gf_mesh_get(M,'nbpts'), especially if some
%   points have been removed from the mesh. You can use
%   gf_mesh_set(M,'optimize_structure') to enforce a contiguous numbering. Pid is
%   a row vector.
% 
%   * CVid = gf_mesh_get(M,'cvid')
%   Return the list of all convex #id.  Note that their numbering is not supposed
%   to be contiguous from 1 to gf_mesh_get(M,'nbcvs'), especially if some points
%   have been removed from the mesh. You can use
%   gf_mesh_set(M,'optimize_structure') to enforce a contiguous numbering. CVid is
%   a row vector.
% 
%   * m = gf_mesh_get(M,'max_pid')
%   Return the maximum #id of all points in the mesh (see 'max cvid').
% 
%   * m = gf_mesh_get(M,'max_cvid')
%   Return the maximum #id of all convexes in the mesh (see 'max pid').
% 
%   * {Pid, IDx} = gf_mesh_get(M,'pid_from_cvid'[,imat CVIDs])
%   Return the points attached to each convex of the mesh.  If `CVIDs` is omitted,
%   all the convexes will be considered (equivalent to `CVIDs =
%   gf_mesh_get(M,'max_cvid')`). `IDx` is a row vector, length(IDx) =
%   length(CVIDs)+1. `Pid` is a row vector containing the concatenated list of #id
%   of points of each convex in `CVIDs`. Each entry of `IDx` is the position of
%   the corresponding convex point list in `Pid`. Hence, for example, the list of
%   #id of points of the second convex is Pid(IDx(2):IDx(3)-1).
%   
%     If `CVIDs` contains convex #id which do not exist in the mesh, their point
%   list will be empty.
% 
%   * PIDs = gf_mesh_get(M,'pid_from_coords',mat PTS[, scalar radius=0])
%   Search point #id whose coordinates are listed in `PTS`.  `PTS` is an array
%   containing a list of point coordinates. On return, `PIDs` is a row vector
%   containing points #id for each point found in `eps` range, and -1 for those
%   which where not found in the mesh.
% 
%   * CVIDs = gf_mesh_get(M,'cvid_from_pid',ivec PIDs[, bool share=False])
%   Search convex #ids related with the point #ids given in `PIDs`.  If
%   `share=False`, search convex whose vertex #ids are in `PIDs`. If `share=True`,
%   search convex #ids that share the point #ids given in `PIDs`. `CVIDs` is a row
%   vector (possibly empty).
% 
%   * PIDs = gf_mesh_get(M,'orphaned_pid')
%   Search point #id which are not linked to a convex.
% 
%   * CVFIDs = gf_mesh_get(M,'faces_from_pid',ivec PIDs)
%   Return the convex faces whose vertex #ids are in `PIDs`.  For a convex face to
%   be returned, EACH of its points have to be listed in `PIDs`. On output, the
%   first row of `CVf` contains the convex number, and the second row contains the
%   face number (local number in the convex).
% 
%   * CVFIDs = gf_mesh_get(M,'faces_from_cvid'[, ivec CVIDs][, 'merge'])
%   Return a list of convexes faces from a list of convex #id.  `CVFIDs` is a two-
%   rows matrix, the first row lists convex #ids, and the second lists face
%   numbers. If `CVIDs` is not given, all convexes are considered. The optional
%   argument 'merge' merges faces shared by the convex of `CVIDs`.
% 
%   * CVFIDs = gf_mesh_get(M,'outer_faces'[, CVIDs])
%   Return the faces which are not shared by two convexes.  If `CVIDs` is not
%   given, it basically returns the mesh boundary. If `CVIDs` is given, it returns
%   the boundary of the convex set whose #ids are listed in `CVIDs`.
% 
%   * [E,C] = gf_mesh_get(M,'edges'[, CVLST][,'merge'])
%   [OBSOLETE FUNCTION! will be removed in a future release]  Return the list of
%   edges of mesh M for the convexes listed in the row vector CVLST. E is a 2 x
%   nb_edges matrix containing point indices. If CVLST is omitted, then the edges
%   of all convexes are returned. If CVLST has two rows then the first row is
%   supposed to contain convex numbers, and the second face numbers, of which the
%   edges will be returned.  If 'merge' is indicated, all common edges of convexes
%   are merged in a single edge.  If the optional output argument C is specified,
%   it will contain the convex number associated with each edge.
% 
%   * [E,C] = gf_mesh_get(M,'curved_edges',int N [, CVLST])
%   [OBSOLETE FUNCTION! will be removed in a future release]  More sophisticated
%   version of gf_mesh_get(M,'edges') designed for curved elements. This one will
%   return N (N>=2) points of the (curved) edges. With N==2, this is equivalent to
%   gf_mesh_get(M,'edges'). Since the points are no more always part of the mesh,
%   their coordinates are returned instead of points number, in the array E which
%   is a [ mesh_dim x 2 x nb_edges ] array.  If the optional output argument C is
%   specified, it will contain the convex number associated with each edge.
% 
%   * [mat T] = gf_mesh_get(M,'triangulated_surface',int Nrefine [,CVLIST])
%   [OBSOLETE FUNCTION! will be removed in a future release]  Similar function to
%   gf_mesh_get(M,'curved_edges') : split (if necessary, i.e. if the geometric
%   transformation if non-linear) each face into sub-triangles and return their
%   coordinates in T (see also gf_compute(MF, U,'eval on P1 tri mesh'))
% 
%   * N = gf_mesh_get(M,'normal_of_face',int cv, int f[, int nfpt])
%   Evaluates the normal of convex `cv`, face `f` at the `nfpt` point of the face.
%   If `nfpt` is not specified, then the normal is evaluated at each geometrical
%   node of the face.
% 
%   * N = gf_mesh_get(M,'normal_of_faces',imat CVFIDs)
%   Evaluates (at face centers) the normals of convexes.  `CVFIDs` is supposed to
%   contain convex numbers in its first row and convex face number in its second
%   row.
% 
%   * Q = gf_mesh_get(M,'quality'[,ivec CVIDs])
%   Return an estimation of the quality of each convex (0 <= Q <= 1).
% 
%   * {S, CV2S} = gf_mesh_get(M,'cvstruct'[, ivec CVIDs])
%   Return an array of the convex structures.  If `CVIDs` is not given, all
%   convexes are considered. Each convex structure is listed once in `S`, and
%   `CV2S` maps the convexes indice in `CVIDs` to the indice of its structure in
%   `S`.
% 
%   * {GT, CV2GT} = gf_mesh_get(M,'geotrans'[, ivec CVIDs])
%   Returns an array of the geometric transformations.  See also
%   gf_mesh_get(M,'cvstruct').
% 
%   * RIDs = gf_mesh_get(M,'regions')
%   Return the list of valid regions stored in the mesh.
% 
%   * CVFIDs = gf_mesh_get(M,'region',ivec RIDs)
%   Return the list of convexes/faces on the regions `RIDs`.  On output, the first
%   row of `CVFIDs` contains the convex numbers, and the second row contains the
%   face numbers (and 0  when the whole convex is in the regions).
% 
%   * gf_mesh_get(M,'save',string filename)
%   Save the mesh object to an ascii file.  This mesh can be restored with
%   gf_mesh('load', filename).
% 
%   * s = gf_mesh_get(M,'char')
%   Output a string description of the mesh.
% 
%   * gf_mesh_get(M,'export_to_vtk',string filename, ... [,'ascii'][,'quality'])
%   Exports a mesh to a VTK file .  If 'quality' is specified, an estimation of
%   the quality of each convex will be written to the file.  See also
%   gf_mesh_fem_get(MF,'export_to_vtk'), gf_slice_get(SL,'export_to_vtk').
% 
%   * gf_mesh_get(M,'export_to_dx',string filename, ...
%   [,'ascii'][,'append'][,'as',string name,[,'serie',string
%   serie_name]][,'edges'])
%   Exports a mesh to an OpenDX file.  See also
%   gf_mesh_fem_get(MF,'export_to_dx'), gf_slice_get(SL,'export_to_dx').
% 
%   * z = gf_mesh_get(M,'memsize')
%   Return the amount of memory (in bytes) used by the mesh.
% 
% 
%   $Id: gf_mesh_get.cc 2973 2009-04-17 00:05:39Z renard $
function [varargout]=gf_mesh_get(varargin)
  if (nargout),
    [varargout{1:nargout}]=gf_matlab('mesh_get',varargin{:});
  else
    gf_matlab('mesh_get',varargin{:});
    if (exist('ans','var') == 1), varargout{1}=ans; end;
  end;
