/*
** 2002-02-17 -	A brand-new options parser.
*/

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <gtk/gtk.h>

#include "options.h"

/* ----------------------------------------------------------------------------------------- */

/* Slightly hysterical function to print indented text, with word-wrapping. */
static void print_wrapped_and_indented(const gchar *text, gint indent, gint wrap)
{
	gchar	word[1024], *put;
	gint	len, wlen;

	printf("%*s", indent, "");
	for(len = indent; *text;)
	{
		while(isspace((guchar) *text))
			text++;
		for(put = word; *text && !isspace((guchar) *text) && (put - word + 1) < sizeof word;)
			*put++ = *text++;
		*put = '\0';
		wlen = strlen(word);
		if(len + wlen >= wrap)
		{
			printf("\n%*s", indent, "");
			len = indent;
		}
		printf("%s ", word);
		len += wlen + 1;
	}
	if(len != indent)
		printf("\n");
}

/* Print out as detailed usage information as we can possibly construct based on <desc>. */
static void show_usage(const OptDesc *desc, gsize num_desc)
{
	gchar		opt[1024], ns[2] = "?";
	const gchar	*nl;
	gsize		i, len;

	printf("Usage:\n");
	for(i = 0; i < num_desc; i++)
	{
		opt[0] = '\0';
		ns[0] = desc[i].name_short;
		nl    = desc[i].name_long;
		if(desc[i].arg_mode == OPT_ARG_NONE)
			g_snprintf(opt, sizeof opt, "%s%s%s%s%s", ns[0] ? "-" : "", ns, ns[0] && nl ? ", " : "", nl ? "--" : "", nl ? nl : "");
		else if(desc[i].arg_mode == OPT_ARG_OPT)
		{
			len = g_snprintf(opt, sizeof opt, "%s%s%s%s%s%s%s",
					ns[0] ? "-" : "", ns, ns[0] ? " [ARG]" : "", ns[0] && nl ? ", " : "",
					nl ? "--" : "", nl ? nl : "", nl ? "[=ARG]" : "");
			if(desc[i].arg)
				g_snprintf(opt + len, sizeof opt - len, " (default: %s)", (const gchar *) desc[i].arg);
		}
		else
			g_snprintf(opt, sizeof opt, "%s%s%s%s%s%s%s",
					ns[0] ? "-" : "", ns, ns[0] ? " ARG" : "", ns[0] && nl ? ", " : "",
					nl ? "--" : "", nl ? nl : "", nl ? "=ARG" : "");
		printf(" %s\n", opt);
		print_wrapped_and_indented(desc[i].help, 4, 70);
	}
}

/* ----------------------------------------------------------------------------------------- */

/* Delete <index>:th element of <argv>, moving all other elements downward. */
static void argv_delete(gchar **argv, gint index)
{
	for(; argv[index] != NULL; index++)
		argv[index] = argv[index + 1];
}

/* Return index of first element in <argv> that looks like an option. Returns -1 if none found. */
static gint argv_find_option(gchar **argv)
{
	gint	i;

	for(i = 1; argv[i] != NULL; i++)
	{
		if(strcmp(argv[i], "--") == 0)
			return -1;
		else if(argv[i][0] == '-')
			return i;
	}
	return -1;
}

/* Return the next word after <index>, as long as it doesn't start with '-'. */
static gchar * get_next_nonoption_word(gchar **argv, gint index)
{
	gchar	*arg;

	if((arg = argv[++index]) != NULL)
	{
		if(arg[0] != '-')
			argv_delete(argv, index);
		else
			arg = NULL;
	}
	return arg;
}

/* Get argument to short option. Either rest of string, if it exists, or the next word. */
static const gchar * short_get_arg(gchar **argv, gint index, const gchar *ptr)
{
	if(ptr[1])
		return ptr + 1;
	return get_next_nonoption_word(argv, index);
}

/* Store argument, taking care to build list if needed. */
static gboolean arg_store(const gchar *name, OptDesc *desc, const gchar *arg, gboolean is_short)
{
	if(arg)
	{
		if(desc->arg_mode == OPT_ARG_REQ_LIST)
			desc->arg = g_list_append(desc->arg, (gpointer) arg);
		else
			desc->arg = (gpointer) arg;
		return TRUE;
	}
	else if(desc->arg_mode == OPT_ARG_REQ || desc->arg_mode == OPT_ARG_REQ_LIST)
	{
		gchar	opt[64];

		if(is_short)
			g_snprintf(opt, sizeof opt, "-%c", desc->name_short);
		else
			g_snprintf(opt, sizeof opt, "--%s", desc->name_long);
		fprintf(stderr, "%s: Missing argument to option %s; try -h or --help for usage\n", name, opt);
		return FALSE;
	}
	return FALSE;
}

/* Handle a (bunch of) short option(s). */
static guint32 handle_short_options(gchar **argv, gint index, OptDesc *desc, gsize num_desc, guint32 *flags)
{
	const gchar	*ptr;
	gsize		i;

	for(ptr = argv[index] + 1; *ptr; ptr++)
	{
		for(i = 0; i < num_desc; i++)
		{
			if(*ptr == desc[i].name_short)
				break;
		}
		if(i < num_desc)
		{
			*flags |= desc[i].flag;
			if(desc[i].arg_mode)
			{
				const gchar	*arg;

				arg = short_get_arg(argv, index, ptr);
				if(!arg_store(argv[0], desc + i, arg, TRUE))
					return 0;
				break;
			}
		}
		else
		{
			fprintf(stderr, "%s: Unknown option -%c; try -h or --help for usage info\n", argv[0], *ptr);
			return 0;
		}
	}
	return 1;
}

/* Handle a single --long option. */
static gint handle_long_option(gchar **argv, gint index, OptDesc *desc, gsize num_desc, guint32 *flags)
{
	gchar		*opt = argv[index] + 2, *ptr;
	const gchar	*arg = NULL;
	gsize		i;

	/* If there's an equals sign, remove it so option is recognizable. */
	if((ptr = strchr(opt, '=')) != NULL)
	{
		*ptr = '\0';
		arg  = ptr + 1;
		if(*arg == '\0')
		{
			fprintf(stderr, "%s: Missing value after = in %s\n", argv[0], argv[index]);
			return 0;
		}
	}
	for(i = 0; i < num_desc; i++)
	{
		if(desc[i].name_long && strcmp(opt, desc[i].name_long) == 0)
		{
			if(desc[i].arg_mode)
			{
				arg = arg ? arg : get_next_nonoption_word(argv, index);
				if(!arg_store(argv[0], desc + i, arg, FALSE))
					return 0;
			}
			else if(arg != NULL)
				fprintf(stderr, "%s: Superfluous argument to option --%s: \"%s\"", argv[0], desc[i].name_long, arg);
			*flags |= desc[i].flag;
			return 1;
		}
	}
	fprintf(stderr, "%s: Unknown option '--%s'; try -h or --help for usage info\n", argv[0], opt);
	return 0;
}

/* Parse command line options, as described by <argc> and <argv>, using information in
** <desc>. Returns mask of seen flags, and also modifies <desc>->arg members by filling
** in option arguments when encountered. On error, <argc> will be set to something < 0.
*/
guint32 opt_parse(gint *argc, gchar **argv, OptDesc *desc, gsize num_desc)
{
	guint32	flags = 0U;
	gint	i, ok, index;

	/* First check for "-h" or "--help", and handle them specially. */
	for(i = 1; argv[i] != NULL; i++)
	{
		if(strcmp(argv[i], "--") == 0)
			break;
		if(strcmp(argv[i], "-h") == 0 || strcmp(argv[i], "--help") == 0)
		{
			show_usage(desc, num_desc);
			*argc = -1;
			return 0U;
		}
	}
	/* Now handle all option-looking arguments. */
	for(ok = 1; ok && (index = argv_find_option(argv)) > 0;)
	{
		if(argv[index][1] != '-')
			ok = handle_short_options(argv, index, desc, num_desc, &flags);
		else
			ok = handle_long_option(argv, index, desc, num_desc, &flags);
		argv_delete(argv, index);
	}
	/* Finally, clean up the argv vector and adjust the length. */
	for(i = 1; argv[i] != NULL;)
	{
		if(strcmp(argv[i], "--") == 0)
			argv_delete(argv, i);
		else
			i++;
	}
	*argc = ok ? i : -2;

	return flags;
}

/* ----------------------------------------------------------------------------------------- */

/* Return the argument for option <name>, if found. If <name> is single-character, it's
** interpreted as a short name; otherwise it's long. No leading hyphens, please.
*/
gconstpointer opt_arg_get(const gchar *name, const OptDesc *desc, gsize num_desc)
{
	gsize	i;

	if(name == NULL)
		return NULL;

	for(i = 0; i < num_desc; i++)
	{
		if((name[1] == '\0' && desc[i].name_short == name[0]) || strcmp(name, desc[i].name_long) == 0)
			return desc[i].arg;
	}
	fprintf(stderr, "opt_arg_get() called with unknown option \"%s\"\n", name);
	return NULL;
}
