# 25 "cil/src/logic/logic_lexer.mll"
 

  open Logic_parser
  open Lexing
  open Logic_ptree

  type state = Normal | Test

  let state_stack = Stack.create ()

  let () = Stack.push Normal state_stack

  let get_state () = try Stack.top state_stack with Stack.Empty -> Normal

  let pop_state () = try ignore (Stack.pop state_stack) with Stack.Empty -> ()

  exception Error of (int * int) * string

  let loc lexbuf = (lexeme_start lexbuf, lexeme_end lexbuf)

  let lex_error lexbuf s =
    raise (Error (loc lexbuf, "lexical error, " ^ s))

  let find_utf8 =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [ Utf8_logic.forall, FORALL;
        Utf8_logic.exists, EXISTS;
        Utf8_logic.eq, EQ;
        Utf8_logic.neq, NE;
        Utf8_logic.le, LE;
        Utf8_logic.ge, GE;
        Utf8_logic.implies,IMPLIES;
        Utf8_logic.iff, IFF;
        Utf8_logic.conj, AND;
        Utf8_logic.disj, OR;
        Utf8_logic.neg, NOT;
        Utf8_logic.x_or, HATHAT;
        Utf8_logic.minus, MINUS;
        Utf8_logic.boolean, BOOLEAN;
        Utf8_logic.integer, INTEGER;
        Utf8_logic.real, REAL
      ];

    fun s -> try Hashtbl.find h s
    with Not_found -> IDENTIFIER s

  let identifier =
    let all_kw = Hashtbl.create 37 in
    let c_kw = Hashtbl.create 37 in
    let type_kw = Hashtbl.create 3 in
    List.iter
      (fun (i,t,flag) ->
         Hashtbl.add all_kw i t;
         if flag then Hashtbl.add c_kw i t
      )
      [
        "assert", ASSERT, false;
        "assigns", ASSIGNS, false;
        "assumes", ASSUMES, false;
        "axiom", AXIOM, false;
        "axiomatic", AXIOMATIC, false;
        "behavior", BEHAVIOR, false;
        "behaviors", BEHAVIORS, false;
        "breaks", BREAKS, false;
	"case", CASE, true;
        "char", CHAR, true;
        "complete", COMPLETE, false;
        "continues", CONTINUES, false;
        "decreases", DECREASES, false;
        "disjoint", DISJOINT, false;
        "double", DOUBLE, true;
        "else", ELSE, true;
        "ensures", ENSURES, false ;
        "enum", ENUM, true;
        "exits", EXITS, false;
        "float", FLOAT, true;
        "for", FOR, true;
        "if", IF, true;
	"inductive", INDUCTIVE, false;
        "int", INT, true;
        "invariant", INVARIANT, false;
        "global",    GLOBAL, false;
        "label", LABEL, false;
        "lemma", LEMMA, false;
        "logic", LOGIC, false;
        "long", LONG, true;
        "loop", LOOP, false;
        "pragma", PRAGMA, false;
        "predicate", PREDICATE, false;
        "reads", READS, false;
        "requires", REQUIRES, false;
        "returns", RETURNS, false;
        "short", SHORT, true;
        "signed", SIGNED, true;
        "sizeof", SIZEOF, true;
        "slice", SLICE, false;
	"impact", IMPACT, false;
        "struct", STRUCT, true;
        "terminates", TERMINATES, false;
        "type", TYPE, false;
        "union", UNION, true;
        "unsigned", UNSIGNED, true;
        "variant", VARIANT, false;
        "void", VOID, true;
      ];
    List.iter (fun (x, y) -> Hashtbl.add type_kw x y)
      ["integer", INTEGER; "real", REAL; "boolean", BOOLEAN; ];
    fun s ->
      try
        Hashtbl.find (if Logic_utils.is_kw_c_mode () then c_kw else all_kw) s
      with Not_found ->
        if Logic_env.typename_status s then TYPENAME s
        else
          (try
             Hashtbl.find type_kw s
           with Not_found ->
             if Logic_utils.is_rt_type_mode () then TYPENAME s
             else IDENTIFIER s)

  let bs_identifier =
    let h = Hashtbl.create 97 in
    List.iter (fun (i,t) -> Hashtbl.add h i t)
      [
        "\\at", AT;
        "\\base_addr", BASE_ADDR;
        "\\block_length", BLOCK_LENGTH;
        "\\empty", EMPTY;
        "\\exists", EXISTS;
        "\\false", FALSE;
        "\\forall", FORALL;
        "\\fresh", FRESH;
        "\\from", FROM;
        "\\inter", INTER;
        "\\lambda", LAMBDA;
        "\\let", LET;
        "\\nothing", NOTHING;
        "\\null", NULL;
        "\\old", OLD;
        "\\result", RESULT;
        "\\separated", SEPARATED;
        "\\true", TRUE;
        "\\type", BSTYPE;
        "\\typeof", TYPEOF;
        "\\union", BSUNION;
        "\\valid", VALID;
        "\\valid_index", VALID_INDEX;
        "\\valid_range", VALID_RANGE;
      ];
    fun lexbuf ->
      let s = lexeme lexbuf in
      try Hashtbl.find h s with Not_found -> IDENTIFIER s


  let int_of_digit chr =
    match chr with
        '0'..'9' -> (Char.code chr) - (Char.code '0')
      | 'a'..'f' -> (Char.code chr) - (Char.code 'a') + 10
      | 'A'..'F' -> (Char.code chr) - (Char.code 'A') + 10
      | _ -> assert false

  (* Update lexer buffer. *)
  let update_newline_loc lexbuf =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = pos.Lexing.pos_lnum + 1;
	Lexing.pos_bol = pos.Lexing.pos_cnum;
      }

  (* Update lexer buffer. *)
  let update_line_loc lexbuf line absolute chars =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_lnum = if absolute then line else pos.Lexing.pos_lnum + line;
	Lexing.pos_bol = pos.Lexing.pos_cnum - chars;
      }

  (* Update lexer buffer. *)
  let update_file_loc lexbuf file =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_fname = file;
      }

  (* Update lexer buffer. *)
  let update_bol_loc lexbuf bol =
    let pos = lexbuf.Lexing.lex_curr_p in
    lexbuf.Lexing.lex_curr_p <-
      { pos with
	Lexing.pos_bol = bol;
      }

# 198 "cil/src/logic/logic_lexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\197\255\198\255\000\000\204\255\205\255\206\255\207\255\
    \208\255\209\255\196\000\217\255\219\255\165\000\224\255\225\255\
    \202\000\232\255\233\255\199\000\235\255\236\255\201\000\139\000\
    \227\000\208\000\206\000\242\255\004\001\225\000\006\001\254\000\
    \036\001\109\001\198\001\000\002\236\000\254\255\030\001\091\001\
    \252\255\253\255\075\002\188\002\127\002\128\002\129\002\161\002\
    \246\255\189\002\191\002\192\002\243\255\158\002\077\001\208\000\
    \187\002\001\003\024\003\097\003\061\003\167\003\211\000\200\003\
    \232\003\009\004\041\004\082\003\245\255\051\004\080\004\112\004\
    \122\004\141\004\174\004\206\004\216\004\212\000\235\004\245\004\
    \244\255\011\005\021\005\050\005\060\005\089\005\099\005\128\005\
    \239\000\139\005\168\005\178\005\207\005\227\255\224\000\241\255\
    \201\255\203\255\228\000\228\000\240\255\239\255\238\255\210\255\
    \234\255\202\255\230\255\222\255\200\255\213\255\200\000\251\255\
    \252\255\239\005\253\255\097\001\056\006\079\006\190\001\254\255\
    \009\006\251\255\186\000\255\005\254\255\255\255\202\000\212\000\
    \252\255\239\002\252\255\224\001\254\255\255\255\253\255\242\002\
    \092\001\253\255\254\255\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\056\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\040\000\255\255\255\255\034\000\255\255\255\255\
    \024\000\255\255\255\255\029\000\255\255\255\255\018\000\035\000\
    \032\000\039\000\044\000\255\255\058\000\026\000\058\000\008\000\
    \008\000\005\000\005\000\058\000\037\000\255\255\000\000\255\255\
    \255\255\255\255\004\000\255\255\255\255\255\255\255\255\009\000\
    \255\255\255\255\255\255\012\000\255\255\010\000\255\255\008\000\
    \007\000\255\255\255\255\006\000\255\255\255\255\006\000\255\255\
    \255\255\255\255\010\000\255\255\255\255\010\000\255\255\010\000\
    \255\255\255\255\255\255\010\000\255\255\007\000\010\000\255\255\
    \255\255\255\255\010\000\255\255\010\000\255\255\010\000\255\255\
    \027\000\010\000\255\255\010\000\255\255\255\255\043\000\255\255\
    \255\255\255\255\041\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\004\000\255\255\001\000\255\255\000\000\001\000\255\255\
    \255\255\255\255\004\000\002\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\003\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\003\000\000\000\000\000\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\000\000\000\000\255\255\000\000\000\000\
    \255\255\000\000\000\000\255\255\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\000\000\049\000\255\255\045\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\039\000\
    \000\000\000\000\255\255\049\000\045\000\045\000\045\000\045\000\
    \000\000\049\000\049\000\049\000\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\000\000\
    \000\000\000\000\255\255\255\255\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\111\000\000\000\
    \000\000\255\255\000\000\255\255\255\255\255\255\255\255\000\000\
    \121\000\000\000\255\255\255\255\000\000\000\000\255\255\255\255\
    \000\000\130\000\000\000\135\000\000\000\000\000\000\000\135\000\
    \137\000\000\000\000\000\000\000";
  Lexing.lex_trans = 
   "\001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\038\000\037\000\001\000\038\000\038\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \001\000\001\000\001\000\001\000\001\000\001\000\001\000\001\000\
    \038\000\022\000\028\000\027\000\021\000\011\000\024\000\030\000\
    \009\000\008\000\014\000\015\000\020\000\019\000\029\000\036\000\
    \032\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \031\000\031\000\016\000\017\000\025\000\026\000\010\000\018\000\
    \038\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\034\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\005\000\035\000\004\000\013\000\033\000\
    \001\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\007\000\023\000\006\000\012\000\001\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\001\000\
    \002\000\109\000\108\000\107\000\106\000\104\000\103\000\102\000\
    \105\000\101\000\097\000\094\000\096\000\098\000\255\255\088\000\
    \255\255\089\000\089\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\039\000\055\000\093\000\095\000\062\000\
    \077\000\099\000\100\000\126\000\113\000\055\000\255\255\038\000\
    \062\000\077\000\038\000\038\000\053\000\255\255\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \127\000\128\000\000\000\000\000\055\000\000\000\038\000\062\000\
    \077\000\000\000\000\000\054\000\000\000\055\000\000\000\000\000\
    \062\000\077\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\053\000\055\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\038\000\000\000\
    \050\000\000\000\046\000\054\000\000\000\041\000\139\000\000\000\
    \000\000\054\000\055\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\000\000\000\000\055\000\000\000\000\000\000\000\000\000\
    \079\000\055\000\079\000\000\000\057\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\054\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \055\000\118\000\118\000\118\000\118\000\118\000\118\000\118\000\
    \118\000\055\000\000\000\000\000\057\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \112\000\000\000\000\000\000\000\033\000\000\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \043\000\255\255\000\000\000\000\255\255\044\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \000\000\000\000\134\000\000\000\255\255\000\000\255\255\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\000\000\000\000\000\000\000\000\033\000\000\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\040\000\138\000\000\000\000\000\042\000\
    \000\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\048\000\
    \047\000\000\000\042\000\255\255\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\255\255\255\255\
    \048\000\255\255\255\255\000\000\080\000\000\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\082\000\
    \000\000\000\000\000\000\046\000\046\000\046\000\255\255\052\000\
    \255\255\051\000\052\000\081\000\068\000\000\000\000\000\000\000\
    \000\000\053\000\068\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\000\000\000\000\000\000\
    \132\000\133\000\000\000\255\255\132\000\046\000\255\255\000\000\
    \054\000\000\000\000\000\081\000\068\000\000\000\000\000\077\000\
    \000\000\000\000\068\000\000\000\000\000\000\000\000\000\132\000\
    \077\000\131\000\000\000\000\000\134\000\000\000\000\000\000\000\
    \050\000\050\000\000\000\050\000\050\000\000\000\000\000\000\000\
    \054\000\000\000\000\000\000\000\000\000\000\000\000\000\077\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\058\000\
    \077\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\073\000\073\000\073\000\073\000\073\000\073\000\000\000\
    \000\000\000\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \072\000\000\000\072\000\000\000\000\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\000\000\
    \000\000\073\000\073\000\073\000\073\000\073\000\073\000\255\255\
    \255\255\255\255\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\000\000\000\000\000\000\061\000\
    \000\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\059\000\059\000\059\000\059\000\059\000\059\000\
    \000\000\000\000\000\000\000\000\000\000\062\000\000\000\000\000\
    \000\000\060\000\000\000\000\000\000\000\000\000\062\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\255\255\000\000\255\255\
    \255\255\000\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \000\000\000\000\000\000\000\000\000\000\062\000\000\000\000\000\
    \000\000\060\000\000\000\000\000\000\000\000\000\062\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\000\000\255\255\
    \000\000\000\000\255\255\070\000\000\000\070\000\000\000\063\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\063\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\064\000\064\000\064\000\064\000\064\000\064\000\000\000\
    \000\000\000\000\000\000\000\000\067\000\000\000\067\000\000\000\
    \065\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\064\000\064\000\064\000\064\000\064\000\064\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \065\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\000\000\000\000\068\000\
    \000\000\000\000\000\000\000\000\000\000\068\000\000\000\000\000\
    \000\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \000\000\000\000\000\000\000\000\000\000\068\000\000\000\000\000\
    \000\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\000\000\000\000\068\000\000\000\
    \000\000\000\000\000\000\000\000\068\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\000\000\000\000\068\000\000\000\
    \000\000\076\000\000\000\076\000\068\000\074\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\074\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \075\000\075\000\000\000\000\000\068\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\000\000\
    \000\000\068\000\000\000\000\000\068\000\000\000\087\000\068\000\
    \087\000\000\000\068\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\000\000\
    \000\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \000\000\000\000\083\000\068\000\000\000\085\000\000\000\085\000\
    \000\000\068\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\000\000\000\000\
    \000\000\000\000\083\000\068\000\000\000\000\000\000\000\000\000\
    \000\000\068\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \068\000\084\000\084\000\084\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\000\000\000\000\000\000\
    \000\000\000\000\068\000\000\000\000\000\000\000\000\000\000\000\
    \068\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \086\000\086\000\086\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\000\000\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\000\000\000\000\000\000\
    \000\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \090\000\068\000\000\000\092\000\000\000\092\000\000\000\068\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\000\000\000\000\000\000\000\000\
    \090\000\068\000\000\000\000\000\000\000\000\000\000\000\068\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\068\000\091\000\
    \091\000\091\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\114\000\124\000\125\000\000\000\000\000\114\000\000\000\
    \068\000\000\000\000\000\000\000\000\000\000\000\068\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\000\000\
    \000\000\124\000\000\000\000\000\000\000\000\000\114\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\123\000\123\000\000\000\000\000\000\000\000\000\116\000\
    \000\000\000\000\000\000\114\000\000\000\000\000\000\000\000\000\
    \114\000\114\000\000\000\000\000\000\000\114\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\114\000\000\000\000\000\
    \000\000\114\000\000\000\114\000\000\000\114\000\000\000\116\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\000\000\000\000\000\000\122\000\000\000\000\000\
    \000\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\000\000\000\000\
    \000\000\117\000\117\000\117\000\117\000\117\000\117\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    ";
  Lexing.lex_check = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\010\000\010\000\013\000\016\000\019\000\022\000\023\000\
    \016\000\024\000\025\000\026\000\025\000\025\000\028\000\029\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\036\000\055\000\088\000\094\000\062\000\
    \077\000\098\000\099\000\122\000\110\000\055\000\028\000\038\000\
    \062\000\077\000\038\000\038\000\031\000\030\000\031\000\031\000\
    \031\000\031\000\031\000\031\000\031\000\031\000\031\000\031\000\
    \126\000\127\000\255\255\255\255\055\000\255\255\038\000\062\000\
    \077\000\255\255\255\255\031\000\255\255\055\000\255\255\255\255\
    \062\000\077\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\032\000\031\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\038\000\255\255\
    \028\000\255\255\030\000\031\000\255\255\039\000\136\000\255\255\
    \255\255\032\000\031\000\255\255\255\255\255\255\255\255\255\255\
    \032\000\255\255\255\255\031\000\255\255\255\255\255\255\255\255\
    \054\000\032\000\054\000\255\255\032\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\255\255\
    \255\255\032\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \032\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\032\000\255\255\255\255\032\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \110\000\255\255\255\255\255\255\033\000\255\255\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \033\000\033\000\033\000\033\000\033\000\033\000\033\000\033\000\
    \034\000\131\000\255\255\255\255\131\000\034\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \255\255\255\255\131\000\255\255\028\000\255\255\030\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\255\255\255\255\255\255\255\255\034\000\255\255\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\039\000\136\000\255\255\255\255\035\000\
    \255\255\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\035\000\035\000\035\000\035\000\035\000\
    \035\000\035\000\035\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\255\255\255\255\255\255\
    \255\255\044\000\045\000\046\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\044\000\045\000\
    \046\000\255\255\042\000\047\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\042\000\042\000\
    \042\000\042\000\042\000\042\000\042\000\042\000\043\000\049\000\
    \047\000\050\000\051\000\255\255\053\000\255\255\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \255\255\255\255\255\255\044\000\045\000\046\000\043\000\049\000\
    \131\000\050\000\051\000\053\000\053\000\255\255\255\255\255\255\
    \255\255\056\000\053\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\255\255\255\255\255\255\
    \129\000\129\000\255\255\135\000\129\000\047\000\135\000\255\255\
    \056\000\255\255\255\255\053\000\053\000\255\255\255\255\056\000\
    \255\255\255\255\053\000\255\255\255\255\255\255\255\255\129\000\
    \056\000\129\000\255\255\255\255\135\000\255\255\255\255\255\255\
    \043\000\049\000\255\255\050\000\051\000\255\255\255\255\255\255\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\056\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \056\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\058\000\058\000\058\000\058\000\058\000\058\000\255\255\
    \255\255\255\255\057\000\057\000\057\000\057\000\057\000\057\000\
    \060\000\255\255\060\000\255\255\255\255\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\255\255\
    \255\255\058\000\058\000\058\000\058\000\058\000\058\000\044\000\
    \045\000\046\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\255\255\255\255\255\255\059\000\
    \255\255\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\047\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \255\255\255\255\255\255\255\255\255\255\059\000\255\255\255\255\
    \255\255\059\000\255\255\255\255\255\255\255\255\059\000\255\255\
    \255\255\255\255\255\255\255\255\043\000\049\000\255\255\050\000\
    \051\000\255\255\059\000\059\000\059\000\059\000\059\000\059\000\
    \255\255\255\255\255\255\255\255\255\255\059\000\255\255\255\255\
    \255\255\059\000\255\255\255\255\255\255\255\255\059\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\255\255\129\000\
    \255\255\255\255\135\000\063\000\255\255\063\000\255\255\061\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \061\000\061\000\061\000\061\000\061\000\061\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\061\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\064\000\064\000\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\255\255\255\255\065\000\255\255\065\000\255\255\
    \064\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\064\000\064\000\064\000\064\000\064\000\064\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\255\255\255\255\066\000\
    \255\255\255\255\255\255\255\255\255\255\066\000\255\255\255\255\
    \255\255\069\000\255\255\255\255\255\255\255\255\255\255\069\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\255\255\255\255\255\255\255\255\255\255\066\000\
    \255\255\255\255\255\255\255\255\255\255\066\000\255\255\255\255\
    \255\255\069\000\255\255\255\255\255\255\255\255\255\255\069\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\255\255\255\255\071\000\255\255\
    \255\255\255\255\255\255\255\255\071\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\255\255\071\000\255\255\
    \255\255\074\000\255\255\074\000\071\000\073\000\074\000\074\000\
    \074\000\074\000\074\000\074\000\074\000\074\000\074\000\074\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\073\000\073\000\
    \073\000\073\000\073\000\073\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\073\000\075\000\075\000\
    \075\000\075\000\075\000\075\000\075\000\075\000\075\000\075\000\
    \076\000\076\000\076\000\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\255\255\255\255\075\000\255\255\255\255\255\255\
    \255\255\255\255\075\000\078\000\078\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\079\000\079\000\079\000\
    \079\000\079\000\079\000\079\000\079\000\079\000\079\000\255\255\
    \255\255\078\000\255\255\255\255\075\000\255\255\081\000\078\000\
    \081\000\255\255\075\000\081\000\081\000\081\000\081\000\081\000\
    \081\000\081\000\081\000\081\000\081\000\082\000\082\000\082\000\
    \082\000\082\000\082\000\082\000\082\000\082\000\082\000\255\255\
    \255\255\078\000\255\255\255\255\255\255\255\255\255\255\078\000\
    \255\255\255\255\082\000\082\000\255\255\083\000\255\255\083\000\
    \255\255\082\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\083\000\083\000\083\000\084\000\084\000\084\000\084\000\
    \084\000\084\000\084\000\084\000\084\000\084\000\255\255\255\255\
    \255\255\255\255\082\000\082\000\255\255\255\255\255\255\255\255\
    \255\255\082\000\084\000\255\255\255\255\255\255\255\255\255\255\
    \084\000\085\000\085\000\085\000\085\000\085\000\085\000\085\000\
    \085\000\085\000\085\000\086\000\086\000\086\000\086\000\086\000\
    \086\000\086\000\086\000\086\000\086\000\255\255\255\255\255\255\
    \255\255\255\255\084\000\255\255\255\255\255\255\255\255\255\255\
    \084\000\086\000\255\255\255\255\255\255\255\255\255\255\086\000\
    \087\000\087\000\087\000\087\000\087\000\087\000\087\000\087\000\
    \087\000\087\000\255\255\089\000\089\000\089\000\089\000\089\000\
    \089\000\089\000\089\000\089\000\089\000\255\255\255\255\255\255\
    \255\255\086\000\255\255\255\255\255\255\255\255\255\255\086\000\
    \089\000\089\000\255\255\090\000\255\255\090\000\255\255\089\000\
    \090\000\090\000\090\000\090\000\090\000\090\000\090\000\090\000\
    \090\000\090\000\091\000\091\000\091\000\091\000\091\000\091\000\
    \091\000\091\000\091\000\091\000\255\255\255\255\255\255\255\255\
    \089\000\089\000\255\255\255\255\255\255\255\255\255\255\089\000\
    \091\000\255\255\255\255\255\255\255\255\255\255\091\000\092\000\
    \092\000\092\000\092\000\092\000\092\000\092\000\092\000\092\000\
    \092\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\113\000\120\000\120\000\255\255\255\255\113\000\255\255\
    \091\000\255\255\255\255\255\255\255\255\255\255\091\000\113\000\
    \113\000\113\000\113\000\113\000\113\000\113\000\113\000\255\255\
    \255\255\120\000\255\255\255\255\255\255\255\255\113\000\123\000\
    \123\000\123\000\123\000\123\000\123\000\123\000\123\000\123\000\
    \123\000\120\000\120\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\255\255\255\255\255\255\255\255\113\000\
    \255\255\255\255\255\255\113\000\255\255\255\255\255\255\255\255\
    \113\000\113\000\255\255\255\255\255\255\113\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\113\000\255\255\255\255\
    \255\255\113\000\255\255\113\000\255\255\113\000\255\255\113\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\255\255\255\255\255\255\120\000\255\255\255\255\
    \255\255\116\000\116\000\116\000\116\000\116\000\116\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \117\000\117\000\117\000\117\000\117\000\117\000\255\255\255\255\
    \255\255\116\000\116\000\116\000\116\000\116\000\116\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \117\000\117\000\117\000\117\000\117\000\117\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\120\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    ";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \007\000\000\000\000\000\000\000\010\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\007\000\
    \000\000\000\000\000\000\010\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\001\000\000\000\000\000\000\000\004\000\
    \004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\034\000\255\255\255\255\255\255\000\000\
    \034\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\002\255\255\000\002\255\000\001\255";
}

let rec token lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 3 (-1) ;   __ocaml_lex_token_rec lexbuf 0
and __ocaml_lex_token_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 243 "cil/src/logic/logic_lexer.mll"
           ( token lexbuf )
# 865 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 244 "cil/src/logic/logic_lexer.mll"
         ( update_newline_loc lexbuf; token lexbuf )
# 870 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 245 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf )
# 875 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 246 "cil/src/logic/logic_lexer.mll"
                     ( token lexbuf )
# 880 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 248 "cil/src/logic/logic_lexer.mll"
                       ( bs_identifier lexbuf )
# 885 "cil/src/logic/logic_lexer.ml"

  | 5 ->
# 249 "cil/src/logic/logic_lexer.mll"
                        ( let s = lexeme lexbuf in identifier s )
# 890 "cil/src/logic/logic_lexer.ml"

  | 6 ->
# 251 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 895 "cil/src/logic/logic_lexer.ml"

  | 7 ->
# 252 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 900 "cil/src/logic/logic_lexer.ml"

  | 8 ->
# 253 "cil/src/logic/logic_lexer.mll"
                            ( CONSTANT (IntConstant (lexeme lexbuf)) )
# 905 "cil/src/logic/logic_lexer.ml"

  | 9 ->
let
# 254 "cil/src/logic/logic_lexer.mll"
                 prelude
# 911 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 254 "cil/src/logic/logic_lexer.mll"
                                                   content
# 916 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 255 "cil/src/logic/logic_lexer.mll"
      (
        let b = Buffer.create 5 in
        Buffer.add_string b prelude;
        let lbf = Lexing.from_string content in
        CONSTANT (IntConstant (chr b lbf ^ "'"))
      )
# 925 "cil/src/logic/logic_lexer.ml"

  | 10 ->
# 268 "cil/src/logic/logic_lexer.mll"
      ( CONSTANT (FloatConstant (lexeme lexbuf)) )
# 930 "cil/src/logic/logic_lexer.ml"

  | 11 ->
let
# 271 "cil/src/logic/logic_lexer.mll"
            n
# 936 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos (lexbuf.Lexing.lex_curr_pos + -2) in
# 271 "cil/src/logic/logic_lexer.mll"
                            ( lexbuf.lex_curr_pos <- lexbuf.lex_curr_pos - 2;
                              CONSTANT (IntConstant n) )
# 941 "cil/src/logic/logic_lexer.ml"

  | 12 ->
let
# 274 "cil/src/logic/logic_lexer.mll"
                prelude
# 947 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_mem.(0)
and
# 274 "cil/src/logic/logic_lexer.mll"
                                                 content
# 952 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) (lexbuf.Lexing.lex_curr_pos + -1) in
# 275 "cil/src/logic/logic_lexer.mll"
      ( STRING_LITERAL (prelude.[0] = 'L',content) )
# 956 "cil/src/logic/logic_lexer.ml"

  | 13 ->
# 276 "cil/src/logic/logic_lexer.mll"
                            ( hash lexbuf )
# 961 "cil/src/logic/logic_lexer.ml"

  | 14 ->
# 277 "cil/src/logic/logic_lexer.mll"
                            ( IMPLIES )
# 966 "cil/src/logic/logic_lexer.ml"

  | 15 ->
# 278 "cil/src/logic/logic_lexer.mll"
                            ( IFF )
# 971 "cil/src/logic/logic_lexer.ml"

  | 16 ->
# 279 "cil/src/logic/logic_lexer.mll"
                            ( AND )
# 976 "cil/src/logic/logic_lexer.ml"

  | 17 ->
# 280 "cil/src/logic/logic_lexer.mll"
                            ( OR )
# 981 "cil/src/logic/logic_lexer.ml"

  | 18 ->
# 281 "cil/src/logic/logic_lexer.mll"
                            ( NOT )
# 986 "cil/src/logic/logic_lexer.ml"

  | 19 ->
# 282 "cil/src/logic/logic_lexer.mll"
                            ( DOLLAR )
# 991 "cil/src/logic/logic_lexer.ml"

  | 20 ->
# 283 "cil/src/logic/logic_lexer.mll"
                            ( COMMA )
# 996 "cil/src/logic/logic_lexer.ml"

  | 21 ->
# 284 "cil/src/logic/logic_lexer.mll"
                            ( ARROW )
# 1001 "cil/src/logic/logic_lexer.ml"

  | 22 ->
# 285 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Test state_stack; QUESTION )
# 1006 "cil/src/logic/logic_lexer.ml"

  | 23 ->
# 286 "cil/src/logic/logic_lexer.mll"
                            ( SEMICOLON )
# 1011 "cil/src/logic/logic_lexer.ml"

  | 24 ->
# 287 "cil/src/logic/logic_lexer.mll"
                            ( match get_state() with
                                  Normal  -> COLON
                                | Test -> pop_state(); COLON2
                            )
# 1019 "cil/src/logic/logic_lexer.ml"

  | 25 ->
# 291 "cil/src/logic/logic_lexer.mll"
                            ( COLONCOLON )
# 1024 "cil/src/logic/logic_lexer.ml"

  | 26 ->
# 292 "cil/src/logic/logic_lexer.mll"
                            ( DOT )
# 1029 "cil/src/logic/logic_lexer.ml"

  | 27 ->
# 293 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOT )
# 1034 "cil/src/logic/logic_lexer.ml"

  | 28 ->
# 294 "cil/src/logic/logic_lexer.mll"
                            ( DOTDOTDOT )
# 1039 "cil/src/logic/logic_lexer.ml"

  | 29 ->
# 295 "cil/src/logic/logic_lexer.mll"
                            ( MINUS )
# 1044 "cil/src/logic/logic_lexer.ml"

  | 30 ->
# 296 "cil/src/logic/logic_lexer.mll"
                            ( PLUS )
# 1049 "cil/src/logic/logic_lexer.ml"

  | 31 ->
# 297 "cil/src/logic/logic_lexer.mll"
                            ( STAR )
# 1054 "cil/src/logic/logic_lexer.ml"

  | 32 ->
# 298 "cil/src/logic/logic_lexer.mll"
                            ( AMP )
# 1059 "cil/src/logic/logic_lexer.ml"

  | 33 ->
# 299 "cil/src/logic/logic_lexer.mll"
                            ( HATHAT )
# 1064 "cil/src/logic/logic_lexer.ml"

  | 34 ->
# 300 "cil/src/logic/logic_lexer.mll"
                            ( HAT )
# 1069 "cil/src/logic/logic_lexer.ml"

  | 35 ->
# 301 "cil/src/logic/logic_lexer.mll"
                            ( PIPE )
# 1074 "cil/src/logic/logic_lexer.ml"

  | 36 ->
# 302 "cil/src/logic/logic_lexer.mll"
                            ( TILDE )
# 1079 "cil/src/logic/logic_lexer.ml"

  | 37 ->
# 303 "cil/src/logic/logic_lexer.mll"
                            ( SLASH )
# 1084 "cil/src/logic/logic_lexer.ml"

  | 38 ->
# 304 "cil/src/logic/logic_lexer.mll"
                            ( PERCENT )
# 1089 "cil/src/logic/logic_lexer.ml"

  | 39 ->
# 305 "cil/src/logic/logic_lexer.mll"
                            ( LT )
# 1094 "cil/src/logic/logic_lexer.ml"

  | 40 ->
# 306 "cil/src/logic/logic_lexer.mll"
                            ( GT )
# 1099 "cil/src/logic/logic_lexer.ml"

  | 41 ->
# 307 "cil/src/logic/logic_lexer.mll"
                            ( LE )
# 1104 "cil/src/logic/logic_lexer.ml"

  | 42 ->
# 308 "cil/src/logic/logic_lexer.mll"
                            ( GE )
# 1109 "cil/src/logic/logic_lexer.ml"

  | 43 ->
# 309 "cil/src/logic/logic_lexer.mll"
                            ( EQ )
# 1114 "cil/src/logic/logic_lexer.ml"

  | 44 ->
# 310 "cil/src/logic/logic_lexer.mll"
                            ( EQUAL )
# 1119 "cil/src/logic/logic_lexer.ml"

  | 45 ->
# 311 "cil/src/logic/logic_lexer.mll"
                            ( NE )
# 1124 "cil/src/logic/logic_lexer.ml"

  | 46 ->
# 312 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LPAR )
# 1129 "cil/src/logic/logic_lexer.ml"

  | 47 ->
# 313 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RPAR )
# 1134 "cil/src/logic/logic_lexer.ml"

  | 48 ->
# 314 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LBRACE )
# 1139 "cil/src/logic/logic_lexer.ml"

  | 49 ->
# 315 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RBRACE )
# 1144 "cil/src/logic/logic_lexer.ml"

  | 50 ->
# 316 "cil/src/logic/logic_lexer.mll"
                            ( Stack.push Normal state_stack; LSQUARE )
# 1149 "cil/src/logic/logic_lexer.ml"

  | 51 ->
# 317 "cil/src/logic/logic_lexer.mll"
                            ( pop_state(); RSQUARE )
# 1154 "cil/src/logic/logic_lexer.ml"

  | 52 ->
# 318 "cil/src/logic/logic_lexer.mll"
                            ( LTCOLON )
# 1159 "cil/src/logic/logic_lexer.ml"

  | 53 ->
# 319 "cil/src/logic/logic_lexer.mll"
                            ( COLONGT )
# 1164 "cil/src/logic/logic_lexer.ml"

  | 54 ->
# 320 "cil/src/logic/logic_lexer.mll"
                            ( LTLT )
# 1169 "cil/src/logic/logic_lexer.ml"

  | 55 ->
# 321 "cil/src/logic/logic_lexer.mll"
                            ( GTGT )
# 1174 "cil/src/logic/logic_lexer.ml"

  | 56 ->
let
# 322 "cil/src/logic/logic_lexer.mll"
                 c
# 1180 "cil/src/logic/logic_lexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_start_pos lexbuf.Lexing.lex_curr_pos in
# 322 "cil/src/logic/logic_lexer.mll"
                            ( find_utf8 c )
# 1184 "cil/src/logic/logic_lexer.ml"

  | 57 ->
# 323 "cil/src/logic/logic_lexer.mll"
                            ( EOF )
# 1189 "cil/src/logic/logic_lexer.ml"

  | 58 ->
# 324 "cil/src/logic/logic_lexer.mll"
        ( lex_error lexbuf ("illegal character " ^ lexeme lexbuf) )
# 1194 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_token_rec lexbuf __ocaml_lex_state

and chr buffer lexbuf =
    __ocaml_lex_chr_rec buffer lexbuf 110
and __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 328 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 2 (String.length s - 2) in
        let rec add_one_char s =
          let l = String.length s in
          if l = 0 then ()
          else
          let h = int_of_digit s.[0] in
          let c,s =
            if l = 1 then (h,"")
            else
              (16*h + int_of_digit s.[1],
               String.sub s 2 (String.length s - 2))
          in
          Buffer.add_char buffer (Char.chr c); add_one_char s
        in add_one_char real_s; chr buffer lexbuf
      )
# 1220 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 345 "cil/src/logic/logic_lexer.mll"
      ( let s = lexeme lexbuf in
        let real_s = String.sub s 1 (String.length s - 1) in
        let rec value i s =
          if s = "" then i
          else value (8*i+int_of_digit s.[0])
            (String.sub s 1 (String.length s -1))
        in let c = value 0 real_s in
        Buffer.add_char buffer (Char.chr c); chr buffer lexbuf
      )
# 1233 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 355 "cil/src/logic/logic_lexer.mll"
      ( Buffer.add_char buffer
          (match (lexeme lexbuf).[1] with
               'a' -> '\007'
             | 'b' -> '\b'
             | 'f' -> '\012'
             | 'n' -> '\n'
             | 'r' -> '\r'
             | 't' -> '\t'
             | '\'' -> '\''
             | '"' -> '"'
             | '?' -> '?'
             | '\\' -> '\\'
             | _ -> assert false
          ); chr buffer lexbuf)
# 1251 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 369 "cil/src/logic/logic_lexer.mll"
        ( Buffer.contents buffer )
# 1256 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 370 "cil/src/logic/logic_lexer.mll"
       ( Buffer.add_string buffer (lexeme lexbuf); chr buffer lexbuf )
# 1261 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec buffer lexbuf __ocaml_lex_state

and hash lexbuf =
    __ocaml_lex_hash_rec lexbuf 120
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 373 "cil/src/logic/logic_lexer.mll"
        ( update_newline_loc lexbuf; token lexbuf)
# 1272 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 374 "cil/src/logic/logic_lexer.mll"
             ( hash lexbuf)
# 1277 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 375 "cil/src/logic/logic_lexer.mll"
              ( (* We are seeing a line number. This is the number for the
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
		   let src = Cil.source (lexbuf.lex_start_p, lexbuf.lex_curr_p) in
                   Cilmsg.warning ~source:src
                     "Bad line number in preprocessed file: %s"  s;
                   (-1)
                 in
                 update_line_loc lexbuf (lineno - 1) true 0;
                  (* A file name may follow *)
		  file lexbuf )
# 1296 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 390 "cil/src/logic/logic_lexer.mll"
                ( hash lexbuf )
# 1301 "cil/src/logic/logic_lexer.ml"

  | 4 ->
# 391 "cil/src/logic/logic_lexer.mll"
            ( endline lexbuf)
# 1306 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 129
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 394 "cil/src/logic/logic_lexer.mll"
                      ( update_newline_loc lexbuf; token lexbuf)
# 1317 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 395 "cil/src/logic/logic_lexer.mll"
                  ( file lexbuf)
# 1322 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 397 "cil/src/logic/logic_lexer.mll"
            (
              let n = Lexing.lexeme lexbuf in
              let n1 = String.sub n 1
                ((String.length n) - 2) in
              update_file_loc lexbuf n1;
	      endline lexbuf
            )
# 1333 "cil/src/logic/logic_lexer.ml"

  | 3 ->
# 405 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1338 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 136
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 408 "cil/src/logic/logic_lexer.mll"
                ( update_newline_loc lexbuf; token lexbuf)
# 1349 "cil/src/logic/logic_lexer.ml"

  | 1 ->
# 409 "cil/src/logic/logic_lexer.mll"
                                ( EOF )
# 1354 "cil/src/logic/logic_lexer.ml"

  | 2 ->
# 410 "cil/src/logic/logic_lexer.mll"
      ( endline lexbuf)
# 1359 "cil/src/logic/logic_lexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

;;

# 412 "cil/src/logic/logic_lexer.mll"
 

  open Format

  let copy_lexbuf dest_lexbuf src_loc =
    update_file_loc dest_lexbuf src_loc.pos_fname;
    update_line_loc dest_lexbuf src_loc.pos_lnum true 0;
    let bol = src_loc.Lexing.pos_cnum - src_loc.Lexing.pos_bol in
    update_bol_loc dest_lexbuf (-bol)

  let start_pos lexbuf =
    let pos = lexeme_start_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let end_pos lexbuf =
    let pos = lexeme_end_p lexbuf in
    pos.Lexing.pos_cnum - pos.Lexing.pos_bol

  let parse_from_location f (loc, s : Lexing.position * string) =
    let lb = from_string s in
    copy_lexbuf lb loc;
    try
      f token lb
    with
      | Parsing.Parse_error as _e ->
          Cil.error_loc (
	    lb.lex_curr_p.Lexing.pos_fname,
	    lb.lex_curr_p.Lexing.pos_lnum)
            "syntax error while parsing annotation@.";
          Logic_utils.exit_kw_c_mode ();
          raise Parsing.Parse_error

      | Error (_, m) ->
          Cil.error_loc (
	    lb.lex_curr_p.Lexing.pos_fname,
	    lb.lex_curr_p.Lexing.pos_lnum)
            "%s@." m;
          Logic_utils.exit_kw_c_mode ();
          raise Parsing.Parse_error
      | Logic_utils.Not_well_formed (loc, m) ->
          Cil.error_loc
            ((fst loc).Lexing.pos_fname,(fst loc).Lexing.pos_lnum)
            "%s@." m;
          Logic_utils.exit_kw_c_mode ();
          raise Parsing.Parse_error

  let lexpr = parse_from_location Logic_parser.lexpr_eof

  let annot = parse_from_location Logic_parser.annot

  let spec = parse_from_location Logic_parser.spec


# 1419 "cil/src/logic/logic_lexer.ml"
