# 47 "cil/src/frontc/clexer.mll"
 
open Cparser
exception Eof
exception InternalError of string
module H = Hashtbl
module E = Errorloc

let matchingParsOpen = ref 0

let currentLoc () = Cabshelper.currentLoc ()

let one_line_ghost = ref false
let is_oneline_ghost () = !one_line_ghost
let enter_oneline_ghost () = one_line_ghost := true
let exit_oneline_ghost () = one_line_ghost := false

let ghost_code = ref false
let is_ghost_code () = !ghost_code
let enter_ghost_code () = ghost_code := true
let exit_ghost_code () = ghost_code := false

let keepComments = ref false

(* string -> unit *)
let addComment c =
  let l = currentLoc() in
  let i = GrowArray.max_init_index Cabshelper.commentsGA in
  GrowArray.setg Cabshelper.commentsGA (i+1) (l,c,false)

(* track whitespace for the current token *)
let white = ref ""

let addWhite lexbuf =
  if !Whitetrack.enabled then
    let w = Lexing.lexeme lexbuf in
    white := !white ^ w
let clear_white () = white := ""
let get_white () = !white

let lexeme = ref ""
let addLexeme lexbuf =
    let l = Lexing.lexeme lexbuf in
    lexeme := !lexeme ^ l
let clear_lexeme () = lexeme := ""
let get_extra_lexeme () = !lexeme

let int64_to_char value =
  if (compare value (Int64.of_int 255) > 0) || (compare value Int64.zero < 0) then
    begin
      let msg = Printf.sprintf "clexer:intlist_to_string: character 0x%Lx too big" value in
      E.parse_error msg;
    end
  else
    Char.chr (Int64.to_int value)

(* takes a not-nul-terminated list, and converts it to a string. *)
let rec intlist_to_string (str: int64 list):string =
  match str with
    [] -> ""  (* add nul-termination *)
  | value::rest ->
      let this_char = int64_to_char value in
      (String.make 1 this_char) ^ (intlist_to_string rest)

(* Some debugging support for line numbers *)
let dbgToken (t: token) =
  if false then begin
    let dprintf fmt = Cilmsg.debug fmt in
    (match t with
         IDENT (n, l) -> dprintf "IDENT(%s,%d)\n" n (fst l).Lexing.pos_lnum
       | LBRACE l -> dprintf "LBRACE(%d)\n" (fst l).Lexing.pos_lnum
       | RBRACE l -> dprintf "RBRACE(%d)\n" (fst l).Lexing.pos_lnum
       | IF l -> dprintf "IF(%d)\n" (fst l).Lexing.pos_lnum
       | SWITCH l -> dprintf "SWITCH(%d)\n" (fst l).Lexing.pos_lnum
       | RETURN l -> dprintf "RETURN(%d)\n" (fst l).Lexing.pos_lnum
       | _ -> ()) ;
    t
  end else
    t

(*
** Keyword hashtable
*)
let lexicon = H.create 211
let init_lexicon _ =
  H.clear lexicon;
  Logic_env.reset_typenames ();
  Logic_env.builtin_types_as_typenames ();
  List.iter
    (fun (key, builder) -> H.add lexicon key builder)
    [ ("auto", fun loc -> AUTO loc);
      ("const", fun loc -> CONST loc);
      ("__const", fun loc -> CONST loc);
      ("__const__", fun loc -> CONST loc);
      ("static", fun loc -> STATIC loc);
      ("extern", fun loc -> EXTERN loc);
      ("long", fun loc -> LONG loc);
      ("short", fun loc -> SHORT loc);
      ("register", fun loc -> REGISTER loc);
      ("signed", fun loc -> SIGNED loc);
      ("__signed", fun loc -> SIGNED loc);
      ("unsigned", fun loc -> UNSIGNED loc);
      ("volatile", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);
      (* WW: see /usr/include/sys/cdefs.h for why __signed and __volatile
       * are accepted GCC-isms *)
      ("char", fun loc -> CHAR loc);
      ("_Bool", fun loc -> BOOL loc);
      ("int", fun loc -> INT loc);
      ("float", fun loc -> FLOAT loc);
      ("double", fun loc -> DOUBLE loc);
      ("void", fun loc -> VOID loc);
      ("enum", fun loc -> ENUM loc);
      ("struct", fun loc -> STRUCT loc);
      ("typedef", fun loc -> TYPEDEF loc);
      ("union", fun loc -> UNION loc);
      ("break", fun loc -> BREAK loc);
      ("continue", fun loc -> CONTINUE loc);
      ("goto", fun loc -> GOTO loc);
      ("return", fun loc -> dbgToken (RETURN loc));
      ("switch", fun loc -> dbgToken (SWITCH loc));
      ("case", fun loc -> CASE loc);
      ("default", fun loc -> DEFAULT loc);
      ("while", fun loc -> WHILE loc);
      ("do", fun loc -> DO loc);
      ("for", fun loc -> FOR loc);
      ("if", fun loc -> dbgToken (IF loc));
      ("else", fun _ -> ELSE);
      (*** Implementation specific keywords ***)
      ("__signed__", fun loc -> SIGNED loc);
      ("__inline__", fun loc -> INLINE loc);
      ("inline", fun loc -> INLINE loc);
      ("__inline", fun loc -> INLINE loc);
      ("_inline", fun loc ->
                      if !Cprint.msvcMode then
                        INLINE loc
                      else
                        IDENT ("_inline", loc));
      ("__attribute__", fun loc -> ATTRIBUTE loc);
      ("__attribute", fun loc -> ATTRIBUTE loc);
(*
      ("__attribute_used__", fun loc -> ATTRIBUTE_USED loc);
*)
      ("__blockattribute__", fun _ -> BLOCKATTRIBUTE);
      ("__blockattribute", fun _ -> BLOCKATTRIBUTE);
      ("__asm__", fun loc -> ASM loc);
      ("asm", fun loc -> ASM loc);
      ("__typeof__", fun loc -> TYPEOF loc);
      ("__typeof", fun loc -> TYPEOF loc);
      ("typeof", fun loc -> TYPEOF loc);
      ("__alignof", fun loc -> ALIGNOF loc);
      ("__alignof__", fun loc -> ALIGNOF loc);
      ("__volatile__", fun loc -> VOLATILE loc);
      ("__volatile", fun loc -> VOLATILE loc);

      ("__FUNCTION__", fun loc -> FUNCTION__ loc);
      ("__func__", fun loc -> FUNCTION__ loc); (* ISO 6.4.2.2 *)
      ("__PRETTY_FUNCTION__", fun loc -> PRETTY_FUNCTION__ loc);
      ("__label__", fun _ -> LABEL__);
      (*** weimer: GCC arcana ***)
      ("__restrict", fun loc -> RESTRICT loc);
      ("restrict", fun loc -> RESTRICT loc);
(*      ("__extension__", EXTENSION); *)
      (**** MS VC ***)
      ("__int64", fun _ -> INT64 (currentLoc ()));
      ("__int32", fun loc -> INT loc);
      ("_cdecl",  fun _ -> MSATTR ("_cdecl", currentLoc ()));
      ("__cdecl", fun _ -> MSATTR ("__cdecl", currentLoc ()));
      ("_stdcall", fun _ -> MSATTR ("_stdcall", currentLoc ()));
      ("__stdcall", fun _ -> MSATTR ("__stdcall", currentLoc ()));
      ("_fastcall", fun _ -> MSATTR ("_fastcall", currentLoc ()));
      ("__fastcall", fun _ -> MSATTR ("__fastcall", currentLoc ()));
      ("__w64", fun _ -> MSATTR("__w64", currentLoc ()));
      ("__declspec", fun loc -> DECLSPEC loc);
      ("__forceinline", fun loc -> INLINE loc); (* !! we turn forceinline
                                                 * into inline *)
      ("__try", fun loc -> TRY loc);
      ("__except", fun loc -> EXCEPT loc);
      ("__finally", fun loc -> FINALLY loc);
      (* weimer: some files produced by 'GCC -E' expect this type to be
       * defined *)
      ("__builtin_va_list",
       fun _ -> NAMED_TYPE ("__builtin_va_list", currentLoc ()));
      ("__builtin_va_arg", fun loc -> BUILTIN_VA_ARG loc);
      ("__builtin_types_compatible_p", fun loc -> BUILTIN_TYPES_COMPAT loc);
      ("__builtin_offsetof", fun loc -> BUILTIN_OFFSETOF loc);
      (* On some versions of GCC __thread is a regular identifier *)
      ("__thread", fun loc ->
                      if Machdep.state.Machdep.__thread_is_keyword then
                         THREAD loc
                       else
                         IDENT ("__thread", loc));
    ]

(* Mark an identifier as a type name. The old mapping is preserved and will
 * be reinstated when we exit this context *)
let add_type name =
   (* ignore (print_string ("adding type name " ^ name ^ "\n"));  *)
  H.add lexicon name (fun loc -> NAMED_TYPE (name, loc));
  Logic_env.add_typename name

let context : string list list ref = ref [ [] ]

let push_context _ = context := []::!context

let pop_context _ =
  match !context with
    [] -> Cilmsg.fatal "Empty context stack"
  | con::sub ->
		(context := sub;
		List.iter (fun name ->
                            (* Format.eprintf
                              "removing lexicon for %s@." name; *)
                             H.remove lexicon name;
                             Logic_env.remove_typename name
                          ) con)

(* Mark an identifier as a variable name. The old mapping is preserved and
 * will be reinstated when we exit this context  *)
let add_identifier name =
  match !context with
    [] -> Cilmsg.fatal "Empty context stack"
  | con::sub ->
      (context := (name::con)::sub;
       (*Format.eprintf "adding IDENT for %s@." name;*)
       H.add lexicon name (fun loc ->
         dbgToken (IDENT (name, loc)));
       Logic_env.hide_typename name
      )


(*
** Useful primitives
*)
let scan_ident id =
  let here = currentLoc () in
  try (H.find lexicon id) here
  (* default to variable name, as opposed to type *)
  with Not_found -> dbgToken (IDENT (id, here))


(*
** Buffer processor
*)


let init ~(filename: string) : Lexing.lexbuf =
  init_lexicon ();
  (* Inititialize the pointer in Errormsg *)
  Lexerhack.add_type := add_type;
  Lexerhack.push_context := push_context;
  Lexerhack.pop_context := pop_context;
  Lexerhack.add_identifier := add_identifier;
  E.startParsing ~useBasename:false filename


let finish () =
  E.finishParsing ()

(*** Error handling ***)
let error msg =
  E.parse_error msg


(*** escape character management ***)
let scan_escape (char: char) : int64 =
  let result = match char with
    'n' -> '\n'
  | 'r' -> '\r'
  | 't' -> '\t'
  | 'b' -> '\b'
  | 'f' -> '\012'  (* ASCII code 12 *)
  | 'v' -> '\011'  (* ASCII code 11 *)
  | 'a' -> '\007'  (* ASCII code 7 *)
  | 'e' | 'E' -> '\027'  (* ASCII code 27. This is a GCC extension *)
  | '\'' -> '\''
  | '"'-> '"'     (* '"' *)
  | '?' -> '?'
  | '(' when not !Cprint.msvcMode -> '('
  | '{' when not !Cprint.msvcMode -> '{'
  | '[' when not !Cprint.msvcMode -> '['
  | '%' when not !Cprint.msvcMode -> '%'
  | '\\' -> '\\'
  | other -> error ("Unrecognized escape sequence: \\" ^ (String.make 1 other))
  in
  Int64.of_int (Char.code result)

let scan_hex_escape str =
  let radix = Int64.of_int 16 in
  let the_value = ref Int64.zero in
  (* start at character 2 to skip the \x *)
  for i = 2 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 16 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let scan_oct_escape str =
  let radix = Int64.of_int 8 in
  let the_value = ref Int64.zero in
  (* start at character 1 to skip the \x *)
  for i = 1 to (String.length str) - 1 do
    let thisDigit = Cabshelper.valueOfDigit (String.get str i) in
    (* the_value := !the_value * 8 + thisDigit *)
    the_value := Int64.add (Int64.mul !the_value radix) thisDigit
  done;
  !the_value

let lex_hex_escape remainder lexbuf =
  let prefix = scan_hex_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_oct_escape remainder lexbuf =
  let prefix = scan_oct_escape (Lexing.lexeme lexbuf) in
  prefix :: remainder lexbuf

let lex_simple_escape remainder lexbuf =
  let lexchar = Lexing.lexeme_char lexbuf 1 in
  let prefix = scan_escape lexchar in
  prefix :: remainder lexbuf

let lex_unescaped remainder lexbuf =
  let prefix = Int64.of_int (Char.code (Lexing.lexeme_char lexbuf 0)) in
  prefix :: remainder lexbuf

let lex_comment remainder buffer lexbuf =
  let ch = Lexing.lexeme_char lexbuf 0 in
  if ch = '\n' then E.newline() ;
  (match buffer with None -> () | Some b -> Buffer.add_char b ch) ;
  remainder buffer lexbuf

let do_lex_comment remainder lexbuf =
  let buffer = 
    if !keepComments then Some(Buffer.create 80) else None
  in remainder buffer lexbuf ;
  match buffer with
    | Some b -> addComment (Buffer.contents b)
    | None -> ()

let make_char (i:int64):char =
  let min_val = Int64.zero in
  let max_val = Int64.of_int 255 in
  (* if i < 0 || i > 255 then error*)
  if compare i min_val < 0 || compare i max_val > 0 then begin
    let msg = Printf.sprintf "clexer:make_char: character 0x%Lx too big" i in
    error msg
  end;
  Char.chr (Int64.to_int i)


(* ISO standard locale-specific function to convert a wide character
 * into a sequence of normal characters. Here we work on strings.
 * We convert L"Hi" to "H\000i\000"
  matth: this seems unused.
let wbtowc wstr =
  let len = String.length wstr in
  let dest = String.make (len * 2) '\000' in
  for i = 0 to len-1 do
    dest.[i*2] <- wstr.[i] ;
  done ;
  dest
*)

(* This function converst the "Hi" in L"Hi" to { L'H', L'i', L'\0' }
  matth: this seems unused.
let wstr_to_warray wstr =
  let len = String.length wstr in
  let res = ref "{ " in
  for i = 0 to len-1 do
    res := !res ^ (Printf.sprintf "L'%c', " wstr.[i])
  done ;
  res := !res ^ "}" ;
  !res
*)

(* Pragmas get explicit end-of-line tokens.
 * Elsewhere they are silently discarded as whitespace. *)
let pragmaLine = ref false

let annot_char = ref '@'
let annot_start_pos = ref Cabshelper.cabslu
let buf = Buffer.create 1024

let save_current_pos () =
  annot_start_pos := currentLoc ()

let make_annot s =
  let start = snd !annot_start_pos in
  match Logic_lexer.annot (start, s) with
    | Logic_ptree.Adecl d -> DECL d
    | Logic_ptree.Aspec -> SPEC (start,s)
        (* At this point, we only have identified a function spec. Complete
           parsing of the annotation will only occur in the cparser.mly rule.
         *)
    | Logic_ptree.Acode_annot (loc,a) -> CODE_ANNOT (a, loc)
    | Logic_ptree.Aloop_annot (loc,a) -> LOOP_ANNOT (a,loc)
    | Logic_ptree.Aattribute_annot (loc,a) -> ATTRIBUTE_ANNOT (a, loc)

# 401 "cil/src/frontc/clexer.ml"
let __ocaml_lex_tables = {
  Lexing.lex_base = 
   "\000\000\180\255\181\255\091\000\113\000\194\000\193\255\194\255\
    \195\255\196\255\197\255\198\255\199\255\200\255\201\255\202\255\
    \203\255\035\000\067\000\070\000\071\000\177\000\092\000\072\000\
    \073\000\090\000\093\000\121\000\015\001\027\001\065\001\244\255\
    \138\001\246\255\225\001\248\255\001\000\250\255\209\000\178\000\
    \233\255\127\000\145\000\254\255\252\255\249\255\005\000\056\002\
    \143\002\230\002\061\003\148\003\235\003\066\004\153\004\240\004\
    \071\005\158\005\245\005\076\006\163\006\250\006\081\007\168\007\
    \255\007\086\008\173\008\004\009\091\009\178\009\009\010\096\010\
    \183\010\014\011\243\255\245\255\091\011\106\001\125\011\084\000\
    \028\001\137\011\117\011\191\011\095\000\154\011\189\011\183\011\
    \185\000\079\001\100\001\039\001\098\000\186\000\203\011\152\011\
    \103\011\131\000\242\255\006\012\017\012\046\012\056\012\038\001\
    \085\012\214\001\095\012\045\002\156\012\194\012\130\000\034\001\
    \141\000\067\001\124\012\008\013\232\012\041\013\132\002\074\013\
    \219\002\241\255\168\000\109\000\111\000\172\000\181\000\237\255\
    \223\255\116\000\131\000\121\000\003\000\238\255\217\255\236\255\
    \215\255\216\255\235\255\234\255\232\255\207\255\231\255\208\255\
    \230\255\229\255\222\255\179\000\228\255\221\255\191\000\227\255\
    \224\255\118\013\205\013\036\014\123\014\210\014\160\000\168\000\
    \178\000\204\000\001\001\245\000\241\000\255\000\247\000\247\000\
    \253\000\038\001\245\000\011\001\010\001\187\255\025\001\031\001\
    \026\001\035\001\033\001\042\001\051\001\186\255\055\001\054\001\
    \185\255\062\001\077\001\184\255\132\001\255\255\190\001\254\255\
    \134\001\255\255\021\002\254\255\255\255\068\015\249\255\250\255\
    \141\001\252\255\253\255\093\003\255\255\251\255\073\015\249\255\
    \083\001\093\001\083\015\180\003\255\255\145\001\126\001\079\011\
    \099\011\253\255\199\001\124\001\121\001\178\001\162\001\182\001\
    \252\255\188\001\183\001\179\001\192\001\011\004\132\015\206\001\
    \210\001\252\001\220\001\244\001\255\001\003\002\015\002\017\002\
    \051\002\051\002\022\002\006\002\037\002\044\002\040\002\098\002\
    \251\255\095\002\136\002\103\002\095\002\090\002\108\002\118\002\
    \130\002\125\002\127\002\123\002\183\002\178\002\180\002\182\002\
    \191\002\198\002\179\002\220\002\223\002\207\002\219\002\215\002\
    \244\002\255\002\255\002\255\002\254\002\250\002\006\003\013\003\
    \019\003\001\003\006\003\013\003\027\003\043\003\052\003\071\003\
    \060\003\060\003\077\003\081\003\084\003\078\003\079\003\093\003\
    \097\003\087\003\112\003\094\003\108\003\107\003\106\003\124\003\
    \124\003\140\003\128\003\145\003\175\003\157\003\169\003\164\003\
    \167\003\013\012\252\255\212\010\098\004\255\255\253\255\042\011\
    \028\004\253\255\254\255\255\255\029\004\254\255\255\255\153\010\
    \250\255\251\255\180\012\255\255\252\255\142\004\156\015\211\015\
    \060\005\253\255\150\010\250\255\251\255\098\015\255\255\252\255\
    \147\005\234\015\034\016\234\005\253\255\046\011\253\255\254\255\
    \108\006\157\003\254\255\255\255\214\011\253\255\183\003\255\255\
    \184\003\181\003\188\003\254\255\095\004\253\255\254\255\194\003\
    \215\003\212\003\212\003\255\255\116\004\252\255\253\255\254\255\
    \026\004\255\255\014\005\254\255\227\003\248\003\247\003\249\003\
    \255\255\118\004\254\255\255\255";
  Lexing.lex_backtrk = 
   "\255\255\255\255\255\255\073\000\075\000\073\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\035\000\037\000\036\000\051\000\049\000\050\000\045\000\
    \043\000\042\000\041\000\046\000\063\000\016\000\016\000\255\255\
    \073\000\255\255\073\000\255\255\075\000\255\255\004\000\044\000\
    \255\255\002\000\000\000\255\255\255\255\255\255\255\255\073\000\
    \073\000\073\000\073\000\073\000\073\000\008\000\073\000\073\000\
    \073\000\073\000\065\000\073\000\073\000\073\000\073\000\066\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\072\000\255\255\255\255\255\255\255\255\013\000\255\255\
    \016\000\016\000\016\000\015\000\255\255\015\000\015\000\015\000\
    \015\000\015\000\015\000\015\000\255\255\016\000\016\000\016\000\
    \016\000\255\255\255\255\255\255\013\000\255\255\013\000\255\255\
    \013\000\255\255\013\000\255\255\255\255\014\000\255\255\014\000\
    \014\000\014\000\255\255\255\255\255\255\013\000\255\255\013\000\
    \255\255\255\255\014\000\014\000\014\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\029\000\255\255\255\255\030\000\255\255\
    \255\255\073\000\073\000\073\000\073\000\064\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\067\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\001\000\255\255\255\255\255\255\
    \001\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \006\000\255\255\255\255\001\000\255\255\255\255\255\255\255\255\
    \006\000\006\000\002\000\001\000\255\255\255\255\002\000\002\000\
    \002\000\255\255\002\000\002\000\002\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\005\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\003\000\001\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\005\000\255\255\255\255\002\000\003\000\001\000\
    \002\000\255\255\255\255\255\255\255\255\005\000\255\255\255\255\
    \002\000\003\000\001\000\002\000\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\002\000\255\255\
    \255\255\255\255\255\255\255\255\003\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \003\000\255\255\002\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255";
  Lexing.lex_default = 
   "\001\000\000\000\000\000\255\255\255\255\255\255\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\000\000\
    \255\255\000\000\255\255\000\000\255\255\000\000\255\255\255\255\
    \000\000\044\000\043\000\000\000\000\000\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\000\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\255\255\000\000\000\000\255\255\000\000\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\000\000\255\255\255\255\
    \000\000\255\255\255\255\000\000\255\255\000\000\191\000\000\000\
    \255\255\000\000\195\000\000\000\000\000\198\000\000\000\000\000\
    \255\255\000\000\000\000\255\255\000\000\000\000\207\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\255\255\255\255\255\255\
    \255\255\000\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \000\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\058\001\000\000\063\001\255\255\000\000\000\000\063\001\
    \065\001\000\000\000\000\000\000\069\001\000\000\000\000\072\001\
    \000\000\000\000\076\001\000\000\000\000\255\255\255\255\255\255\
    \255\255\000\000\083\001\000\000\000\000\087\001\000\000\000\000\
    \255\255\255\255\255\255\255\255\000\000\094\001\000\000\000\000\
    \255\255\098\001\000\000\000\000\101\001\000\000\255\255\000\000\
    \255\255\255\255\255\255\000\000\255\255\000\000\000\000\255\255\
    \255\255\255\255\255\255\000\000\117\001\000\000\000\000\000\000\
    \255\255\000\000\255\255\000\000\255\255\255\255\255\255\255\255\
    \000\000\130\001\000\000\000\000";
  Lexing.lex_trans = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\038\000\037\000\045\000\038\000\038\000\046\000\045\000\
    \000\000\000\000\046\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\038\000\000\000\000\000\000\000\000\000\000\000\
    \038\000\027\000\031\000\035\000\133\000\023\000\021\000\033\000\
    \009\000\008\000\024\000\026\000\006\000\025\000\028\000\039\000\
    \030\000\029\000\029\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\015\000\007\000\019\000\017\000\018\000\016\000\
    \004\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\032\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\011\000\036\000\010\000\020\000\034\000\
    \152\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\005\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\013\000\022\000\012\000\014\000\003\000\
    \149\000\150\000\147\000\146\000\145\000\140\000\139\000\137\000\
    \134\000\255\255\097\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\092\000\088\000\138\000\
    \136\000\142\000\135\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\128\000\093\000\
    \125\000\121\000\003\000\255\255\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\159\000\143\000\
    \141\000\122\000\038\000\121\000\042\000\038\000\038\000\158\000\
    \121\000\041\000\121\000\127\000\160\000\161\000\003\000\078\000\
    \078\000\130\000\129\000\038\000\131\000\132\000\144\000\040\000\
    \148\000\038\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\151\000\121\000\077\000\077\000\
    \002\000\185\000\121\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\121\000\077\000\077\000\
    \182\000\003\000\174\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\153\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\126\000\162\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\078\000\078\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\078\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \077\000\077\000\163\000\164\000\165\000\166\000\167\000\081\000\
    \095\000\168\000\169\000\170\000\077\000\171\000\123\000\078\000\
    \080\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\029\000\029\000\172\000\173\000\078\000\175\000\255\255\
    \077\000\077\000\176\000\177\000\079\000\079\000\077\000\082\000\
    \096\000\178\000\179\000\110\000\077\000\081\000\124\000\180\000\
    \080\000\255\255\078\000\088\000\077\000\107\000\080\000\107\000\
    \121\000\076\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\088\000\181\000\077\000\183\000\
    \184\000\077\000\079\000\186\000\074\000\082\000\003\000\122\000\
    \088\000\075\000\187\000\189\000\077\000\193\000\080\000\205\000\
    \121\000\076\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\088\000\225\000\222\000\221\000\
    \217\000\077\000\219\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\217\000\217\000\213\000\
    \192\000\003\000\220\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \223\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\224\000\217\000\226\000\227\000\196\000\
    \228\000\229\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\048\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\217\000\051\001\038\001\031\001\
    \047\000\017\001\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\106\000\106\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\008\001\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\007\001\003\001\254\000\251\000\250\000\249\000\
    \244\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\245\000\243\000\246\000\247\000\003\000\
    \242\000\056\000\003\000\003\000\003\000\054\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \055\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\255\255\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\248\000\248\000\248\000\252\000\253\000\248\000\255\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\001\001\001\002\001\248\000\003\000\004\001\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\049\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\196\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \005\001\006\001\248\000\248\000\009\001\010\001\011\001\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\012\001\013\001\014\001\015\001\003\000\016\001\050\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\248\000\018\001\019\001\020\001\203\000\021\001\
    \022\001\203\000\203\000\023\001\024\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\203\000\
    \025\001\026\001\027\001\028\001\029\001\203\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \030\001\248\000\032\001\033\001\003\000\034\001\003\000\003\000\
    \003\000\003\000\003\000\003\000\051\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\035\001\036\001\037\001\248\000\211\000\039\001\040\001\
    \211\000\211\000\041\001\042\001\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\211\000\043\001\
    \044\001\045\001\046\001\047\001\211\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\048\001\
    \049\001\050\001\248\000\003\000\052\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\052\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \053\001\054\001\055\001\056\001\230\000\248\000\104\001\230\000\
    \230\000\105\001\099\001\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\230\000\067\001\070\001\
    \106\001\107\001\112\001\230\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\113\001\114\001\
    \115\001\121\001\003\000\125\001\053\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\126\001\
    \110\001\109\001\127\001\060\001\110\001\128\001\060\001\060\001\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\060\001\000\000\118\001\110\001\
    \131\001\000\000\060\001\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\255\255\120\001\110\001\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\080\001\080\001\
    \080\001\080\001\080\001\080\001\080\001\080\001\111\001\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\064\000\003\000\003\000\003\000\000\000\000\000\123\001\
    \000\000\000\000\000\000\123\001\066\001\255\255\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\123\001\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\123\001\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\059\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\081\001\081\001\081\001\081\001\
    \081\001\081\001\081\001\081\001\119\001\124\001\255\255\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\057\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\091\001\091\001\091\001\091\001\091\001\
    \091\001\091\001\091\001\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\058\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\092\001\092\001\092\001\092\001\092\001\092\001\
    \092\001\092\001\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\096\001\000\000\000\000\
    \096\001\096\001\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\096\001\000\000\
    \000\000\000\000\000\000\000\000\096\001\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\060\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\061\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\062\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\063\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\003\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\003\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\065\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\066\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\067\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\068\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\069\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\070\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \003\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\071\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\075\001\000\000\086\001\000\000\072\000\
    \000\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\255\255\000\000\000\000\
    \255\255\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\085\001\000\000\000\000\074\001\062\001\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\000\000\000\000\000\000\000\000\073\000\000\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\255\255\000\000\000\000\255\255\096\001\
    \000\000\000\000\096\001\096\001\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \096\001\000\000\000\000\000\000\062\001\000\000\096\001\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\108\000\000\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\109\000\078\000\084\001\000\000\
    \000\000\073\001\000\000\218\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\000\000\078\000\217\000\000\000\000\000\000\000\
    \000\000\095\001\000\000\000\000\077\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\100\000\100\000\100\000\100\000\078\000\
    \217\000\000\000\077\000\000\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\099\000\098\000\217\000\000\000\078\000\000\000\
    \078\000\098\000\093\000\000\000\077\000\000\000\077\000\218\000\
    \000\000\000\000\000\000\093\000\255\255\094\000\000\000\000\000\
    \217\000\000\000\077\000\000\000\000\000\077\000\093\000\077\000\
    \103\001\094\000\099\000\098\000\093\000\078\000\090\000\000\000\
    \000\000\098\000\093\000\078\000\000\000\078\000\077\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\029\000\
    \029\000\078\000\000\000\000\000\077\000\077\000\093\000\077\000\
    \000\000\000\000\077\000\084\000\077\000\000\000\091\000\000\000\
    \000\000\089\000\000\000\086\000\088\000\000\000\000\000\000\000\
    \077\000\103\001\088\000\000\000\085\000\000\000\060\001\061\001\
    \000\000\060\001\060\001\000\000\077\000\000\000\000\000\000\000\
    \093\000\000\000\077\000\089\000\077\000\000\000\000\000\060\001\
    \084\000\000\000\255\255\087\000\088\000\060\001\255\255\059\001\
    \077\000\105\000\088\000\105\000\085\000\102\001\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\104\000\
    \093\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\103\001\000\000\000\000\101\000\098\000\
    \000\000\103\000\000\000\103\000\000\000\098\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\000\000\000\000\000\000\000\000\101\000\098\000\
    \000\000\000\000\000\000\000\000\000\000\098\000\098\000\000\000\
    \000\000\000\000\000\000\000\000\098\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\000\000\000\000\098\000\000\000\000\000\098\000\000\000\
    \000\000\098\000\000\000\000\000\098\000\098\000\000\000\120\000\
    \000\000\120\000\000\000\098\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\119\000\119\000\000\000\000\000\
    \000\000\000\000\000\000\098\000\000\000\000\000\000\000\000\000\
    \000\000\098\000\000\000\000\000\000\000\098\000\000\000\000\000\
    \000\000\000\000\000\000\098\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\255\255\000\000\
    \000\000\000\000\000\000\000\000\000\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\000\000\077\001\077\001\077\001\077\001\
    \077\001\077\001\077\001\077\001\000\000\000\000\000\000\000\000\
    \115\000\000\000\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\000\000\115\000\115\000\115\000\
    \115\000\115\000\115\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\000\000\078\001\255\255\112\000\000\000\
    \000\000\000\000\114\000\118\000\000\000\118\000\000\000\111\000\
    \117\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\000\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\000\000\000\000\110\000\078\001\000\000\113\000\000\000\
    \000\000\000\000\114\000\000\000\000\000\000\000\000\000\111\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\115\000\115\000\115\000\115\000\115\000\115\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \116\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\115\000\115\000\115\000\115\000\115\000\115\000\098\000\
    \000\000\000\000\000\000\000\000\000\000\098\000\000\000\000\000\
    \116\000\000\000\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\098\000\
    \098\000\000\000\000\000\000\000\000\000\098\000\098\000\000\000\
    \000\000\000\000\003\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \098\000\000\000\000\000\000\000\255\255\000\000\098\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\003\000\000\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \154\000\003\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\003\000\000\000\003\000\003\000\
    \003\000\003\000\155\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\003\000\000\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\156\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\000\000\000\000\
    \000\000\000\000\003\000\000\000\003\000\003\000\003\000\003\000\
    \003\000\157\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\000\000\000\000\000\000\
    \000\000\003\000\000\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\203\000\204\000\000\000\
    \203\000\203\000\211\000\212\000\000\000\211\000\211\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\203\000\000\000\
    \000\000\000\000\000\000\211\000\203\000\000\000\199\000\000\000\
    \000\000\211\000\000\000\000\000\202\000\201\000\000\000\000\000\
    \000\000\000\000\000\000\200\000\000\000\000\000\000\000\000\000\
    \000\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\230\000\000\000\000\000\
    \230\000\230\000\088\001\088\001\088\001\088\001\088\001\088\001\
    \088\001\088\001\000\000\000\000\000\000\000\000\230\000\215\000\
    \000\000\000\000\000\000\000\000\230\000\000\000\000\000\000\000\
    \214\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\209\000\000\000\000\000\
    \000\000\208\000\089\001\000\000\213\000\000\000\000\000\216\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \214\000\000\000\000\000\240\000\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\000\000\000\000\
    \233\000\000\000\089\001\000\000\000\000\079\001\079\001\079\001\
    \079\001\079\001\079\001\000\000\000\000\236\000\000\000\000\000\
    \000\000\000\000\000\000\232\000\000\000\239\000\000\000\000\000\
    \231\000\000\000\000\000\237\000\000\000\000\000\234\000\238\000\
    \000\000\235\000\000\000\241\000\000\000\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\079\001\079\001\079\001\079\001\
    \079\001\079\001\090\001\090\001\090\001\090\001\090\001\090\001\
    \090\001\090\001\090\001\090\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\090\001\090\001\090\001\090\001\090\001\
    \090\001\000\000\000\000\000\000\079\001\079\001\079\001\079\001\
    \079\001\079\001\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\255\255\000\000\000\000\000\000\
    \000\000\255\255\000\000\090\001\090\001\090\001\090\001\090\001\
    \090\001\000\000\090\001\090\001\090\001\090\001\090\001\090\001\
    \090\001\090\001\090\001\090\001\000\000\000\000\000\000\000\000\
    \000\000\000\000\255\255\090\001\090\001\090\001\090\001\090\001\
    \090\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\090\001\090\001\090\001\090\001\090\001\
    \090\001\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\000\000\000\000\036\000\000\000\000\000\036\000\046\000\
    \255\255\255\255\046\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\000\000\255\255\255\255\255\255\255\255\255\255\
    \000\000\000\000\000\000\000\000\132\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \017\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\003\000\
    \018\000\018\000\019\000\019\000\020\000\023\000\024\000\025\000\
    \026\000\041\000\079\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\084\000\092\000\025\000\
    \025\000\022\000\026\000\042\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\027\000\097\000\
    \110\000\123\000\003\000\042\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\003\000\003\000\
    \003\000\003\000\003\000\003\000\003\000\003\000\004\000\021\000\
    \022\000\112\000\038\000\124\000\039\000\038\000\038\000\004\000\
    \125\000\039\000\112\000\126\000\004\000\004\000\005\000\088\000\
    \093\000\129\000\027\000\038\000\130\000\131\000\021\000\039\000\
    \147\000\038\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\150\000\122\000\088\000\093\000\
    \000\000\158\000\112\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\122\000\088\000\093\000\
    \159\000\005\000\160\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\005\000\005\000\005\000\
    \005\000\005\000\005\000\005\000\005\000\028\000\161\000\028\000\
    \028\000\028\000\028\000\028\000\028\000\028\000\028\000\028\000\
    \028\000\029\000\080\000\029\000\029\000\029\000\029\000\029\000\
    \029\000\029\000\029\000\029\000\029\000\091\000\103\000\103\000\
    \103\000\103\000\103\000\103\000\103\000\103\000\103\000\103\000\
    \029\000\080\000\162\000\163\000\164\000\165\000\166\000\029\000\
    \080\000\167\000\168\000\169\000\091\000\170\000\111\000\030\000\
    \029\000\030\000\030\000\030\000\030\000\030\000\030\000\030\000\
    \030\000\030\000\030\000\171\000\172\000\089\000\174\000\041\000\
    \029\000\080\000\175\000\176\000\029\000\080\000\030\000\029\000\
    \080\000\177\000\178\000\111\000\091\000\030\000\111\000\179\000\
    \029\000\042\000\090\000\091\000\089\000\077\000\030\000\077\000\
    \113\000\030\000\077\000\077\000\077\000\077\000\077\000\077\000\
    \077\000\077\000\077\000\077\000\089\000\180\000\030\000\182\000\
    \183\000\090\000\030\000\185\000\032\000\030\000\032\000\113\000\
    \090\000\032\000\186\000\188\000\089\000\192\000\030\000\200\000\
    \113\000\030\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\089\000\208\000\209\000\213\000\
    \219\000\090\000\214\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\220\000\221\000\214\000\
    \190\000\032\000\214\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\032\000\032\000\032\000\
    \032\000\032\000\032\000\032\000\032\000\034\000\105\000\105\000\
    \105\000\105\000\105\000\105\000\105\000\105\000\105\000\105\000\
    \222\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\223\000\218\000\225\000\226\000\194\000\
    \227\000\228\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\218\000\231\000\232\000\233\000\
    \034\000\234\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\034\000\034\000\034\000\034\000\
    \034\000\034\000\034\000\034\000\047\000\107\000\107\000\107\000\
    \107\000\107\000\107\000\107\000\107\000\107\000\107\000\235\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\236\000\237\000\238\000\239\000\240\000\242\000\
    \243\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\244\000\241\000\245\000\246\000\047\000\
    \241\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\047\000\047\000\047\000\047\000\047\000\
    \047\000\047\000\047\000\048\000\118\000\118\000\118\000\118\000\
    \118\000\118\000\118\000\118\000\118\000\118\000\190\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\247\000\249\000\250\000\251\000\252\000\253\000\254\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\255\000\000\001\001\001\002\001\048\000\003\001\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\048\000\048\000\048\000\048\000\048\000\048\000\
    \048\000\048\000\049\000\120\000\120\000\120\000\120\000\120\000\
    \120\000\120\000\120\000\120\000\120\000\194\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \004\001\005\001\006\001\007\001\008\001\009\001\010\001\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\011\001\012\001\013\001\014\001\049\000\015\001\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\049\000\049\000\049\000\049\000\049\000\049\000\049\000\
    \049\000\050\000\016\001\017\001\018\001\019\001\203\000\020\001\
    \021\001\203\000\203\000\022\001\023\001\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\203\000\
    \024\001\025\001\026\001\027\001\028\001\203\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \029\001\030\001\031\001\032\001\050\000\033\001\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \050\000\050\000\050\000\050\000\050\000\050\000\050\000\050\000\
    \051\000\034\001\035\001\036\001\037\001\211\000\038\001\039\001\
    \211\000\211\000\040\001\041\001\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\211\000\042\001\
    \043\001\044\001\045\001\046\001\211\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\047\001\
    \048\001\049\001\050\001\051\000\051\001\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\051\000\
    \051\000\051\000\051\000\051\000\051\000\051\000\051\000\052\000\
    \052\001\053\001\054\001\055\001\229\000\056\001\102\001\229\000\
    \229\000\104\001\097\001\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\229\000\064\001\068\001\
    \105\001\106\001\111\001\229\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\112\001\113\001\
    \114\001\120\001\052\000\124\001\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\052\000\052\000\
    \052\000\052\000\052\000\052\000\052\000\052\000\053\000\125\001\
    \108\001\108\001\126\001\060\001\108\001\127\001\060\001\060\001\
    \255\255\255\255\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\060\001\255\255\116\001\108\001\
    \129\001\255\255\060\001\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\097\001\116\001\108\001\
    \255\255\053\000\255\255\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\053\000\053\000\053\000\
    \053\000\053\000\053\000\053\000\053\000\054\000\077\001\077\001\
    \077\001\077\001\077\001\077\001\077\001\077\001\108\001\255\255\
    \255\255\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\255\255\255\255\255\255\255\255\
    \054\000\255\255\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\054\000\054\000\054\000\054\000\
    \054\000\054\000\054\000\054\000\055\000\255\255\255\255\122\001\
    \255\255\255\255\255\255\122\001\064\001\068\001\255\255\255\255\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\255\255\255\255\255\255\255\255\122\001\255\255\
    \255\255\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\255\255\255\255\255\255\122\001\055\000\
    \255\255\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\055\000\055\000\055\000\055\000\055\000\
    \055\000\055\000\055\000\056\000\080\001\080\001\080\001\080\001\
    \080\001\080\001\080\001\080\001\116\001\122\001\129\001\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\255\255\255\255\255\255\255\255\056\000\255\255\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\056\000\056\000\056\000\056\000\056\000\056\000\
    \056\000\056\000\057\000\088\001\088\001\088\001\088\001\088\001\
    \088\001\088\001\088\001\255\255\255\255\255\255\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\255\255\255\255\255\255\255\255\057\000\255\255\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\057\000\057\000\057\000\057\000\057\000\057\000\057\000\
    \057\000\058\000\091\001\091\001\091\001\091\001\091\001\091\001\
    \091\001\091\001\255\255\255\255\255\255\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \255\255\255\255\255\255\255\255\058\000\255\255\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \058\000\058\000\058\000\058\000\058\000\058\000\058\000\058\000\
    \059\000\255\255\255\255\255\255\255\255\096\001\255\255\255\255\
    \096\001\096\001\255\255\255\255\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\096\001\255\255\
    \255\255\255\255\255\255\255\255\096\001\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\255\255\
    \255\255\255\255\255\255\059\000\255\255\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\059\000\
    \059\000\059\000\059\000\059\000\059\000\059\000\059\000\060\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\255\255\255\255\
    \255\255\255\255\060\000\255\255\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\060\000\060\000\
    \060\000\060\000\060\000\060\000\060\000\060\000\061\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\255\255\255\255\255\255\
    \255\255\061\000\255\255\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\061\000\061\000\061\000\
    \061\000\061\000\061\000\061\000\061\000\062\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\255\255\255\255\255\255\255\255\
    \062\000\255\255\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\062\000\062\000\062\000\062\000\
    \062\000\062\000\062\000\062\000\063\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\255\255\255\255\255\255\255\255\063\000\
    \255\255\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\063\000\063\000\063\000\063\000\063\000\
    \063\000\063\000\063\000\064\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\255\255\255\255\255\255\255\255\064\000\255\255\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\064\000\064\000\064\000\064\000\064\000\064\000\
    \064\000\064\000\065\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\255\255\255\255\255\255\255\255\065\000\255\255\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\065\000\065\000\065\000\065\000\065\000\065\000\065\000\
    \065\000\066\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \255\255\255\255\255\255\255\255\066\000\255\255\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \066\000\066\000\066\000\066\000\066\000\066\000\066\000\066\000\
    \067\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\255\255\
    \255\255\255\255\255\255\067\000\255\255\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\067\000\
    \067\000\067\000\067\000\067\000\067\000\067\000\067\000\068\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\255\255\255\255\
    \255\255\255\255\068\000\255\255\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\068\000\068\000\
    \068\000\068\000\068\000\068\000\068\000\068\000\069\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\255\255\255\255\255\255\
    \255\255\069\000\255\255\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\069\000\069\000\069\000\
    \069\000\069\000\069\000\069\000\069\000\070\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\255\255\255\255\255\255\255\255\
    \070\000\255\255\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\070\000\070\000\070\000\070\000\
    \070\000\070\000\070\000\070\000\071\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\001\255\255\082\001\255\255\071\000\
    \255\255\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\071\000\071\000\071\000\071\000\071\000\
    \071\000\071\000\071\000\072\000\255\255\059\001\255\255\255\255\
    \059\001\255\255\255\255\255\255\255\255\255\255\255\255\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\255\255\082\001\255\255\255\255\071\001\059\001\255\255\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\255\255\255\255\255\255\255\255\072\000\255\255\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\072\000\072\000\072\000\072\000\072\000\072\000\
    \072\000\072\000\073\000\063\001\255\255\255\255\063\001\093\001\
    \255\255\255\255\093\001\093\001\255\255\255\255\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \093\001\255\255\255\255\255\255\063\001\255\255\093\001\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\255\255\255\255\255\255\255\255\073\000\255\255\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\073\000\073\000\073\000\073\000\073\000\073\000\073\000\
    \073\000\076\000\255\255\076\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\076\000\076\000\076\000\096\000\082\001\255\255\
    \255\255\071\001\255\255\215\000\076\000\076\000\076\000\076\000\
    \076\000\076\000\255\255\082\000\215\000\255\255\255\255\255\255\
    \255\255\093\001\255\255\255\255\096\000\078\000\078\000\078\000\
    \078\000\078\000\078\000\078\000\078\000\078\000\078\000\081\000\
    \216\000\255\255\082\000\255\255\076\000\076\000\076\000\076\000\
    \076\000\076\000\078\000\078\000\215\000\255\255\095\000\255\255\
    \085\000\078\000\082\000\255\255\096\000\255\255\081\000\216\000\
    \255\255\255\255\255\255\096\000\059\001\081\000\255\255\255\255\
    \216\000\255\255\082\000\255\255\255\255\095\000\081\000\085\000\
    \100\001\082\000\078\000\078\000\095\000\087\000\085\000\255\255\
    \255\255\078\000\082\000\086\000\255\255\083\000\081\000\083\000\
    \083\000\083\000\083\000\083\000\083\000\083\000\083\000\083\000\
    \083\000\094\000\255\255\255\255\087\000\095\000\081\000\085\000\
    \255\255\255\255\086\000\085\000\083\000\255\255\085\000\255\255\
    \255\255\086\000\255\255\083\000\087\000\255\255\255\255\255\255\
    \094\000\100\001\086\000\255\255\083\000\255\255\057\001\057\001\
    \255\255\057\001\057\001\255\255\087\000\255\255\255\255\255\255\
    \094\000\255\255\086\000\087\000\083\000\255\255\255\255\057\001\
    \083\000\255\255\063\001\083\000\087\000\057\001\093\001\057\001\
    \094\000\099\000\086\000\099\000\083\000\100\001\099\000\099\000\
    \099\000\099\000\099\000\099\000\099\000\099\000\099\000\099\000\
    \094\000\100\000\100\000\100\000\100\000\100\000\100\000\100\000\
    \100\000\100\000\100\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\100\001\255\255\255\255\100\000\100\000\
    \255\255\101\000\255\255\101\000\255\255\100\000\101\000\101\000\
    \101\000\101\000\101\000\101\000\101\000\101\000\101\000\101\000\
    \102\000\102\000\102\000\102\000\102\000\102\000\102\000\102\000\
    \102\000\102\000\255\255\255\255\255\255\255\255\100\000\100\000\
    \255\255\255\255\255\255\255\255\255\255\100\000\102\000\255\255\
    \255\255\255\255\255\255\255\255\102\000\104\000\104\000\104\000\
    \104\000\104\000\104\000\104\000\104\000\104\000\104\000\106\000\
    \106\000\106\000\106\000\106\000\106\000\106\000\106\000\106\000\
    \106\000\255\255\255\255\104\000\255\255\255\255\102\000\255\255\
    \255\255\104\000\255\255\255\255\102\000\106\000\255\255\114\000\
    \255\255\114\000\255\255\106\000\114\000\114\000\114\000\114\000\
    \114\000\114\000\114\000\114\000\114\000\114\000\255\255\255\255\
    \255\255\255\255\255\255\104\000\255\255\255\255\255\255\255\255\
    \255\255\104\000\255\255\255\255\255\255\106\000\255\255\255\255\
    \255\255\255\255\255\255\106\000\108\000\108\000\108\000\108\000\
    \108\000\108\000\108\000\108\000\108\000\108\000\100\001\255\255\
    \255\255\255\255\255\255\255\255\255\255\108\000\108\000\108\000\
    \108\000\108\000\108\000\255\255\074\001\074\001\074\001\074\001\
    \074\001\074\001\074\001\074\001\255\255\255\255\255\255\255\255\
    \109\000\255\255\109\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\109\000\109\000\109\000\255\255\108\000\108\000\108\000\
    \108\000\108\000\108\000\109\000\109\000\109\000\109\000\109\000\
    \109\000\255\255\255\255\255\255\074\001\057\001\109\000\255\255\
    \255\255\255\255\109\000\116\000\255\255\116\000\255\255\109\000\
    \116\000\116\000\116\000\116\000\116\000\116\000\116\000\116\000\
    \116\000\116\000\255\255\109\000\109\000\109\000\109\000\109\000\
    \109\000\255\255\255\255\109\000\074\001\255\255\109\000\255\255\
    \255\255\255\255\109\000\255\255\255\255\255\255\255\255\109\000\
    \115\000\115\000\115\000\115\000\115\000\115\000\115\000\115\000\
    \115\000\115\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\115\000\115\000\115\000\115\000\115\000\115\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \115\000\117\000\117\000\117\000\117\000\117\000\117\000\117\000\
    \117\000\117\000\117\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\115\000\115\000\115\000\115\000\115\000\115\000\117\000\
    \255\255\255\255\255\255\255\255\255\255\117\000\255\255\255\255\
    \115\000\255\255\119\000\119\000\119\000\119\000\119\000\119\000\
    \119\000\119\000\119\000\119\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\117\000\
    \119\000\255\255\255\255\255\255\255\255\117\000\119\000\255\255\
    \255\255\255\255\153\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \119\000\255\255\255\255\255\255\074\001\255\255\119\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\255\255\255\255\255\255\255\255\153\000\255\255\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\153\000\153\000\153\000\153\000\153\000\153\000\153\000\
    \153\000\154\000\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \255\255\255\255\255\255\255\255\154\000\255\255\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \154\000\154\000\154\000\154\000\154\000\154\000\154\000\154\000\
    \155\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\255\255\
    \255\255\255\255\255\255\155\000\255\255\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\155\000\
    \155\000\155\000\155\000\155\000\155\000\155\000\155\000\156\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\255\255\255\255\
    \255\255\255\255\156\000\255\255\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\156\000\156\000\
    \156\000\156\000\156\000\156\000\156\000\156\000\157\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\255\255\255\255\255\255\
    \255\255\157\000\255\255\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\157\000\157\000\157\000\
    \157\000\157\000\157\000\157\000\157\000\197\000\197\000\255\255\
    \197\000\197\000\206\000\206\000\255\255\206\000\206\000\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\197\000\255\255\
    \255\255\255\255\255\255\206\000\197\000\255\255\197\000\255\255\
    \255\255\206\000\255\255\255\255\197\000\197\000\255\255\255\255\
    \255\255\255\255\255\255\197\000\255\255\255\255\255\255\255\255\
    \255\255\206\000\206\000\206\000\206\000\206\000\206\000\206\000\
    \206\000\206\000\206\000\210\000\210\000\210\000\210\000\210\000\
    \210\000\210\000\210\000\210\000\210\000\230\000\255\255\255\255\
    \230\000\230\000\085\001\085\001\085\001\085\001\085\001\085\001\
    \085\001\085\001\255\255\255\255\255\255\255\255\230\000\210\000\
    \255\255\255\255\255\255\255\255\230\000\255\255\255\255\255\255\
    \210\000\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\206\000\255\255\255\255\
    \255\255\206\000\085\001\255\255\210\000\255\255\255\255\210\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \210\000\255\255\255\255\230\000\078\001\078\001\078\001\078\001\
    \078\001\078\001\078\001\078\001\078\001\078\001\255\255\255\255\
    \230\000\255\255\085\001\255\255\255\255\078\001\078\001\078\001\
    \078\001\078\001\078\001\255\255\255\255\230\000\255\255\255\255\
    \255\255\255\255\255\255\230\000\255\255\230\000\255\255\255\255\
    \230\000\255\255\255\255\230\000\255\255\255\255\230\000\230\000\
    \255\255\230\000\255\255\230\000\255\255\078\001\078\001\078\001\
    \078\001\078\001\078\001\079\001\079\001\079\001\079\001\079\001\
    \079\001\079\001\079\001\079\001\079\001\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\079\001\079\001\079\001\079\001\
    \079\001\079\001\089\001\089\001\089\001\089\001\089\001\089\001\
    \089\001\089\001\089\001\089\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\089\001\089\001\089\001\089\001\089\001\
    \089\001\255\255\255\255\255\255\079\001\079\001\079\001\079\001\
    \079\001\079\001\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\197\000\255\255\255\255\255\255\
    \255\255\206\000\255\255\089\001\089\001\089\001\089\001\089\001\
    \089\001\255\255\090\001\090\001\090\001\090\001\090\001\090\001\
    \090\001\090\001\090\001\090\001\255\255\255\255\255\255\255\255\
    \255\255\255\255\085\001\090\001\090\001\090\001\090\001\090\001\
    \090\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\090\001\090\001\090\001\090\001\090\001\
    \090\001\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_base_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\002\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \004\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_backtrk_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_default_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000";
  Lexing.lex_trans_code = 
   "\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\001\000\000\000\001\000\001\000\001\000\001\000\001\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\001\000\000\000\001\000\000\000\000\000\000\000\
    \001\000\000\000\001\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\
    \000\000\000\000\000\000";
  Lexing.lex_check_code = 
   "\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\229\000\255\255\230\000\229\000\229\000\230\000\230\000\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\229\000\255\255\230\000\255\255\255\255\255\255\
    \229\000\255\255\230\000\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\255\
    \255\255\255\255\255\255";
  Lexing.lex_code = 
   "\255\001\255\255\000\001\255";
}

let rec initial lexbuf =
    __ocaml_lex_initial_rec lexbuf 0
and __ocaml_lex_initial_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 500 "cil/src/frontc/clexer.mll"
      ( 
	do_lex_comment comment lexbuf ;
	addWhite lexbuf ;
        initial lexbuf
      )
# 1886 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 506 "cil/src/frontc/clexer.mll"
                        c
# 1892 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 507 "cil/src/frontc/clexer.mll"
    ( if c = !annot_char then begin
	Cabshelper.continue_annot
	  (currentLoc ())
	  (fun () ->
             save_current_pos ();
	     Buffer.clear buf;
	     annot_first_token lexbuf)
	  (fun () ->
	     initial lexbuf)
	  "Skipping annotation"
      end else 
	begin
	  do_lex_comment comment lexbuf ;
          addWhite lexbuf;
          initial lexbuf
	end 
    )
# 1912 "cil/src/frontc/clexer.ml"

  | 2 ->
# 526 "cil/src/frontc/clexer.mll"
    ( do_lex_comment onelinecomment lexbuf ;
      E.newline();
      if is_oneline_ghost () then begin
        exit_oneline_ghost ();
        RGHOST
      end else begin
        addWhite lexbuf;
        initial lexbuf
      end
    )
# 1926 "cil/src/frontc/clexer.ml"

  | 3 ->
let
# 537 "cil/src/frontc/clexer.mll"
                    c
# 1932 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf (lexbuf.Lexing.lex_start_pos + 2) in
# 538 "cil/src/frontc/clexer.mll"
    ( if c = !annot_char then begin
	Cabshelper.continue_annot
	  (currentLoc())
	  (fun () ->
             save_current_pos ();
	     Buffer.clear buf;
	     annot_one_line lexbuf)
	  (fun () -> initial lexbuf)
	  "Skipping annotation"
      end else 
	begin
	  do_lex_comment onelinecomment lexbuf ;
	  E.newline();
	  if is_oneline_ghost () then 
	    begin
              exit_oneline_ghost ();
              RGHOST
	    end 
	  else 
	    begin
              addWhite lexbuf;
              initial lexbuf
	    end
	end
    )
# 1960 "cil/src/frontc/clexer.ml"

  | 4 ->
# 563 "cil/src/frontc/clexer.mll"
           (addWhite lexbuf; initial lexbuf)
# 1965 "cil/src/frontc/clexer.ml"

  | 5 ->
# 564 "cil/src/frontc/clexer.mll"
                                        ( E.newline ();
                                          if !pragmaLine then
                                            begin
                                              pragmaLine := false;
                                              PRAGMA_EOL
                                            end
                                          else if is_oneline_ghost () then
                                            begin
                                              exit_oneline_ghost ();
                                              RGHOST
                                            end
                                          else begin
                                            addWhite lexbuf;
                                            initial lexbuf
                                          end )
# 1984 "cil/src/frontc/clexer.ml"

  | 6 ->
# 579 "cil/src/frontc/clexer.mll"
                                        ( addWhite lexbuf;
                                          E.newline ();
                                          initial lexbuf
                                        )
# 1992 "cil/src/frontc/clexer.ml"

  | 7 ->
# 583 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 1997 "cil/src/frontc/clexer.ml"

  | 8 ->
# 584 "cil/src/frontc/clexer.mll"
                                   ( PRAGMA (currentLoc ()) )
# 2002 "cil/src/frontc/clexer.ml"

  | 9 ->
# 585 "cil/src/frontc/clexer.mll"
          ( CST_CHAR (chr lexbuf, currentLoc ()))
# 2007 "cil/src/frontc/clexer.ml"

  | 10 ->
# 586 "cil/src/frontc/clexer.mll"
          ( CST_WCHAR (chr lexbuf, currentLoc ()) )
# 2012 "cil/src/frontc/clexer.ml"

  | 11 ->
# 587 "cil/src/frontc/clexer.mll"
         (  addLexeme lexbuf; (* '"' *)
(* matth: BUG:  this could be either a regular string or a wide string.
 *  e.g. if it's the "world" in
 *     L"Hello, " "world"
 *  then it should be treated as wide even though there's no L immediately
 *  preceding it.  See test/small1/wchar5.c for a failure case. *)
                                          try CST_STRING (str lexbuf, currentLoc ())
                                          with e ->
                                             raise (InternalError
                                                     ("str: " ^
                                                      Printexc.to_string e)))
# 2027 "cil/src/frontc/clexer.ml"

  | 12 ->
# 598 "cil/src/frontc/clexer.mll"
           ( (* weimer: wchar_t string literal *)
                                          try CST_WSTRING(str lexbuf, currentLoc ())
                                          with e ->
                                             raise (InternalError
                                                     ("wide string: " ^
                                                      Printexc.to_string e)))
# 2037 "cil/src/frontc/clexer.ml"

  | 13 ->
# 604 "cil/src/frontc/clexer.mll"
             (CST_FLOAT (Lexing.lexeme lexbuf, currentLoc ()))
# 2042 "cil/src/frontc/clexer.ml"

  | 14 ->
# 605 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 2047 "cil/src/frontc/clexer.ml"

  | 15 ->
# 606 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 2052 "cil/src/frontc/clexer.ml"

  | 16 ->
# 607 "cil/src/frontc/clexer.mll"
            (CST_INT (Lexing.lexeme lexbuf, currentLoc ()))
# 2057 "cil/src/frontc/clexer.ml"

  | 17 ->
# 608 "cil/src/frontc/clexer.mll"
             (EOF)
# 2062 "cil/src/frontc/clexer.ml"

  | 18 ->
# 609 "cil/src/frontc/clexer.mll"
           (ELLIPSIS)
# 2067 "cil/src/frontc/clexer.ml"

  | 19 ->
# 610 "cil/src/frontc/clexer.mll"
          (PLUS_EQ)
# 2072 "cil/src/frontc/clexer.ml"

  | 20 ->
# 611 "cil/src/frontc/clexer.mll"
          (MINUS_EQ)
# 2077 "cil/src/frontc/clexer.ml"

  | 21 ->
# 612 "cil/src/frontc/clexer.mll"
          (STAR_EQ)
# 2082 "cil/src/frontc/clexer.ml"

  | 22 ->
# 613 "cil/src/frontc/clexer.mll"
          (SLASH_EQ)
# 2087 "cil/src/frontc/clexer.ml"

  | 23 ->
# 614 "cil/src/frontc/clexer.mll"
          (PERCENT_EQ)
# 2092 "cil/src/frontc/clexer.ml"

  | 24 ->
# 615 "cil/src/frontc/clexer.mll"
          (PIPE_EQ)
# 2097 "cil/src/frontc/clexer.ml"

  | 25 ->
# 616 "cil/src/frontc/clexer.mll"
          (AND_EQ)
# 2102 "cil/src/frontc/clexer.ml"

  | 26 ->
# 617 "cil/src/frontc/clexer.mll"
          (CIRC_EQ)
# 2107 "cil/src/frontc/clexer.ml"

  | 27 ->
# 618 "cil/src/frontc/clexer.mll"
           (INF_INF_EQ)
# 2112 "cil/src/frontc/clexer.ml"

  | 28 ->
# 619 "cil/src/frontc/clexer.mll"
           (SUP_SUP_EQ)
# 2117 "cil/src/frontc/clexer.ml"

  | 29 ->
# 620 "cil/src/frontc/clexer.mll"
          (INF_INF)
# 2122 "cil/src/frontc/clexer.ml"

  | 30 ->
# 621 "cil/src/frontc/clexer.mll"
          (SUP_SUP)
# 2127 "cil/src/frontc/clexer.ml"

  | 31 ->
# 622 "cil/src/frontc/clexer.mll"
           (EQ_EQ)
# 2132 "cil/src/frontc/clexer.ml"

  | 32 ->
# 623 "cil/src/frontc/clexer.mll"
           (EXCLAM_EQ)
# 2137 "cil/src/frontc/clexer.ml"

  | 33 ->
# 624 "cil/src/frontc/clexer.mll"
          (INF_EQ)
# 2142 "cil/src/frontc/clexer.ml"

  | 34 ->
# 625 "cil/src/frontc/clexer.mll"
          (SUP_EQ)
# 2147 "cil/src/frontc/clexer.ml"

  | 35 ->
# 626 "cil/src/frontc/clexer.mll"
         (EQ)
# 2152 "cil/src/frontc/clexer.ml"

  | 36 ->
# 627 "cil/src/frontc/clexer.mll"
         (INF)
# 2157 "cil/src/frontc/clexer.ml"

  | 37 ->
# 628 "cil/src/frontc/clexer.mll"
         (SUP)
# 2162 "cil/src/frontc/clexer.ml"

  | 38 ->
# 629 "cil/src/frontc/clexer.mll"
          (PLUS_PLUS (currentLoc ()))
# 2167 "cil/src/frontc/clexer.ml"

  | 39 ->
# 630 "cil/src/frontc/clexer.mll"
          (MINUS_MINUS (currentLoc ()))
# 2172 "cil/src/frontc/clexer.ml"

  | 40 ->
# 631 "cil/src/frontc/clexer.mll"
          (ARROW)
# 2177 "cil/src/frontc/clexer.ml"

  | 41 ->
# 632 "cil/src/frontc/clexer.mll"
         (PLUS (currentLoc ()))
# 2182 "cil/src/frontc/clexer.ml"

  | 42 ->
# 633 "cil/src/frontc/clexer.mll"
         (MINUS (currentLoc ()))
# 2187 "cil/src/frontc/clexer.ml"

  | 43 ->
# 635 "cil/src/frontc/clexer.mll"
                    ( if is_ghost_code () then might_end_ghost lexbuf
                      else
                        STAR (currentLoc ()))
# 2194 "cil/src/frontc/clexer.ml"

  | 44 ->
# 638 "cil/src/frontc/clexer.mll"
         (SLASH)
# 2199 "cil/src/frontc/clexer.ml"

  | 45 ->
# 639 "cil/src/frontc/clexer.mll"
         (PERCENT)
# 2204 "cil/src/frontc/clexer.ml"

  | 46 ->
# 640 "cil/src/frontc/clexer.mll"
         (EXCLAM (currentLoc ()))
# 2209 "cil/src/frontc/clexer.ml"

  | 47 ->
# 641 "cil/src/frontc/clexer.mll"
          (AND_AND (currentLoc ()))
# 2214 "cil/src/frontc/clexer.ml"

  | 48 ->
# 642 "cil/src/frontc/clexer.mll"
          (PIPE_PIPE)
# 2219 "cil/src/frontc/clexer.ml"

  | 49 ->
# 643 "cil/src/frontc/clexer.mll"
         (AND (currentLoc ()))
# 2224 "cil/src/frontc/clexer.ml"

  | 50 ->
# 644 "cil/src/frontc/clexer.mll"
         (PIPE)
# 2229 "cil/src/frontc/clexer.ml"

  | 51 ->
# 645 "cil/src/frontc/clexer.mll"
         (CIRC)
# 2234 "cil/src/frontc/clexer.ml"

  | 52 ->
# 646 "cil/src/frontc/clexer.mll"
         (QUEST)
# 2239 "cil/src/frontc/clexer.ml"

  | 53 ->
# 647 "cil/src/frontc/clexer.mll"
         (COLON)
# 2244 "cil/src/frontc/clexer.ml"

  | 54 ->
# 648 "cil/src/frontc/clexer.mll"
                (TILDE (currentLoc ()))
# 2249 "cil/src/frontc/clexer.ml"

  | 55 ->
# 650 "cil/src/frontc/clexer.mll"
               (dbgToken (LBRACE (currentLoc ())))
# 2254 "cil/src/frontc/clexer.ml"

  | 56 ->
# 651 "cil/src/frontc/clexer.mll"
               (dbgToken (RBRACE (currentLoc ())))
# 2259 "cil/src/frontc/clexer.ml"

  | 57 ->
# 652 "cil/src/frontc/clexer.mll"
          (LBRACKET)
# 2264 "cil/src/frontc/clexer.ml"

  | 58 ->
# 653 "cil/src/frontc/clexer.mll"
          (RBRACKET)
# 2269 "cil/src/frontc/clexer.ml"

  | 59 ->
# 654 "cil/src/frontc/clexer.mll"
               (dbgToken (LPAREN (currentLoc ())) )
# 2274 "cil/src/frontc/clexer.ml"

  | 60 ->
# 655 "cil/src/frontc/clexer.mll"
          (RPAREN)
# 2279 "cil/src/frontc/clexer.ml"

  | 61 ->
# 656 "cil/src/frontc/clexer.mll"
               (dbgToken (SEMICOLON (currentLoc ())) )
# 2284 "cil/src/frontc/clexer.ml"

  | 62 ->
# 657 "cil/src/frontc/clexer.mll"
          (COMMA)
# 2289 "cil/src/frontc/clexer.ml"

  | 63 ->
# 658 "cil/src/frontc/clexer.mll"
          (DOT)
# 2294 "cil/src/frontc/clexer.ml"

  | 64 ->
# 659 "cil/src/frontc/clexer.mll"
             (SIZEOF (currentLoc ()))
# 2299 "cil/src/frontc/clexer.ml"

  | 65 ->
# 660 "cil/src/frontc/clexer.mll"
                                        ( if !Cprint.msvcMode then
                                             MSASM (msasm lexbuf, currentLoc ())
                                          else (ASM (currentLoc ())) )
# 2306 "cil/src/frontc/clexer.ml"

  | 66 ->
# 665 "cil/src/frontc/clexer.mll"
                                        ( matchingParsOpen := 0;
                                          let _ = matchingpars lexbuf in
                                          addWhite lexbuf;
                                          initial lexbuf
                                        )
# 2315 "cil/src/frontc/clexer.ml"

  | 67 ->
# 672 "cil/src/frontc/clexer.mll"
                                        (AT_TRANSFORM (currentLoc ()))
# 2320 "cil/src/frontc/clexer.ml"

  | 68 ->
# 673 "cil/src/frontc/clexer.mll"
                                        (AT_TRANSFORMEXPR (currentLoc ()))
# 2325 "cil/src/frontc/clexer.ml"

  | 69 ->
# 674 "cil/src/frontc/clexer.mll"
                                        (AT_SPECIFIER (currentLoc ()))
# 2330 "cil/src/frontc/clexer.ml"

  | 70 ->
# 675 "cil/src/frontc/clexer.mll"
                                        (AT_EXPR (currentLoc ()))
# 2335 "cil/src/frontc/clexer.ml"

  | 71 ->
# 676 "cil/src/frontc/clexer.mll"
                                        (AT_NAME)
# 2340 "cil/src/frontc/clexer.ml"

  | 72 ->
# 680 "cil/src/frontc/clexer.mll"
                                        (addWhite lexbuf; initial lexbuf )
# 2345 "cil/src/frontc/clexer.ml"

  | 73 ->
# 681 "cil/src/frontc/clexer.mll"
           (scan_ident (Lexing.lexeme lexbuf))
# 2350 "cil/src/frontc/clexer.ml"

  | 74 ->
# 683 "cil/src/frontc/clexer.mll"
  ( if is_oneline_ghost() then begin
      exit_oneline_ghost (); RGHOST
    end
    else EOF
  )
# 2359 "cil/src/frontc/clexer.ml"

  | 75 ->
# 688 "cil/src/frontc/clexer.mll"
       (E.parse_error "Invalid symbol")
# 2364 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_initial_rec lexbuf __ocaml_lex_state

and might_end_ghost lexbuf =
    __ocaml_lex_might_end_ghost_rec lexbuf 188
and __ocaml_lex_might_end_ghost_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 691 "cil/src/frontc/clexer.mll"
        ( exit_ghost_code(); RGHOST )
# 2375 "cil/src/frontc/clexer.ml"

  | 1 ->
# 692 "cil/src/frontc/clexer.mll"
       ( STAR (currentLoc()) )
# 2380 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_might_end_ghost_rec lexbuf __ocaml_lex_state

and comment buffer lexbuf =
    __ocaml_lex_comment_rec buffer lexbuf 190
and __ocaml_lex_comment_rec buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 695 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; )
# 2391 "cil/src/frontc/clexer.ml"

  | 1 ->
# 696 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; lex_comment comment buffer lexbuf )
# 2396 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_comment_rec buffer lexbuf __ocaml_lex_state

and onelinecomment buffer lexbuf =
    __ocaml_lex_onelinecomment_rec buffer lexbuf 194
and __ocaml_lex_onelinecomment_rec buffer lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 699 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; )
# 2407 "cil/src/frontc/clexer.ml"

  | 1 ->
# 700 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; lex_comment onelinecomment buffer lexbuf )
# 2412 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_onelinecomment_rec buffer lexbuf __ocaml_lex_state

and matchingpars lexbuf =
    __ocaml_lex_matchingpars_rec lexbuf 197
and __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 703 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; E.newline (); matchingpars lexbuf )
# 2423 "cil/src/frontc/clexer.ml"

  | 1 ->
# 704 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; matchingpars lexbuf )
# 2428 "cil/src/frontc/clexer.ml"

  | 2 ->
# 705 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; incr matchingParsOpen; matchingpars lexbuf )
# 2433 "cil/src/frontc/clexer.ml"

  | 3 ->
# 706 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; decr matchingParsOpen;
                  if !matchingParsOpen = 0 then
                     ()
                  else
                     matchingpars lexbuf
                )
# 2443 "cil/src/frontc/clexer.ml"

  | 4 ->
# 712 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; 
		  do_lex_comment comment lexbuf ;
                  matchingpars lexbuf )
# 2450 "cil/src/frontc/clexer.ml"

  | 5 ->
# 715 "cil/src/frontc/clexer.mll"
        ( addWhite lexbuf; (* '"' *)
                  let _ = str lexbuf in
                  matchingpars lexbuf )
# 2457 "cil/src/frontc/clexer.ml"

  | 6 ->
# 718 "cil/src/frontc/clexer.mll"
                 ( addWhite lexbuf; matchingpars lexbuf )
# 2462 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_matchingpars_rec lexbuf __ocaml_lex_state

and hash lexbuf =
  lexbuf.Lexing.lex_mem <- Array.create 2 (-1) ;   __ocaml_lex_hash_rec lexbuf 206
and __ocaml_lex_hash_rec lexbuf __ocaml_lex_state =
  match Lexing.new_engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 722 "cil/src/frontc/clexer.mll"
        ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2473 "cil/src/frontc/clexer.ml"

  | 1 ->
# 723 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; hash lexbuf)
# 2478 "cil/src/frontc/clexer.ml"

  | 2 ->
# 724 "cil/src/frontc/clexer.mll"
         ( addWhite lexbuf; (* We are seeing a line number. This is the number for the
                   * next line *)
                 let s = Lexing.lexeme lexbuf in
                 let lineno = try
                   int_of_string s
                 with Failure ("int_of_string") ->
                   (* the int is too big. *)
                   Cilmsg.warning "Bad line number in preprocessed file: %s" s;
                   (-1)
                 in
                 E.setCurrentLine (lineno - 1);
                  (* A file name may follow *)
		  file lexbuf )
# 2495 "cil/src/frontc/clexer.ml"

  | 3 ->
# 737 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; hash lexbuf )
# 2500 "cil/src/frontc/clexer.ml"

  | 4 ->
let
# 740 "cil/src/frontc/clexer.mll"
                                     pragmaName
# 2506 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme lexbuf lexbuf.Lexing.lex_mem.(0) lexbuf.Lexing.lex_curr_pos in
# 741 "cil/src/frontc/clexer.mll"
                ( let here = currentLoc () in
                  PRAGMA_LINE (pragmaName ^ pragma lexbuf, here)
                )
# 2512 "cil/src/frontc/clexer.ml"

  | 5 ->
# 744 "cil/src/frontc/clexer.mll"
                ( pragmaLine := true; PRAGMA (currentLoc ()) )
# 2517 "cil/src/frontc/clexer.ml"

  | 6 ->
# 745 "cil/src/frontc/clexer.mll"
            ( addWhite lexbuf; endline lexbuf)
# 2522 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_hash_rec lexbuf __ocaml_lex_state

and file lexbuf =
    __ocaml_lex_file_rec lexbuf 313
and __ocaml_lex_file_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 748 "cil/src/frontc/clexer.mll"
                      (addWhite lexbuf; E.newline (); initial lexbuf)
# 2533 "cil/src/frontc/clexer.ml"

  | 1 ->
# 749 "cil/src/frontc/clexer.mll"
          (addWhite lexbuf; file lexbuf)
# 2538 "cil/src/frontc/clexer.ml"

  | 2 ->
# 750 "cil/src/frontc/clexer.mll"
                                ( addWhite lexbuf;  (* '"' *)
                                   let n = Lexing.lexeme lexbuf in
                                   let n1 = String.sub n 1
                                       ((String.length n) - 2) in
                                   E.setCurrentFile n1;
				 endline lexbuf)
# 2548 "cil/src/frontc/clexer.ml"

  | 3 ->
# 757 "cil/src/frontc/clexer.mll"
      (addWhite lexbuf; endline lexbuf)
# 2553 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_file_rec lexbuf __ocaml_lex_state

and endline lexbuf =
    __ocaml_lex_endline_rec lexbuf 320
and __ocaml_lex_endline_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 760 "cil/src/frontc/clexer.mll"
                ( addWhite lexbuf; E.newline (); initial lexbuf)
# 2564 "cil/src/frontc/clexer.ml"

  | 1 ->
# 761 "cil/src/frontc/clexer.mll"
                                ( EOF )
# 2569 "cil/src/frontc/clexer.ml"

  | 2 ->
# 762 "cil/src/frontc/clexer.mll"
      ( addWhite lexbuf; endline lexbuf)
# 2574 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_endline_rec lexbuf __ocaml_lex_state

and pragma lexbuf =
    __ocaml_lex_pragma_rec lexbuf 324
and __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 765 "cil/src/frontc/clexer.mll"
                        ( E.newline (); "" )
# 2585 "cil/src/frontc/clexer.ml"

  | 1 ->
# 766 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (pragma lexbuf) )
# 2591 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_pragma_rec lexbuf __ocaml_lex_state

and str lexbuf =
    __ocaml_lex_str_rec lexbuf 327
and __ocaml_lex_str_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 770 "cil/src/frontc/clexer.mll"
                        ([])
# 2602 "cil/src/frontc/clexer.ml"

  | 1 ->
# 771 "cil/src/frontc/clexer.mll"
             (addLexeme lexbuf; lex_hex_escape str lexbuf)
# 2607 "cil/src/frontc/clexer.ml"

  | 2 ->
# 772 "cil/src/frontc/clexer.mll"
             (addLexeme lexbuf; lex_oct_escape str lexbuf)
# 2612 "cil/src/frontc/clexer.ml"

  | 3 ->
# 773 "cil/src/frontc/clexer.mll"
          (addLexeme lexbuf; lex_simple_escape str lexbuf)
# 2617 "cil/src/frontc/clexer.ml"

  | 4 ->
# 774 "cil/src/frontc/clexer.mll"
                        (E.parse_error "unterminated string" )
# 2622 "cil/src/frontc/clexer.ml"

  | 5 ->
# 775 "cil/src/frontc/clexer.mll"
     (addLexeme lexbuf; lex_unescaped str lexbuf)
# 2627 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_str_rec lexbuf __ocaml_lex_state

and chr lexbuf =
    __ocaml_lex_chr_rec lexbuf 338
and __ocaml_lex_chr_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 778 "cil/src/frontc/clexer.mll"
              ([])
# 2638 "cil/src/frontc/clexer.ml"

  | 1 ->
# 779 "cil/src/frontc/clexer.mll"
             (lex_hex_escape chr lexbuf)
# 2643 "cil/src/frontc/clexer.ml"

  | 2 ->
# 780 "cil/src/frontc/clexer.mll"
             (lex_oct_escape chr lexbuf)
# 2648 "cil/src/frontc/clexer.ml"

  | 3 ->
# 781 "cil/src/frontc/clexer.mll"
          (lex_simple_escape chr lexbuf)
# 2653 "cil/src/frontc/clexer.ml"

  | 4 ->
# 782 "cil/src/frontc/clexer.mll"
                        ( E.parse_error "unterminated char" )
# 2658 "cil/src/frontc/clexer.ml"

  | 5 ->
# 783 "cil/src/frontc/clexer.mll"
     (lex_unescaped chr lexbuf)
# 2663 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_chr_rec lexbuf __ocaml_lex_state

and msasm lexbuf =
    __ocaml_lex_msasm_rec lexbuf 349
and __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 786 "cil/src/frontc/clexer.mll"
                        ( msasm lexbuf )
# 2674 "cil/src/frontc/clexer.ml"

  | 1 ->
# 787 "cil/src/frontc/clexer.mll"
                        ( msasminbrace lexbuf )
# 2679 "cil/src/frontc/clexer.ml"

  | 2 ->
# 788 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (msasmnobrace lexbuf) )
# 2685 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasm_rec lexbuf __ocaml_lex_state

and msasminbrace lexbuf =
    __ocaml_lex_msasminbrace_rec lexbuf 353
and __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 792 "cil/src/frontc/clexer.mll"
                        ( "" )
# 2696 "cil/src/frontc/clexer.ml"

  | 1 ->
# 793 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in
                          cur ^ (msasminbrace lexbuf) )
# 2702 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasminbrace_rec lexbuf __ocaml_lex_state

and msasmnobrace lexbuf =
    __ocaml_lex_msasmnobrace_rec lexbuf 356
and __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 796 "cil/src/frontc/clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <-
                               lexbuf.Lexing.lex_curr_pos - 1;
                          "" )
# 2715 "cil/src/frontc/clexer.ml"

  | 1 ->
# 799 "cil/src/frontc/clexer.mll"
                        ( lexbuf.Lexing.lex_curr_pos <-
                               lexbuf.Lexing.lex_curr_pos - 5;
                          "" )
# 2722 "cil/src/frontc/clexer.ml"

  | 2 ->
# 802 "cil/src/frontc/clexer.mll"
                        ( let cur = Lexing.lexeme lexbuf in

                          cur ^ (msasmnobrace lexbuf) )
# 2729 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_msasmnobrace_rec lexbuf __ocaml_lex_state

and annot_first_token lexbuf =
    __ocaml_lex_annot_first_token_rec lexbuf 364
and __ocaml_lex_annot_first_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 807 "cil/src/frontc/clexer.mll"
            (
      if is_oneline_ghost () then E.parse_error "nested ghost code";
      Buffer.clear buf;
      enter_ghost_code ();
      LGHOST
    )
# 2745 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 813 "cil/src/frontc/clexer.mll"
                         c
# 2751 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 813 "cil/src/frontc/clexer.mll"
                           ( Buffer.add_char buf c; annot_first_token lexbuf )
# 2755 "cil/src/frontc/clexer.ml"

  | 2 ->
# 814 "cil/src/frontc/clexer.mll"
         ( E.newline(); Buffer.add_char buf '\n'; annot_first_token lexbuf )
# 2760 "cil/src/frontc/clexer.ml"

  | 3 ->
# 815 "cil/src/frontc/clexer.mll"
       ( annot_token lexbuf )
# 2765 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_first_token_rec lexbuf __ocaml_lex_state

and annot_token lexbuf =
    __ocaml_lex_annot_token_rec lexbuf 372
and __ocaml_lex_annot_token_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 817 "cil/src/frontc/clexer.mll"
         ( let s = Buffer.contents buf in
           make_annot s )
# 2777 "cil/src/frontc/clexer.ml"

  | 1 ->
# 819 "cil/src/frontc/clexer.mll"
         ( E.parse_error "Unterminated annotation" )
# 2782 "cil/src/frontc/clexer.ml"

  | 2 ->
# 820 "cil/src/frontc/clexer.mll"
         (E.newline(); Buffer.add_char buf '\n'; annot_token lexbuf )
# 2787 "cil/src/frontc/clexer.ml"

  | 3 ->
let
# 821 "cil/src/frontc/clexer.mll"
         c
# 2793 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 821 "cil/src/frontc/clexer.mll"
           ( Buffer.add_char buf c; annot_token lexbuf )
# 2797 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_token_rec lexbuf __ocaml_lex_state

and annot_one_line lexbuf =
    __ocaml_lex_annot_one_line_rec lexbuf 378
and __ocaml_lex_annot_one_line_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 824 "cil/src/frontc/clexer.mll"
            (
      if is_oneline_ghost () then E.parse_error "nested ghost code";
      enter_oneline_ghost (); LGHOST
    )
# 2811 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 828 "cil/src/frontc/clexer.mll"
                         c
# 2817 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 828 "cil/src/frontc/clexer.mll"
                           ( Buffer.add_char buf c; annot_one_line lexbuf )
# 2821 "cil/src/frontc/clexer.ml"

  | 2 ->
# 829 "cil/src/frontc/clexer.mll"
       ( annot_one_line_logic lexbuf )
# 2826 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_one_line_rec lexbuf __ocaml_lex_state

and annot_one_line_logic lexbuf =
    __ocaml_lex_annot_one_line_logic_rec lexbuf 385
and __ocaml_lex_annot_one_line_logic_rec lexbuf __ocaml_lex_state =
  match Lexing.engine __ocaml_lex_tables __ocaml_lex_state lexbuf with
      | 0 ->
# 831 "cil/src/frontc/clexer.mll"
         ( E.newline (); make_annot (Buffer.contents buf) )
# 2837 "cil/src/frontc/clexer.ml"

  | 1 ->
let
# 832 "cil/src/frontc/clexer.mll"
         c
# 2843 "cil/src/frontc/clexer.ml"
= Lexing.sub_lexeme_char lexbuf lexbuf.Lexing.lex_start_pos in
# 832 "cil/src/frontc/clexer.mll"
           ( Buffer.add_char buf c; annot_one_line_logic lexbuf )
# 2847 "cil/src/frontc/clexer.ml"

  | __ocaml_lex_state -> lexbuf.Lexing.refill_buff lexbuf; __ocaml_lex_annot_one_line_logic_rec lexbuf __ocaml_lex_state

;;

# 834 "cil/src/frontc/clexer.mll"
 


# 2857 "cil/src/frontc/clexer.ml"
