/*=============================================================================

    This file is part of ARB.

    ARB is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    ARB is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with ARB; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA

=============================================================================*/
/******************************************************************************

    Copyright (C) 2013 Fredrik Johansson

******************************************************************************/

#include "arb.h"

int main()
{
    slong iter;
    flint_rand_t state;

    flint_printf("div_2expm1_ui....");
    fflush(stdout);

    flint_randinit(state);

    for (iter = 0; iter < 100000; iter++)
    {
        arb_t a, b, c;
        ulong n;
        slong prec, acc1, acc2;
        fmpz_t t;

        arb_init(a);
        arb_init(b);
        arb_init(c);
        fmpz_init(t);

        prec = 2 + n_randint(state, 10000);
        arb_randtest(a, state, 1 + n_randint(state, 10000), 10);

        if (n_randint(state, 2))
            n = 1 + (n_randtest(state) % (10 * prec));
        else
            n = n_randtest(state);

        arb_div_2expm1_ui(b, a, n, prec);

        arb_one(c);
        if (n >= (UWORD(1) << (FLINT_BITS-1)))
        {
            arb_mul_2exp_si(c, c, (UWORD(1) << (FLINT_BITS-2)));
            arb_mul_2exp_si(c, c, (UWORD(1) << (FLINT_BITS-2)));
            arb_mul_2exp_si(c, c, n - (UWORD(1) << (FLINT_BITS-1)));
        }
        else
        {
            arb_mul_2exp_si(c, c, n);
        }

        arb_sub_ui(c, c, 1, prec);
        arb_div(c, a, c, prec);

        acc1 = arb_rel_accuracy_bits(a);
        acc2 = arb_rel_accuracy_bits(b);

        if (!arb_overlaps(b, c))
        {
            flint_printf("FAIL: containment\n\n");
            flint_printf("n = %wu\n", n);
            flint_printf("a = "); arb_print(a); flint_printf("\n\n");
            flint_printf("b = "); arb_print(b); flint_printf("\n\n");
            flint_printf("c = "); arb_print(c); flint_printf("\n\n");
            abort();
        }

        if (n > 0 && (acc2 < FLINT_MIN(prec, acc1) - 10) &&
                !(acc1 == -ARF_PREC_EXACT && acc2 == -ARF_PREC_EXACT))
        {
            flint_printf("FAIL: poor accuracy\n\n");
            flint_printf("prec=%wd, acc1=%wd, acc2=%wd\n\n", prec, acc1, acc2);
            flint_printf("n = %wu\n\n", n);
            flint_printf("a = "); arb_print(a); flint_printf("\n\n");
            flint_printf("b = "); arb_print(b); flint_printf("\n\n");
            flint_printf("c = "); arb_print(c); flint_printf("\n\n");
            abort();
        }

        arb_clear(a);
        arb_clear(b);
        arb_clear(c);
        fmpz_clear(t);
    }

    flint_randclear(state);
    flint_cleanup();
    flint_printf("PASS\n");
    return EXIT_SUCCESS;
}

