/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-folder-conf.c,v 1.14 2004/11/12 13:57:02 hoa Exp $
 */

#include "etpan-folder-conf.h"
#include "etpan-subapp.h"
#include "etpan-subapp-thread.h"
#include "etpan-errors.h"
#include "etpan-folder-params.h"
#include "etpan-app-subapp.h"
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ncurses.h>
#include <libetpan/libetpan.h>
#include <libetpan/charconv.h>
#include "etpan-msg-list-app.h"
#include "etpan-cfg-vfolder.h"
#include "etpan-app.h"
#include "etpan-msg-new.h"
#include "etpan-folder-common.h"
#include "etpan-folder-edit.h"
#include "etpan-thread-manager.h"
#include "etpan-config.h"
#include "etpan-folder-params.h"
#include "etpan-cfg-global-edit.h"
#include "etpan-help-viewer.h"
#include "etpan-search-input.h"

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd);
static void handle_fd(struct etpan_subapp * app, fd_set * fds);
static void handle_key(struct etpan_subapp * app, int key);
static void display(struct etpan_subapp * app, WINDOW * w);
static void set_color(struct etpan_subapp * app);
static int init(struct etpan_subapp * subapp);
static void done(struct etpan_subapp * subapp);
static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app);
static int display_init(struct etpan_subapp * app);

static struct etpan_subapp_driver etpan_folder_conf_app_driver = {
  .name = "folder-conf",
  .always_handle_key = 0,
  .always_on_top = 1,
  .get_idle_delay = NULL,
  .idle = NULL,
  .set_fd = set_fd,
  .handle_fd = handle_fd,
  .handle_key = handle_key,
  .handle_resize = NULL,
  .display = display,
  .set_color = set_color,
  .init = init,
  .done = done,
  .enter = NULL,
  .leave = leave,
  .display_init = display_init,
  .display_done = NULL,
};

enum {
  STATE_NORMAL,
  STATE_MOVING_NODE,
#if 0
  STATE_LOADING_NEW_CONFIG,
#endif
};

struct app_state {
  struct etpan_folder_common_app_state common_state;
  int config_init;
  struct etpan_app_config config;
  
  int state;
  int valid_conf;
  
  /* upcall */
  void (* upcall)(struct etpan_subapp *, int, void *);
  void * upcall_data;
};

static void set_fd(struct etpan_subapp * app, fd_set * fds, int * maxfd)
{
  etpan_subapp_thread_set_fd(app, fds, maxfd);
}

static void handle_fd(struct etpan_subapp * app, fd_set * fds)
{
  etpan_subapp_thread_handle_fd(app, fds);
}

static int edit_folder(struct etpan_subapp * app, int do_add);

static int delete_folder(struct etpan_subapp * app);

static int set_default_folder(struct etpan_subapp * app);

static int set_default_sent_folder(struct etpan_subapp * app);

static int set_default_draft_folder(struct etpan_subapp * app);

static int move_node(struct etpan_subapp * app, int direction);

static int reparent_node(struct etpan_subapp * app, int direction);

static int show_help(struct etpan_subapp * app);

static int global_conf(struct etpan_subapp * app);

static int validate_conf(struct etpan_subapp * app, int valid);

static void ask_quit(struct etpan_subapp * app);

static void handle_key(struct etpan_subapp * app, int key)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->state == STATE_MOVING_NODE) {
    switch (key) {
    case KEY_UP:
      move_node(app, 0);
      break;
    case KEY_DOWN:
      move_node(app, 1);
      break;
    case KEY_LEFT:
      reparent_node(app, 0);
      break;
    case KEY_RIGHT:
      reparent_node(app, 1);
      break;
    case '\n':
      state->state = STATE_NORMAL;
      break;
    }
  }
  else {
    etpan_folder_common_handle_key(app, &state->common_state, key);

#if 0
    if (state->state == STATE_LOADING_NEW_CONFIG)
      return;
#endif
    
    switch (key) {
    case KEY_F(1):
    case '?':
      show_help(app);
      break;
      
    case 'e':
    case '\n':
      edit_folder(app, 0);
      break;

    case 's':
      set_default_folder(app);
      break;

    case 'S':
      set_default_sent_folder(app);
      break;
      
    case 'p':
      set_default_draft_folder(app);
      break;
      
    case KEY_CTRL('G'):
      validate_conf(app, 0);
      break;
    
    case 'y':
      validate_conf(app, 1);
      break;
    
    case 'a':
      edit_folder(app, 1);
      break;

    case 'd':
      delete_folder(app);
      break;
    
    case 'm':
      state->state = STATE_MOVING_NODE;
      break;
      
    case 'g':
      global_conf(app);
      break;
      
    case 'q':
      ask_quit(app);
      break;
    }
  }
}

#define TITLE "etPan! - folder tree configuration"

static int display_init(struct etpan_subapp * app)
{
  etpan_subapp_set_title(app, TITLE);
  return etpan_app_subapp_display_init(app);
}

static void display(struct etpan_subapp * app, WINDOW * w)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->state == STATE_NORMAL) {
    etpan_folder_common_display(app, &state->common_state, w,
        "y: save  ^G: quit");
  }
  else {
    etpan_folder_common_display(app, &state->common_state, w,
        "[enter]: ok");
  }
}

static void set_color(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  etpan_folder_common_set_color(app, &state->common_state);
}

static int init(struct etpan_subapp * subapp)
{
  struct app_state * state;
  int r;
  
  state = malloc(sizeof(* state));
  if (state == NULL)
    goto err;
  
  state->config_init = 0;
  state->state = STATE_NORMAL;
  state->valid_conf = 0;
  state->upcall = NULL;
  state->upcall_data = NULL;
  
  r = etpan_folder_common_init(&state->common_state);
  if (r != NO_ERROR)
    goto free;
  
  subapp->data = state;
  
  return NO_ERROR;
  
 free:
  free(subapp->data);
 err:
  return ERROR_MEMORY;
}

static void done(struct etpan_subapp * subapp)
{
  struct app_state * state;
  
  state = subapp->data;

  etpan_folder_common_done(&state->common_state);

  free(subapp->data);
}

void etpan_folder_conf_app_flush(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;
  
  if (state->config_init)
    etpan_app_config_done(&state->config);
  state->config_init = 0;
  state->state = STATE_NORMAL;
  state->valid_conf = 0;
  state->upcall = NULL;
  state->upcall_data = NULL;
}

static void
expand_node(struct etpan_subapp * app, struct mailfolder * folder)
{
  unsigned int i;
  struct app_state * state;
  
  state = app->data;
  
  etpan_folder_set_opened(state->common_state.params, folder, TRUE);

  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;

    child = carray_get(folder->fld_children, i);
    
    expand_node(app, child);
  }
}

static int set_root_folder(struct etpan_subapp * app)
{
  int r;
  struct app_state * state;
  unsigned int old_selected;
  unsigned int old_first;
  struct mailfolder * folder;
  int type;
  
  state = app->data;
  
  old_selected = state->common_state.selected;
  old_first = state->common_state.first;
  
  r = etpan_folder_common_set_root_folder(&state->common_state,
      state->config.vfolder_config->root,
      ETPAN_FOLDER_COMMON_SHOW_SPECIAL_FOLDERS);
  if (r != NO_ERROR)
    return r;
  
  folder = state->config.vfolder_config->default_folder;
  if (folder != NULL) {
    type = etpan_folder_get_folder_type(state->common_state.params, folder);
    etpan_folder_set_folder_type(state->common_state.params, folder,
        type | ETPAN_FOLDER_TYPE_DEFAULT);
  }

  folder = state->config.vfolder_config->sent_folder;
  if (folder != NULL) {
    type = etpan_folder_get_folder_type(state->common_state.params, folder);
    etpan_folder_set_folder_type(state->common_state.params, folder,
        type | ETPAN_FOLDER_TYPE_SENT);
  }

  folder = state->config.vfolder_config->draft_folder;
  if (folder != NULL) {
    type = etpan_folder_get_folder_type(state->common_state.params, folder);
    etpan_folder_set_folder_type(state->common_state.params, folder,
        type | ETPAN_FOLDER_TYPE_DRAFT);
  }
  
  expand_node(app, state->config.vfolder_config->root);
  
  state->common_state.selected = old_selected;
  state->common_state.first = old_first;
  
  return NO_ERROR;
}

int etpan_folder_conf_app_set(struct etpan_subapp * app,
    void (* upcall)(struct etpan_subapp *, int, void *),
    void * upcall_data)
{
  struct app_state * state;
  int r;
  
  state = app->data;
  
  r = etpan_app_config_init(app->app->config_path,
      app->app, &state->config);
  if (r != NO_ERROR)
    goto err;
  
  state->config_init = 1;
  
  r = set_root_folder(app);
  if (r != NO_ERROR)
    goto done_config;
  
  state->upcall = upcall;
  state->upcall_data = upcall_data;
  
  etpan_subapp_set_title(app, TITLE);
  
  return NO_ERROR;
  
 done_config:
  etpan_app_config_done(&state->config);
 err:
  return ERROR_MEMORY;
}

struct etpan_subapp * etpan_folder_conf_app_new(struct etpan_app * app)
{
  return etpan_subapp_new(app, &etpan_folder_conf_app_driver);
}

static void leave(struct etpan_subapp * app, struct etpan_subapp * new_app)
{
  struct app_state * state;

  state = app->data;
  
  etpan_folder_conf_app_flush(app);
  etpan_folder_common_flush(app, &state->common_state);
}




static void
recursive_folder_replace_properties(struct etpan_subapp * app,
    struct etpan_vfolder_config * config,
    struct mailfolder * folder,
    struct mailfolder * old_folder,
    struct mailfolder * new_folder)
{
  unsigned int i;
  struct mailfolder * sent;
  struct mailfolder * draft;
  
  sent = etpan_vfolder_get_sent_folder(config, folder);
  
  if (sent == old_folder)
    etpan_vfolder_set_sent_folder(config, folder, new_folder);

  draft = etpan_vfolder_get_sent_folder(config, folder);
  
  if (draft == old_folder)
    etpan_vfolder_set_sent_folder(config, folder, new_folder);
  
  for(i = 0 ; i < carray_count(folder->fld_children) ; i ++) {
    struct mailfolder * child;
    
    child = carray_get(folder->fld_children, i);
    recursive_folder_replace_properties(app, config, child,
        old_folder, new_folder);
  }
}


enum {
  NEW_FOLDER,
  MODIFY_FOLDER,
};

struct folder_edit_data {
  struct etpan_subapp * app;
  int type;
};

static void
folder_edit_upcall(struct etpan_subapp * folder_editor_app,
    int valid, void * data)
{
  struct etpan_subapp * app;
  struct app_state * state;
  struct mailfolder * folder;
  struct mailfolder * new_folder;
  int r;
  int type;
  struct folder_edit_data * upcall_data;
  unsigned int i;
  struct etpan_vfolder_property prop;
  
  upcall_data = data;
  app = upcall_data->app;
  type = upcall_data->type;
  free(data);
  
  state = app->data;
  
  if (valid == ETPAN_FOLDER_EDIT_CANCEL) {
    etpan_app_quit_subapp(folder_editor_app);
    return;
  }
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  
  r = etpan_folder_editor_app_get_folder(folder_editor_app,
      &new_folder, &prop);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "edit folder - not enough memory"));
    goto no_quit;
  }
  
  if (type == NEW_FOLDER) {
    r = mailfolder_add_child(folder, new_folder);
    if (r != MAIL_NO_ERROR) {
      ETPAN_APP_LOG((app->app, "edit folder - not enough memory"));
      mailfolder_free(new_folder);
      goto no_quit;
    }
  }
  else {
    uint32_t sibling_index;
    struct mailfolder * parent;
    carray * children;
    
    sibling_index = folder->fld_sibling_index;
    parent = folder->fld_parent;
    
    /* reparent */
    new_folder->fld_parent = parent;
    new_folder->fld_sibling_index = sibling_index;
    carray_set(parent->fld_children, sibling_index, new_folder);
    
    /* detach from parent */
    folder->fld_parent = NULL;
    
    /* set children */
    children = folder->fld_children;
    folder->fld_children = new_folder->fld_children;
    new_folder->fld_children = children;
    for(i = 0 ; i < carray_count(children) ; i ++) {
      struct mailfolder * child;
      
      child = carray_get(children, i);
      child->fld_parent = new_folder;
    }
    
    /* change references the folder */
    
    if (folder == state->config.vfolder_config->sent_folder)
      state->config.vfolder_config->sent_folder = new_folder;
    if (folder == state->config.vfolder_config->draft_folder)
      state->config.vfolder_config->draft_folder = new_folder;
    if (folder == state->config.vfolder_config->default_folder)
      state->config.vfolder_config->default_folder = new_folder;
    
    /* change references on sent folder */
    recursive_folder_replace_properties(app, state->config.vfolder_config,
        state->config.vfolder_config->root, folder, new_folder);
    
    /* handle case where the sent folder is itself */
    if (prop.sent_folder == folder) {
      prop.sent_folder = new_folder;
      prop.draft_folder = new_folder;
    }
    
    etpan_cfg_vfolder_remove_prop(state->config.vfolder_config, folder);
    
    /* destroy folder */
    mailfolder_free(folder);
  }

  r = etpan_cfg_vfolder_add_prop(state->config.vfolder_config, new_folder);
  etpan_vfolder_set_account(state->config.vfolder_config, new_folder,
      prop.account);
  etpan_vfolder_set_sent_folder(state->config.vfolder_config, new_folder,
      prop.sent_folder);
  etpan_vfolder_set_draft_folder(state->config.vfolder_config, new_folder,
      prop.draft_folder);
  etpan_vfolder_set_mail_to(state->config.vfolder_config, new_folder,
      prop.mail_to);
  etpan_vfolder_set_poll(state->config.vfolder_config, new_folder, prop.poll);
  etpan_vfolder_set_max(state->config.vfolder_config, new_folder, prop.max);
  
  set_root_folder(app);
  
  for(i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    struct mailfolder * cur_folder;
    
    cur_folder = carray_get(state->common_state.folder_tab, i);
    
    if (cur_folder == new_folder)
      state->common_state.selected = i;
  }
  
  etpan_app_quit_subapp(folder_editor_app);
  
 no_quit:
  return;
}


static int edit_folder(struct etpan_subapp * app, int do_add)
{
  struct etpan_subapp * edit_app;
  struct mailfolder * folder;
  struct app_state * state;
  struct folder_edit_data * upcall_data;
  int type;
  
  state = app->data;
  
  edit_app = etpan_app_find_subapp(app->app, "folder-edit",
      0, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_folder_editor_app_new(app->app);
  if (edit_app == NULL)
    return ERROR_MEMORY;

  etpan_subapp_set_parent(edit_app, app);
  
  if (do_add) {
    folder = NULL;
    
    type = NEW_FOLDER;
  }
  else {
    folder = etpan_folder_common_get_selected_folder(&state->common_state);
    if (folder->fld_parent == NULL) {
      ETPAN_APP_LOG((app->app, "edit folder - can't edit root folder"));
      return ERROR_INVAL;
    }
    
    type = MODIFY_FOLDER;
  }
  
  upcall_data = malloc(sizeof(* upcall_data));
  if (upcall_data == NULL)
    return ERROR_MEMORY;
  
  upcall_data->app = app;
  upcall_data->type = type;
  
  etpan_folder_editor_app_set(edit_app, &state->config,
      folder, folder_edit_upcall, upcall_data);
  
  etpan_app_switch_subapp(edit_app, 0);

  return NO_ERROR;
}

static int delete_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (carray_count(folder->fld_children) != 0) {
    ETPAN_APP_LOG((app->app, "edit folder - folder has children"));
    return ERROR_INVAL;
  }
  
  if (folder->fld_parent == NULL) {
    ETPAN_APP_LOG((app->app, "edit folder - can't delete root folder"));
    return ERROR_INVAL;
  }
  
  if (folder == state->config.vfolder_config->sent_folder)
    state->config.vfolder_config->sent_folder = NULL;
  if (folder == state->config.vfolder_config->draft_folder)
    state->config.vfolder_config->draft_folder = NULL;
  if (folder == state->config.vfolder_config->default_folder)
    state->config.vfolder_config->default_folder = NULL;
  
  /* change references on folder */
  recursive_folder_replace_properties(app, state->config.vfolder_config,
      state->config.vfolder_config->root, folder, NULL);
  
  etpan_cfg_vfolder_remove_prop(state->config.vfolder_config, folder);
  
  mailfolder_detach_parent(folder);
  mailfolder_free(folder);
  
  set_root_folder(app);

  return NO_ERROR;
}

void etpan_folder_conf_app_write_config(struct etpan_subapp * app)
{
  struct app_state * state;
  
  state = app->data;

  if (state->config_init) {
    etpan_config_write(app->app, &state->config);
  }
  else {
    ETPAN_APP_DEBUG((app->app, "BUG detected, trying to write flushed configuration"));
  }
}

static int set_default_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  state->config.vfolder_config->default_folder = folder;
  
  set_root_folder(app);
  
  return NO_ERROR;
}

static int set_default_sent_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  state->config.vfolder_config->sent_folder = folder;
  
  set_root_folder(app);
  
  return NO_ERROR;
}

static int set_default_draft_folder(struct etpan_subapp * app)
{
  struct app_state * state;
  struct mailfolder * folder;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  state->config.vfolder_config->draft_folder = folder;
  
  set_root_folder(app);
  
  return NO_ERROR;
}

#if 0
static int move_node(struct etpan_subapp * app, int direction)
{
  struct app_state * state;
  struct mailfolder * folder;
  struct mailfolder * parent;
  struct mailfolder * exchange;
  uint32_t index;
  uint32_t i;
  
  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  parent = folder->parent;
  if (parent == NULL)
    return ERROR_INVAL;
  
  index = folder->sibling_index;
  
  if (direction) {
    if (index == carray_count(parent->children) - 1)
      return ERROR_INVAL;
    
    exchange = carray_get(parent->children, index + 1);
    carray_set(parent->children, index + 1, folder);
    folder->sibling_index = index + 1;
    carray_set(parent->children, index, exchange);
    exchange->sibling_index = index;
  }
  else {
    if (index == 0)
      return ERROR_INVAL;
    
    exchange = carray_get(parent->children, index - 1);
    carray_set(parent->children, index - 1, folder);
    folder->sibling_index = index - 1;
    carray_set(parent->children, index, exchange);
    exchange->sibling_index = index;
  }
  
  set_root_folder(app);

  for(i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    struct mailfolder * cur_folder;
    
    cur_folder = carray_get(state->common_state.folder_tab, i);
    
    if (cur_folder == folder)
      state->common_state.selected = i;
  }
  
  return NO_ERROR;
}
#endif


static int move_node(struct etpan_subapp * app, int direction)
{
  struct app_state * state;
  struct mailfolder * folder;
  struct mailfolder * parent;
  struct mailfolder * exchange;
  unsigned int index;
  unsigned int i; 

  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  parent = folder->fld_parent;
  if (parent == NULL)
    return ERROR_INVAL;
  
  index = folder->fld_sibling_index;
  
  if (direction) {
    if (index == carray_count(parent->fld_children) - 1)
      return ERROR_INVAL;
    
    exchange = carray_get(parent->fld_children, index + 1);
    carray_set(parent->fld_children, index + 1, folder);
    folder->fld_sibling_index = index + 1;
    carray_set(parent->fld_children, index, exchange);
    exchange->fld_sibling_index = index;
  }
  else {
    if (index == 0)
      return ERROR_INVAL;
    
    exchange = carray_get(parent->fld_children, index - 1);
    carray_set(parent->fld_children, index - 1, folder);
    folder->fld_sibling_index = index - 1;
    carray_set(parent->fld_children, index, exchange);
    exchange->fld_sibling_index = index;
  }

  set_root_folder(app);

  for(i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    struct mailfolder * cur_folder;
    
    cur_folder = carray_get(state->common_state.folder_tab, i);
    
    if (cur_folder == folder)
      state->common_state.selected = i;
  }
  
  return NO_ERROR;
}

static int reparent_node(struct etpan_subapp * app, int direction)
{
  struct app_state * state;
  struct mailfolder * folder;
  struct mailfolder * parent;
  unsigned int i;
  int r;

  state = app->data;
  
  folder = etpan_folder_common_get_selected_folder(&state->common_state);
  if (folder == NULL)
    return ERROR_INVAL;
  
  parent = folder->fld_parent;
  if (parent == NULL)
    return ERROR_INVAL;

  if (direction) {
    unsigned int index;
    unsigned int child_count;
    struct mailfolder * after_folder;
    
    index = folder->fld_sibling_index;
  
    if (index == carray_count(parent->fld_children) - 1)
      return ERROR_INVAL;
    
    after_folder = carray_get(parent->fld_children, index + 1);
    
    child_count = carray_count(after_folder->fld_children);
    r = carray_set_size(after_folder->fld_children, child_count + 1);
    if (r < 0)
      return ERROR_MEMORY;
    
    mailfolder_detach_parent(folder);
    if (child_count >= 1) {
      i = child_count - 1;
      while (1) {
        struct mailfolder * cur_folder;
        
        cur_folder = carray_get(after_folder->fld_children, i);
        carray_set(after_folder->fld_children, i + 1, cur_folder);
        cur_folder->fld_sibling_index = i + 1;
        
        if (i == 0)
          break;
        
        i --;
      }
    }
    carray_set(after_folder->fld_children, 0, folder);
      
    folder->fld_parent = after_folder;
    folder->fld_sibling_index = 0;
  }
  else {
    unsigned int index;
    unsigned int child_count;
    struct mailfolder * parent_parent;
    unsigned int parent_index;
    
    index = folder->fld_sibling_index;
    
    parent_parent = parent->fld_parent;
    if (parent_parent == NULL)
      return ERROR_INVAL;
    
    parent_index = parent->fld_sibling_index;
    
    child_count = carray_count(parent_parent->fld_children);
    r = carray_set_size(parent_parent->fld_children, child_count + 1);
    if (r < 0)
      return ERROR_MEMORY;
    
    mailfolder_detach_parent(folder);
    if (child_count >= 1) {
      i = child_count - 1;
      while (1) {
        struct mailfolder * cur_folder;
        
        cur_folder = carray_get(parent_parent->fld_children, i);
        carray_set(parent_parent->fld_children, i + 1, cur_folder);
        cur_folder->fld_sibling_index = i + 1;
        
        if (i == parent_index)
          break;
        
        i --;
      }
    }
    carray_set(parent_parent->fld_children, parent_index, folder);
    folder->fld_parent = parent_parent;
    folder->fld_sibling_index = parent_index;
  }
  
  set_root_folder(app);

  for(i = 0 ; i < carray_count(state->common_state.folder_tab) ; i ++) {
    struct mailfolder * cur_folder;
    
    cur_folder = carray_get(state->common_state.folder_tab, i);
    
    if (cur_folder == folder)
      state->common_state.selected = i;
  }
  
  return NO_ERROR;
}

static void global_conf_upcall(struct etpan_subapp * edit_app,
    int valid, void * data)
{
  int r;
  struct etpan_subapp * app;
  struct app_state * state;
  int network_delay;

  if (valid == ETPAN_CFG_GLOBAL_EDIT_CANCEL) {
    etpan_app_quit_subapp(edit_app);
    return;
  }
  
  app = data;
  state = app->data;

  ETPAN_APP_DEBUG((app->app, "conf upcall ok"));
  
  r = etpan_cfg_global_edit_app_get(edit_app,
      state->config.global_config);
  /* ignore error */
  
  network_delay = state->config.global_config->network_timeout;
  if ((network_delay == -1) || (network_delay != 0))
    network_delay = ETPAN_DEFAULT_NETWORK_TIMEOUT;
  
  mailstream_network_delay.tv_sec = network_delay;
  mailstream_network_delay.tv_usec = 0;
  
  etpan_app_quit_subapp(edit_app);
}

static int global_conf(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * edit_app;
  
  state = app->data;

  edit_app = etpan_app_find_subapp(app->app, "cfg-global-edit",
      0, NULL, NULL);
  if (edit_app == NULL)
    edit_app = etpan_cfg_global_edit_app_new(app->app);
  if (edit_app == NULL)
    return ERROR_MEMORY;
  
  etpan_cfg_global_edit_app_set(edit_app, state->config.global_config,
      global_conf_upcall, app);

  etpan_subapp_set_parent(edit_app, app);
  etpan_app_switch_subapp(edit_app, 0);
  
  return NO_ERROR;
}

#if 0
static void thread_stop_storage_cancel(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;

  state = app->data;
  state->state = STATE_NORMAL;
}

static void thread_stop_storage_callback(struct etpan_subapp * app,
    struct etpan_thread_op * op, int app_op_type, void * data)
{
  struct app_state * state;

  state = app->data;
  state->state = STATE_NORMAL;
  
  if (state->upcall != NULL) {
    if (state->valid_conf)
      state->upcall(app, ETPAN_FOLDER_CONF_VALID, state->upcall_data);
    else
      state->upcall(app, ETPAN_FOLDER_CONF_CANCEL, state->upcall_data);
  }
}
#endif

static int validate_conf(struct etpan_subapp * app, int valid)
{
  struct app_state * state;
  
  state = app->data;
  state->valid_conf = valid;
  
  if (valid) {
    /* write config */
    etpan_folder_conf_app_write_config(app);
  }

#if 0  
  state->state = STATE_LOADING_NEW_CONFIG;
#endif

  if (state->upcall != NULL) {
    if (state->valid_conf)
      state->upcall(app, ETPAN_FOLDER_CONF_VALID, state->upcall_data);
    else
      state->upcall(app, ETPAN_FOLDER_CONF_CANCEL, state->upcall_data);
  }

#if 0  
  ETPAN_APP_LOG((app->app, "disconnecting storage ..."));
  ETPAN_APP_DEBUG((app->app, "waiting for thread to finish"));
  r = etpan_subapp_thread_op_add(app,
      THREAD_ID_FOLDERCONF_STOP_STORAGE_THREAD,
      ETPAN_THREAD_STOP_STORAGE_THREAD,
      NULL, NULL,
      NULL, NULL, NULL,
      thread_stop_storage_callback, NULL,
      thread_stop_storage_cancel);
  if (r != NO_ERROR) {
    ETPAN_APP_LOG((app->app, "disconnecting storage - not enough memory"));
    state->state = STATE_NORMAL;
    return ERROR_MEMORY;
  }
#endif
  
  return NO_ERROR;
}


static void ask_quit_upcall(struct etpan_subapp * input_app,
    int valid, void * data)
{
  struct app_state * state;
  char * name;
  char * dup_name;
  struct etpan_subapp * app;
  int r;
  char * result;
  int do_quit;

  app = data;
  state = app->data;
  
  if (valid == ETPAN_INPUT_COMMON_CANCEL) {
    etpan_app_quit_subapp(input_app);
    return;
  }
  
  result = etpan_search_input_get_value(input_app);
  if ((* result != 'y') && (* result != 'n')) {
    etpan_app_quit_subapp(input_app);
    return;
  }

  do_quit = (* result == 'y');
  etpan_app_quit_subapp(input_app);
  
  if (do_quit)
    validate_conf(app, 1);
  else
    validate_conf(app, 0);
}


static void ask_quit(struct etpan_subapp * app)
{
  struct app_state * state;
  struct etpan_subapp * input;
  int r;
  
  input = etpan_app_find_subapp(app->app, "search-input",
      0, NULL, NULL);
  if (input == NULL) {
    input = etpan_search_input_new(app->app);
    if (input == NULL)
      goto err;
  }
  
  r = etpan_search_input_set(input,
      "save the configuration (y/n) ? ", 1,
      NULL, 0,
      ask_quit_upcall, app);
  if (input == NULL)
    goto err;
  
  etpan_subapp_set_parent(input, app);
  etpan_app_switch_subapp(input, 0);
  
  return;
  
 err:
  ETPAN_APP_LOG((app->app, "edit folder - not enough memory"));
}


#define HELP_TEXT \
"\
Help for folder tree configuration\n\
----------------------------------\n\
\n\
This application will let you configure your folder tree.\n\
\n\
normal mode :\n\
\n\
- arrow keys : move cursor\n\
\n\
- e or\n\
  [Enter]    : edit folder properties\n\
- a          : add new folder\n\
- d          : delete folder\n\
- m          : go into move mode\n\
\n\
- g          : edit global configuration\n\
\n\
- s          : set default folder\n\
- Shift-S    : set default sent folder\n\
- p          : set default draft folder\n\
\n\
- y          : edition of folder tree is finished\n\
- Ctrl-G     : cancel\n\
\n\
- ?          : help\n\
- Ctrl-L     : Console log\n\
\n\
move mode :\n\
\n\
- up, down\n\
  arrow keys : move node up & down\n\
- left, right\n\
  arrow keys : change level of node in the tree\n\
- [Enter]    : confirm position\n\
\n\
(? or q to exit help)\n\
"

static int show_help(struct etpan_subapp * app)
{
  return etpan_show_help(app, HELP_TEXT, sizeof(HELP_TEXT) - 1);
}
