/* ethos-mono-plugin-loader.c
 *
 * Copyright (C) 2009 Christian Hergert <chris@dronelabs.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston MA 
 * 02110-1301 USA
 */

#include <gmodule.h>
#include <mono/jit/jit.h>
#include <mono/metadata/assembly.h>
#include <mono/metadata/mono-config.h>

#include <ethos/ethos-error.h>
#include <ethos/ethos-plugin-loader.h>
#include <ethos/ethos-plugin-info-private.h>

#include "ethos-mono-plugin-loader.h"

struct _EthosMonoPluginLoaderPrivate
{
	MonoDomain *domain;
	gboolean    initialized;
};

static G_CONST_RETURN gchar*
ethos_mono_plugin_loader_get_name (EthosPluginLoader *plugin_loader)
{
	return "mono";
}

static void
ethos_mono_plugin_loader_gc (EthosPluginLoader *plugin_loader)
{
}

static void
ethos_mono_plugin_loader_initialize (EthosPluginLoader *plugin_loader,
                                     EthosManager      *manager)
{
	EthosMonoPluginLoaderPrivate *priv;
	MonoDomain                   *domain;

	g_return_if_fail (ETHOS_IS_MONO_PLUGIN_LOADER (plugin_loader));

	priv = ETHOS_MONO_PLUGIN_LOADER (plugin_loader)->priv;

	if (!(domain = mono_jit_init (ETHOS_MONO_DLL))) {
		g_warning ("Could not initialize mono plugins");
		return;
	}

	mono_config_parse (NULL);

	priv->initialized = TRUE;
	priv->domain = domain;
}

static void
ethos_mono_plugin_loader_unload (EthosPluginLoader *plugin_loader)
{
	EthosMonoPluginLoaderPrivate *priv;

	priv = ETHOS_MONO_PLUGIN_LOADER (plugin_loader)->priv;

	if (!priv->initialized)
		return;

	mono_jit_cleanup (priv->domain);
	priv->domain      = NULL;
	priv->initialized = FALSE;
}

static EthosPlugin*
ethos_mono_plugin_loader_load (EthosPluginLoader  *plugin_loader,
                               EthosPluginInfo    *plugin_info,
                               GError            **error)
{
	EthosMonoPluginLoaderPrivate *priv;

	gchar       *name;
	gchar       *plugin_dir;
	gchar       *plugin_path;
	EthosPlugin *plugin = NULL;

	g_return_val_if_fail (ETHOS_IS_PLUGIN_LOADER (plugin_loader), NULL);
	g_return_val_if_fail (ETHOS_IS_PLUGIN_INFO (plugin_info), NULL);
	g_return_val_if_fail (g_module_supported (), NULL);

	priv = ETHOS_MONO_PLUGIN_LOADER (plugin_loader)->priv;

	plugin_dir = g_path_get_dirname (ethos_plugin_info_get_filename (plugin_info));
	name = g_strconcat (ethos_plugin_info_get_module (plugin_info), ".dll", NULL);
	plugin_path = g_build_filename (plugin_dir, name, NULL);

	/* load the assembly, get the plugin instance */
	if (!g_file_test (plugin_path, G_FILE_TEST_IS_REGULAR)) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_FILE_NOT_FOUND,
		             "Could not locate plugin library \"%s\"",
		             plugin_path);
		goto cleanup;
	}

	MonoAssembly *assembly = NULL;
	MonoImage    *image    = NULL;

	if (!(assembly = mono_domain_assembly_open (priv->domain, plugin_path))) {
		g_set_error (error, ETHOS_ERROR, ETHOS_ERROR_PLUGIN_LOADER,
		             "Could not load the plugin assembly");
		goto cleanup;
	}

	image = mono_assembly_get_image (assembly);

	g_debug ("We loaded the assembly!");

cleanup:
	g_free (plugin_path);
	return plugin;
}

static void
ethos_mono_plugin_loader_base_init (EthosPluginLoaderIface *iface)
{
	iface->get_name   = ethos_mono_plugin_loader_get_name;
	iface->gc         = ethos_mono_plugin_loader_gc;
	iface->initialize = ethos_mono_plugin_loader_initialize;
	iface->unload     = ethos_mono_plugin_loader_unload;
	iface->load       = ethos_mono_plugin_loader_load;
}

G_DEFINE_TYPE_EXTENDED (EthosMonoPluginLoader,
                        ethos_mono_plugin_loader,
                        G_TYPE_OBJECT,
                        0,
                        G_IMPLEMENT_INTERFACE (ETHOS_TYPE_PLUGIN_LOADER,
                                               ethos_mono_plugin_loader_base_init));

static void
ethos_mono_plugin_loader_finalize (GObject *object)
{
	G_OBJECT_CLASS (ethos_mono_plugin_loader_parent_class)->finalize (object);
}

static void
ethos_mono_plugin_loader_class_init (EthosMonoPluginLoaderClass *klass)
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS (klass);
	object_class->finalize = ethos_mono_plugin_loader_finalize;
	g_type_class_add_private (object_class, sizeof (EthosMonoPluginLoaderPrivate));
}

static void
ethos_mono_plugin_loader_init (EthosMonoPluginLoader *plugin_loader)
{
	plugin_loader->priv = G_TYPE_INSTANCE_GET_PRIVATE (plugin_loader,
	                                                   ETHOS_TYPE_MONO_PLUGIN_LOADER,
	                                                   EthosMonoPluginLoaderPrivate);
}

EthosPluginLoader*
ethos_mono_plugin_loader_new ()
{
	return g_object_new (ETHOS_TYPE_MONO_PLUGIN_LOADER, NULL);
}

G_MODULE_EXPORT EthosPluginLoader*
ethos_plugin_loader_register (void)
{
	return ethos_mono_plugin_loader_new ();
}
