﻿/* ****************************************************************************
 *
 * Copyright (c) Microsoft Corporation. 
 *
 * This source code is subject to terms and conditions of the Microsoft Public License. A 
 * copy of the license can be found in the License.html file at the root of this distribution. If 
 * you cannot locate the  Microsoft Public License, please send an email to 
 * dlr@microsoft.com. By using this source code in any fashion, you are agreeing to be bound 
 * by the terms of the Microsoft Public License.
 *
 * You must not remove this notice, or any other, from this software.
 *
 *
 * ***************************************************************************/

using System;
using System.Linq.Expressions;
using System.Reflection;

using Microsoft.Scripting;
using Microsoft.Scripting.Runtime;
using Microsoft.Scripting.Utils;

using IronPython.Runtime;
using IronPython.Runtime.Operations;

namespace IronPython.Compiler {
    /// <summary>
    /// Small reducable node which just fetches the value from a ClosureCell
    /// object.  Like w/ global variables the compiler recognizes these on 
    /// sets and turns them into assignments on the python global object.
    /// </summary>
    class ClosureExpression : Expression, IPythonVariableExpression {
        private readonly Expression/*!*/ _closureCell;
        private readonly Expression _parameter;
        private readonly Ast.PythonVariable/*!*/ _variable;
        private static readonly FieldInfo _cellField = typeof(ClosureCell).GetField("Value");

        public ClosureExpression(Ast.PythonVariable/*!*/ variable, Expression/*!*/ closureCell, Expression parameter) {
            Assert.NotNull(closureCell);

            _variable = variable;
            _closureCell = closureCell;
            _parameter = parameter;
        }

        #region ClosureExpression Public API

        /// <summary>
        /// Gets the expression which points at the closure cell.
        /// </summary>
        public Expression/*!*/ ClosureCell {
            get {
                return _closureCell;
            }
        }

        /// <summary>
        /// The original expression for the incoming parameter if this is a parameter closure.  Otherwise
        /// the value is null.
        /// </summary>
        public Expression OriginalParameter {
            get {
                return _parameter;
            }
        }

        /// <summary>
        /// Gets the PythonVariable for which this closure expression was created.
        /// </summary>
        public Ast.PythonVariable/*!*/ PythonVariable {
            get {
                return _variable;
            }
        }

        /// <summary>
        /// Creates the storage for the closure cell.  If this is a closure over a parameter it
        /// captures the initial incoming parameter value.
        /// </summary>
        public Expression/*!*/ Create() {
            if (OriginalParameter != null) {
                return Expression.Assign(_closureCell, Expression.Call(typeof(PythonOps).GetMethod("MakeClosureCellWithValue"), OriginalParameter));
            }
            return Expression.Assign(_closureCell, Expression.Call(typeof(PythonOps).GetMethod("MakeClosureCell")));
        }

        #endregion

        #region Expression overrides

        public sealed override ExpressionType NodeType {
            get { return ExpressionType.Extension; }
        }

        public sealed override Type/*!*/ Type {
            get { return typeof(object); }
        }

        public override bool CanReduce {
            get {
                return true;
            }
        }

        public SymbolId Name {
            get {
                return _variable.Name;
            }
        }

        /// <summary>
        /// Reduces the closure cell to a read of the value stored in the cell.
        /// </summary>
        public override Expression/*!*/ Reduce() {
            return Expression.Field(
                _closureCell,
                _cellField
            );
        }

        #endregion

        #region IPythonVariableExpression implementation

        /// <summary>
        /// Assigns a value to the closure cell.
        /// </summary>
        public Expression/*!*/ Assign(Expression/*!*/ value) {
            return Expression.Assign(
                Expression.Field(_closureCell, _cellField),
                value
            );
        }

        /// <summary>
        /// Removes the current value from the closure cell.
        /// </summary>
        public Expression/*!*/ Delete() {
            return Expression.Assign(
                Expression.Field(_closureCell, _cellField),
                Expression.Field(null, typeof(Uninitialized).GetField("Instance"))
            );
        }

        #endregion
    }
}
