/*
Copyright (c) 2009, Regents of the University of Alaska

All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

    * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
    * Neither the name of the Geographic Information Network of Alaska nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

This code was developed by Dan Stahlke for the Geographic Information Network of Alaska.
*/



#include "mask.h"
#include "mask-tracer.h"
#include "common.h"
#include "polygon.h"
#include "polygon-rasterizer.h"

#define DIR_UP 0
#define DIR_RT 1
#define DIR_DN 2
#define DIR_LF 3

typedef int pixquad_t;

int dbg_idx = 0;
static void debug_write_mask(const BitGrid mask, size_t w, size_t h) {
	char fn[1000];
	sprintf(fn, "zz-debug-%04d.pgm", dbg_idx++);

	uint8_t *row = new uint8_t[w];

	FILE *fh = fopen(fn, "w");
	if(!fh) fatal_error("cannot open %s", fn);
	fprintf(fh, "P5\n%zd %zd\n255\n", w, h);
	for(size_t y=0; y<h; y++) {
		for(size_t x=0; x<w; x++) {
			row[x] = mask(x, y) ? 255 : 0;
		}
		fwrite(row, w, 1, fh);
	}
	fclose(fh);

	delete[] row;
}

static ring_t *make_enclosing_ring(size_t w, size_t h) {
	ring_t *ring = MYALLOC(ring_t, 1);
	ring->npts = 4;
	ring->pts = MYALLOC(vertex_t, ring->npts);
	ring->pts[0].x = -1;
	ring->pts[0].y = -1;
	ring->pts[1].x = w;
	ring->pts[1].y = -1;
	ring->pts[2].x = w;
	ring->pts[2].y = h;
	ring->pts[3].x = -1;
	ring->pts[3].y = h;
	return ring;
}

static long compute_area(row_crossings_t *crossings, int num_rows) {
	long area = 0;
	for(int y=0; y<num_rows; y++) {
		int nc = crossings[y].num_crossings;
		int *rc = crossings[y].crossings;
		for(int cidx=0; cidx<nc/2; cidx++) {
			int from = rc[cidx*2  ];
			int to   = rc[cidx*2+1];
			area += to - from;
		}
	}
	return area;
}

/*
static int is_inside_crossings(row_crossings_t *c, int x) {
	int inside = 0;
	for(int i=0; i<c->num_crossings; i++) {
		if(x >= c->crossings[i]) inside = !inside;
	}
	return inside;
}
*/

static inline pixquad_t get_quad(const BitGrid &mask, int x, int y, bool select_color) {
	// 1 2
	// 8 4
	pixquad_t quad =
		(mask(x-1, y-1) ? 1 : 0) +
		(mask(x  , y-1) ? 2 : 0) +
		(mask(x  , y  ) ? 4 : 0) +
		(mask(x-1, y  ) ? 8 : 0);
	if(!select_color) quad ^= 0xf;
	return quad;
}

static inline pixquad_t rotate_quad(pixquad_t q, int dir) {
	return ((q + (q<<4)) >> dir) & 0xf;
}

static ring_t trace_single_mpoly(const BitGrid mask, size_t w, size_t h,
int initial_x, int initial_y, bool select_color) {
	//printf("trace_single_mpoly enter (%d,%d)\n", initial_x, initial_y);

	ring_t ring;
	int ringbuf_size = 4;
	ring.pts = MYALLOC(vertex_t, ringbuf_size);
	ring.pts[0].x = initial_x;
	ring.pts[0].y = initial_y;
	ring.npts = 1;

	int x = initial_x;
	int y = initial_y;
	pixquad_t quad = get_quad(mask, x, y, select_color);
	int dir;
	for(dir=0; dir<4; dir++) {
		pixquad_t rq = rotate_quad(quad, dir);
		if((rq & 3) == 2) break;
	}
	if(dir == 4) fatal_error("couldn't choose a starting direction (q=%d)", quad);
	for(;;) {
		//printf("xy=(%d,%d)\n", x, y);

		switch(dir) {
			case DIR_UP: y -= 1; break;
			case DIR_RT: x += 1; break;
			case DIR_DN: y += 1; break;
			case DIR_LF: x -= 1; break;
			default: fatal_error("bad direction");
		}
		if(x == initial_x && y == initial_y) break;
		if(x<0 || y<0 || x>(int)w || y>(int)h) fatal_error("fell off edge (%d,%d)", x, y);
		pixquad_t quad = get_quad(mask, x, y, select_color);
		quad = rotate_quad(quad, dir);
		if((quad & 12) != 4) fatal_error("tracer was not on the right side of things (%d)", quad);
		int rot;
		switch(quad & 3) {
			case 0: rot =  1; break; // N N
			case 1: rot =  1; break; // Y N
			case 2: rot =  0; break; // N Y
			case 3: rot = -1; break; // Y Y
			default: fatal_error("not possible");
		}
		dir = (dir + rot + 4) % 4;

		if(rot) {
			if(ring.npts == ringbuf_size) {
				if(ringbuf_size) ringbuf_size *= 2;
				else ringbuf_size = 4;
				ring.pts = REMYALLOC(vertex_t, ring.pts, ringbuf_size);
			}
			ring.pts[ring.npts].x = x;
			ring.pts[ring.npts].y = y;
			ring.npts++;
		}
	}

	if(ringbuf_size > ring.npts) {
		ring.pts = REMYALLOC(vertex_t, ring.pts, ring.npts);
	}

	return ring;
}

static int recursive_trace(BitGrid mask, size_t w, size_t h,
ring_t *bounds, int depth, mpoly_t *out_poly, int parent_id, 
long min_area, int no_donuts) {
	//printf("recursive_trace enter: depth=%d\n", depth);

	bool select_color = !(depth & 1);

	int must_free_bounds = 0;
	if(!bounds) {
		bounds = make_enclosing_ring(w, h);
		must_free_bounds = 1;
	}

	int bound_left, bound_right;
	int bound_top, bound_bottom;
	bound_left = bound_right = (int)bounds->pts[0].x;
	bound_top = bound_bottom = (int)bounds->pts[0].y;
	for(int v_idx=0; v_idx<bounds->npts; v_idx++) {
		int x = (int)bounds->pts[v_idx].x;
		int y = (int)bounds->pts[v_idx].y;
		if(x < bound_left) bound_left = x;
		if(x > bound_right) bound_right = x;
		if(y < bound_top) bound_top = y;
		if(y > bound_bottom) bound_bottom = y;
	}

	mpoly_t bounds_mp;
	bounds_mp.num_rings = 1;
	bounds_mp.rings = bounds;

	row_crossings_t *crossings = get_row_crossings(&bounds_mp, bound_top, bound_bottom-bound_top);
	int skip_this = min_area && (compute_area(crossings, bound_bottom-bound_top) < min_area);
	int skip_child = skip_this || (depth && no_donuts);

	row_crossings_t cross_both;
	cross_both.array_size = 0;
	cross_both.crossings = NULL;

	if(!depth) {
		printf("Tracing: ");
		GDALTermProgress(0, NULL, NULL);
	}

	if(!skip_child) {
		for(int y=bound_top+1; y<bound_bottom; y++) {
			if(!depth) {
				GDALTermProgress((double)y/(double)(bound_bottom-bound_top-1), NULL, NULL);
			}

			// make sure the range (y-1,y)*(x-1,x) is in bounds
			crossings_intersection(&cross_both, 
				crossings + (y-bound_top-1),
				crossings + (y-bound_top));
			for(int cidx=0; cidx<cross_both.num_crossings/2; cidx++) {
				// make sure the range (y-1,y)*(x-1,x) is in bounds
				int from = 1+cross_both.crossings[cidx*2  ];
				int to   =   cross_both.crossings[cidx*2+1];

				for(int x=from; x<to; x++) {
					//pixquad_t quad = get_quad(mask, x, y, select_color);
					//int is_seed = (quad != 0 && quad != 0xf);
					int is_seed;
					if(select_color) {
						is_seed = 
							mask(x-1, y-1) ||
							mask(x  , y-1) ||
							mask(x-1, y  ) ||
							mask(x  , y  );
					} else {
						is_seed = 
							(!mask(x-1, y-1)) ||
							(!mask(x  , y-1)) ||
							(!mask(x-1, y  )) ||
							(!mask(x  , y  ));
					}

					if(is_seed) {
						ring_t r = trace_single_mpoly(mask, w, h, x, y, select_color);

						r.parent_id = parent_id;
						r.is_hole = depth % 2;
						//r.parent_id = -1;
						//r.is_hole = 0;

						out_poly->rings = REMYALLOC(ring_t, out_poly->rings, (out_poly->num_rings + 1));
						int outer_ring_id = (out_poly->num_rings++);
						out_poly->rings[outer_ring_id] = r;

						int was_skip = recursive_trace(
							mask, w, h, &r, depth+1, out_poly, outer_ring_id,
							min_area, no_donuts);
						if(was_skip) {
							free(r.pts);
							out_poly->num_rings--;
						}
					}
				} 
			}
		}
		if(cross_both.crossings != NULL) free(cross_both.crossings);
	}

	for(int y=bound_top; y<bound_bottom; y++) {
		row_crossings_t *r = crossings + (y-bound_top);
		if(depth>0) {
			for(int cidx=0; cidx<r->num_crossings/2; cidx++) {
				int from = r->crossings[cidx*2  ];
				int to   = r->crossings[cidx*2+1];
				for(int x=from; x<=to; x++) {
					if(x>=0 && y>=0 && size_t(x)<w && size_t(y)<h) {
						mask.set(x, y, select_color);
					}
				}
			}
		}
	}
	free_row_crossings(crossings, bound_bottom-bound_top);

	if(must_free_bounds) {
		free(bounds->pts);
		free(bounds);
	}

	if(VERBOSE >= 4) debug_write_mask(mask, w, h);

	if(!depth) {
		GDALTermProgress(1, NULL, NULL);
	}

	return skip_this;
}

// this function has the side effect of erasing the mask
mpoly_t trace_mask(BitGrid mask, size_t w, size_t h, long min_area, int no_donuts) {
	if(VERBOSE >= 4) debug_write_mask(mask, w, h);

	mpoly_t out_poly;

	out_poly.num_rings = 0;
	out_poly.rings = NULL;

	recursive_trace(mask, w, h, NULL, 0, &out_poly, -1, min_area, no_donuts);
	printf("Trace found %d rings.\n", out_poly.num_rings);

	//free(mask_8bit);

	//fatal_error("OK");

	return out_poly;
}
