/* disks.c (cpudyn) - Functions to put a disk in stand by            */
/*          - by Ricardo Galli (C) 2003      -- Licence GPL          */
/* Some code extracted from hdparm v 5.3 by Mark Lord                */

#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <endian.h>
#include <sys/ioctl.h>
#include <sys/shm.h>
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <sys/time.h>
#include <sys/times.h>
#include <sys/types.h>
#include <sys/mount.h>
#include <linux/types.h>
#include <linux/hdreg.h>
#include <linux/major.h>
#include <asm/byteorder.h>
#include <fcntl.h>


#include "cpudynd.h"
#include "disks.h"


static device_t *devices;
static get_opts_t get_device_t_ops;
static char *proc_stat_dir = NULL;

extern unsigned standby_timeout;

int is_active(device_t *dev)
{
	unsigned char args[4] = {WIN_CHECKPOWERMODE1,0,0,0};
	unsigned state;

	if (ioctl(dev->fd, HDIO_DRIVE_CMD, &args)
		&& (args[0] = WIN_CHECKPOWERMODE2) /* try again with 0x98 */
		&& ioctl(dev->fd, HDIO_DRIVE_CMD, &args)) {
		if (errno != EIO || args[0] != 0 || args[1] != 0)
			state = 1;
		else {
			state = 0; // it's sleeping
   			errprintf("Drive state is:  sleeping\n");
		}
	} else {
		state = (args[2] == 255) ? 1 : 0; // 1=active, 0=standby
   		errprintf("Drive state is:  %s\n", state==1 ? "active" : "standby");
	}
	return state;
}
		
int flush_buffer_cache (device_t *dev)
{
	int res;
	res = fsync (dev->fd);				  /* flush buffers */
	if (ioctl(dev->fd, BLKFLSBUF, NULL)) { /* do it again, big time */
		errprintf("BLKFLSBUF failed");
		return -1;
	}
	if (ioctl(dev->fd, HDIO_DRIVE_CMD, NULL) && errno != EINVAL) /* await completion */
		            errprintf("HDIO_DRIVE_CMD(null) (wait for flush complete) failed\n");
	dev->flushed = time(NULL);
	errprintf("Buffer cache flushed\n");
	return 0;
}


int get_dev (char *devname)
{
	int fd;
	static long parm, multcount;
	struct stat stat_buf;
	if (stat(devname,&stat_buf)) {
		perror(devname);
		return -1;
	}

	switch(major(stat_buf.st_rdev)) {
#ifdef SCSI_DISK0_MAJOR
	case (SCSI_DISK0_MAJOR):
	case (SCSI_DISK1_MAJOR):
	case (SCSI_DISK2_MAJOR):
	case (SCSI_DISK3_MAJOR):
	case (SCSI_DISK4_MAJOR):
	case (SCSI_DISK5_MAJOR):
	case (SCSI_DISK6_MAJOR):
	case (SCSI_DISK7_MAJOR):
#else
	case (SCSI_DISK_MAJOR):
#endif
#ifdef MD_MAJOR
	case (MD_MAJOR):
#endif
	case (VXVM_MAJOR):
#ifdef LVM_BLK_MAJOR
	case (LVM_BLK_MAJOR):
#endif
		return -1;
		break;
#ifdef XT_DISK_MAJOR
	case (XT_DISK_MAJOR):
#endif
	case IDE0_MAJOR:
	case IDE1_MAJOR:
#ifdef IDE2_MAJOR
	case IDE2_MAJOR:
#endif
#ifdef IDE3_MAJOR
	case IDE3_MAJOR:
#endif
#ifdef IDE4_MAJOR
	case IDE4_MAJOR:
#endif
#ifdef IDE5_MAJOR
	case IDE5_MAJOR:
#endif
#ifdef IDE6_MAJOR
	case IDE6_MAJOR:
#endif
#ifdef IDE7_MAJOR
	case IDE7_MAJOR:
#endif
#ifdef IDE8_MAJOR
	case IDE8_MAJOR:
#endif
#ifdef IDE9_MAJOR
	case IDE9_MAJOR:
#endif
		// Do nothing
		break;
	default:
		return -1;
		break;
	}
	fd = open (devname, O_RDONLY|O_NONBLOCK);
	if (fd < 0) {
		errprintf("Error opening %d\n", devname);
		return -1;
	}
	return fd;
}

int send_standby(device_t *dev)
{

#ifndef WIN_STANDBYNOW1
#define WIN_STANDBYNOW1 0xE0
#endif
#ifndef WIN_STANDBYNOW2
#define WIN_STANDBYNOW2 0x94
#endif
	unsigned char args1[4] = {WIN_STANDBYNOW1,0,0,0};
	unsigned char args2[4] = {WIN_STANDBYNOW2,0,0,0};

	errprintf("------Issuing standby command\n");
	if (ioctl(dev->fd, HDIO_DRIVE_CMD, &args1)
	 && ioctl(dev->fd, HDIO_DRIVE_CMD, &args2)) {
		errprintf(" HDIO_DRIVE_CMD(standby) failed");
		return -1;
	}
	dev->sleeping = 1;
	return 0;
}



void add_device_t(char *dev, int fd)
{
//	device_t *ptr = devices, *prev = NULL;
	device_t *new;
	struct stat stat_buf;
	char *tmp;
				

	new = malloc(sizeof(device_t));
	if (!new) 
		die(FALSE, "Couldn't malloc\n");
	new->dev = malloc(strlen(dev) + 1);
	if (!new->dev) 
		die(FALSE, "Couldn't malloc\n");
	strcpy(new->dev, dev);
	if(proc_stat_dir != NULL) {
		// Linux 2.5, prepare for /sys/block/dev
		tmp = dev + (strlen(dev) -1);
		while(*tmp != '/' && tmp > dev) tmp--;
		if (*tmp == '/') tmp++;
		new->sys_stat = malloc(strlen(proc_stat_dir) + strlen(tmp) + 8);
		if (!new->sys_stat) 
			die(FALSE, "Couldn't malloc\n");
		sprintf(new->sys_stat, "%s/%s/stat", proc_stat_dir, tmp);
		errprintf("The proc device is %s\n", new->sys_stat);
	} else {
		// Linux 2.4, we don't need it
		new->sys_stat = NULL;
	}
	new->fd = fd;
	stat(dev,&stat_buf);
	new->major = major(stat_buf.st_rdev);
	new->minor = minor(stat_buf.st_rdev);
	new->ops = 0;
	new->timestamp = time(NULL);
	new->sleeping = 0;
	new->flushed = 0;
	new->next = devices;
	devices = new;
}


int standby_init(unsigned timeout, char *devnames)
{
	char *dev_s = NULL;
	char devs[1024];
	int fd;
	int count = 0;

	strncpy(devs, devnames, 1023);
	dev_s = strtok(devs, ", ");

	if( (get_device_t_ops = get_device_t_ops_version()) == NULL) {
			errprintf("/proc/stat | /sys/block | /sysfs are not available in the kernel\n");
			return 0;
	}
	while ( dev_s != NULL) {
		fd = get_dev(dev_s);
		if (fd >= 0 ) {
			errprintf("Addind device %s (%d)\n", dev_s, fd);
			count ++;
			add_device_t(dev_s, fd);
		} else {
			errprintf("Device %s doesn't allow standby\n", dev_s);
		}
		dev_s = strtok(NULL, ", ");
	}
	errprintf("Devices for standby: %d\n", count);
	return count;
}

void check_standby()
{
	int count = 0;
	device_t *ptr = devices;
	unsigned ops;
	time_t	now;
	int res;
	unsigned elapsed;

	now = time(NULL);
	while (ptr != NULL) {
		ops = get_device_t_ops(ptr, count);
		count++;
		if ( ops >= 0) {
			if ( ops - ptr->ops  == 0 ) {
				if (ptr->sleeping) {
					// Do nothing for this disk
					ptr = ptr->next;
					continue;
				}
				elapsed = now - ptr->timestamp;
				if( elapsed > standby_timeout) {
					if (is_active(ptr)) {
						if (ptr->flushed == 0) 
							flush_buffer_cache(ptr);
						res = send_standby(ptr);
						if( res < 0) {
							errprintf("Error in standby %s\n", ptr->dev);
							ptr->timestamp = now;
						}
					} else {
						 ptr->sleeping = 1;
					}
					ptr->flushed = 0;
				} else if ( elapsed > (standby_timeout - TIME_TO_FLUSH) 
						&& ptr->flushed == 0 && is_active(ptr))  {
					// We flush in advance before trying to sleep
					flush_buffer_cache(ptr);
				}
			} else {
				if (ptr->flushed > 0 && now - ptr->flushed < TIME_TO_FLUSH) {
					errprintf("Forgiving, it's flushing\n");
				} else {
					ptr->flushed = 0;
					ptr->timestamp = now;
				}
				ptr->sleeping = 0;
				ptr->ops = ops;
			}
		}
		ptr = ptr->next;
	}

}

unsigned get_device_t_ops_24(device_t *dev, int count)
{
	static char *dev_line = NULL;
	static char major_minor[64];
	FILE *fp;
	char *str = NULL; 
	unsigned ops;

	if(dev_line == NULL) 
		dev_line = malloc(1024);
	
	if (!dev) 
		return 0;
	if (count == 0) {
		fp = fopen("/proc/stat", "r");
		while ( fp && (str=fgets(dev_line, 1024, fp)) ) {
			if (strncmp(dev_line, "disk_io:", 8) == 0)
				break;
		}
		fclose(fp);
		if (str == NULL) {
			errprintf("/proc/stat not available for disk_io\n");
			return -1;
		}
	}
	sprintf(major_minor, "(%u,%u):", dev->major, dev->minor);
	str = strstr(dev_line, major_minor);
	if(str != NULL) {
		str = str + strlen(major_minor);
		//"(%u,%u):(%u,%u,%u,%u,%u) "
		if(sscanf(str, "(%u,%*u,%*u,%*u,%*u)", &ops) == 1) {
			errprintf("%s %u operations\n", dev->dev, ops);
			return ops;
		}
	}
	return 0;

}

unsigned get_device_t_ops_25(device_t *dev, int count)
{
	static char *dev_line = NULL;
	static char major_minor[64];
	int fd, bytes;
	unsigned long long in, out;

	if(dev_line == NULL) 
		dev_line = malloc(1024);
	
	if (!dev) 
		return 0;
	fd = open(dev->sys_stat, O_RDONLY);
	if (fd < 0 || (bytes = read(fd, dev_line, 1024)) <= 0) {
		errprintf("%s not available for disk_io\n", dev->sys_stat);
		close(fd);
		return -1;
	}
	close(fd);
	dev_line[bytes] = 0;
		//"%u %u %u %u %u %u %u %u %u %u %u"
		//"%8u %8u %8llu %8u "
		//"%8u %8u %8llu %8u "
		//"%8u %8u %8u"
	if(sscanf(dev_line, "%*u %*u %llu %*u %*u %*u %llu %*u %*u %*u %*u", &in, &out) == 2) {
		errprintf("%s %u operations\n", dev->dev, in+out);
		return (unsigned) (in+out) ;
	}
	return 0;
}


get_opts_t get_device_t_ops_version()
{
	struct stat stat_buf;

	// Linux >2.5
	if (stat("/sys/block",&stat_buf) == 0) {
		errprintf("Detected kernel 2.5\n");
		proc_stat_dir = "/sys/block";
		return get_device_t_ops_25;
	}
	// Linux >2.5
	if (stat("/sysfs/block",&stat_buf) == 0) {
		errprintf("Detected kernel 2.5\n");
		proc_stat_dir = "/sysfs/block";
		return get_device_t_ops_25;
	}
	// Linux 2.4
	if (stat("/proc/stat",&stat_buf) == 0) {
		errprintf("Detected kernel 2.4\n");
		return get_device_t_ops_24;
	}
	return NULL;
}
