;; Test routines for complex-numbers.clj

;; by Konrad Hinsen
;; last updated April 2, 2009

;; Copyright (c) Konrad Hinsen, 2008. All rights reserved.  The use
;; and distribution terms for this software are covered by the Eclipse
;; Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;; which can be found in the file epl-v10.html at the root of this
;; distribution.  By using this software in any fashion, you are
;; agreeing to be bound by the terms of this license.  You must not
;; remove this notice, or any other, from this software.

(ns clojure.contrib.test-contrib.complex-numbers
  (:refer-clojure :exclude [+ - * / = < > <= >=])
  (:use [clojure.test
	 :only (deftest is are run-tests)]
	[clojure.contrib.generic.arithmetic
	 :only (+ - * /)]
	[clojure.contrib.generic.comparison
	 :only (= < > <= >=)]
	[clojure.contrib.generic.math-functions
	 :only (abs approx= conjugate exp sqr sqrt)]
	[clojure.contrib.complex-numbers
	 :only (complex imaginary real imag)]))

(deftest complex-addition
  (is (= (+ (complex 1 2) (complex 1 2)) (complex 2 4)))
  (is (= (+ (complex 1 2) (complex -3 -7)) (complex -2 -5)))
  (is (= (+ (complex -3 -7) (complex 1 2)) (complex -2 -5)))
  (is (= (+ (complex 1 2) 3) (complex 4 2)))
  (is (= (+ 3 (complex 1 2)) (complex 4 2)))
  (is (= (+ (complex 1 2) -1) (imaginary 2)))
  (is (= (+ -1 (complex 1 2)) (imaginary 2)))
  (is (= (+ (complex 1 2) (imaginary -2)) 1))
  (is (= (+ (imaginary -2) (complex 1 2)) 1))
  (is (= (+ (complex 1 2) (imaginary 5)) (complex 1 7)))
  (is (= (+ (imaginary 5) (complex 1 2)) (complex 1 7)))
  (is (= (+ (complex -3 -7) (complex 1 2)) (complex -2 -5)))
  (is (= (+ (complex 1 2) (complex -3 -7)) (complex -2 -5)))
  (is (= (+ (complex -3 -7) (complex -3 -7)) (complex -6 -14)))
  (is (= (+ (complex -3 -7) 3) (imaginary -7)))
  (is (= (+ 3 (complex -3 -7)) (imaginary -7)))
  (is (= (+ (complex -3 -7) -1) (complex -4 -7)))
  (is (= (+ -1 (complex -3 -7)) (complex -4 -7)))
  (is (= (+ (complex -3 -7) (imaginary -2)) (complex -3 -9)))
  (is (= (+ (imaginary -2) (complex -3 -7)) (complex -3 -9)))
  (is (= (+ (complex -3 -7) (imaginary 5)) (complex -3 -2)))
  (is (= (+ (imaginary 5) (complex -3 -7)) (complex -3 -2)))
  (is (= (+ 3 (complex 1 2)) (complex 4 2)))
  (is (= (+ (complex 1 2) 3) (complex 4 2)))
  (is (= (+ 3 (complex -3 -7)) (imaginary -7)))
  (is (= (+ (complex -3 -7) 3) (imaginary -7)))
  (is (= (+ 3 (imaginary -2)) (complex 3 -2)))
  (is (= (+ (imaginary -2) 3) (complex 3 -2)))
  (is (= (+ 3 (imaginary 5)) (complex 3 5)))
  (is (= (+ (imaginary 5) 3) (complex 3 5)))
  (is (= (+ -1 (complex 1 2)) (imaginary 2)))
  (is (= (+ (complex 1 2) -1) (imaginary 2)))
  (is (= (+ -1 (complex -3 -7)) (complex -4 -7)))
  (is (= (+ (complex -3 -7) -1) (complex -4 -7)))
  (is (= (+ -1 (imaginary -2)) (complex -1 -2)))
  (is (= (+ (imaginary -2) -1) (complex -1 -2)))
  (is (= (+ -1 (imaginary 5)) (complex -1 5)))
  (is (= (+ (imaginary 5) -1) (complex -1 5)))
  (is (= (+ (imaginary -2) (complex 1 2)) 1))
  (is (= (+ (complex 1 2) (imaginary -2)) 1))
  (is (= (+ (imaginary -2) (complex -3 -7)) (complex -3 -9)))
  (is (= (+ (complex -3 -7) (imaginary -2)) (complex -3 -9)))
  (is (= (+ (imaginary -2) 3) (complex 3 -2)))
  (is (= (+ 3 (imaginary -2)) (complex 3 -2)))
  (is (= (+ (imaginary -2) -1) (complex -1 -2)))
  (is (= (+ -1 (imaginary -2)) (complex -1 -2)))
  (is (= (+ (imaginary -2) (imaginary -2)) (imaginary -4)))
  (is (= (+ (imaginary -2) (imaginary 5)) (imaginary 3)))
  (is (= (+ (imaginary 5) (imaginary -2)) (imaginary 3)))
  (is (= (+ (imaginary 5) (complex 1 2)) (complex 1 7)))
  (is (= (+ (complex 1 2) (imaginary 5)) (complex 1 7)))
  (is (= (+ (imaginary 5) (complex -3 -7)) (complex -3 -2)))
  (is (= (+ (complex -3 -7) (imaginary 5)) (complex -3 -2)))
  (is (= (+ (imaginary 5) 3) (complex 3 5)))
  (is (= (+ 3 (imaginary 5)) (complex 3 5)))
  (is (= (+ (imaginary 5) -1) (complex -1 5)))
  (is (= (+ -1 (imaginary 5)) (complex -1 5)))
  (is (= (+ (imaginary 5) (imaginary -2)) (imaginary 3)))
  (is (= (+ (imaginary -2) (imaginary 5)) (imaginary 3)))
  (is (= (+ (imaginary 5) (imaginary 5)) (imaginary 10))))

(deftest complex-subtraction
  (is (= (- (complex 1 2) (complex 1 2)) 0))
  (is (= (- (complex 1 2) (complex -3 -7)) (complex 4 9)))
  (is (= (- (complex -3 -7) (complex 1 2)) (complex -4 -9)))
  (is (= (- (complex 1 2) 3) (complex -2 2)))
  (is (= (- 3 (complex 1 2)) (complex 2 -2)))
  (is (= (- (complex 1 2) -1) (complex 2 2)))
  (is (= (- -1 (complex 1 2)) (complex -2 -2)))
  (is (= (- (complex 1 2) (imaginary -2)) (complex 1 4)))
  (is (= (- (imaginary -2) (complex 1 2)) (complex -1 -4)))
  (is (= (- (complex 1 2) (imaginary 5)) (complex 1 -3)))
  (is (= (- (imaginary 5) (complex 1 2)) (complex -1 3)))
  (is (= (- (complex -3 -7) (complex 1 2)) (complex -4 -9)))
  (is (= (- (complex 1 2) (complex -3 -7)) (complex 4 9)))
  (is (= (- (complex -3 -7) (complex -3 -7)) 0))
  (is (= (- (complex -3 -7) 3) (complex -6 -7)))
  (is (= (- 3 (complex -3 -7)) (complex 6 7)))
  (is (= (- (complex -3 -7) -1) (complex -2 -7)))
  (is (= (- -1 (complex -3 -7)) (complex 2 7)))
  (is (= (- (complex -3 -7) (imaginary -2)) (complex -3 -5)))
  (is (= (- (imaginary -2) (complex -3 -7)) (complex 3 5)))
  (is (= (- (complex -3 -7) (imaginary 5)) (complex -3 -12)))
  (is (= (- (imaginary 5) (complex -3 -7)) (complex 3 12)))
  (is (= (- 3 (complex 1 2)) (complex 2 -2)))
  (is (= (- (complex 1 2) 3) (complex -2 2)))
  (is (= (- 3 (complex -3 -7)) (complex 6 7)))
  (is (= (- (complex -3 -7) 3) (complex -6 -7)))
  (is (= (- 3 (imaginary -2)) (complex 3 2)))
  (is (= (- (imaginary -2) 3) (complex -3 -2)))
  (is (= (- 3 (imaginary 5)) (complex 3 -5)))
  (is (= (- (imaginary 5) 3) (complex -3 5)))
  (is (= (- -1 (complex 1 2)) (complex -2 -2)))
  (is (= (- (complex 1 2) -1) (complex 2 2)))
  (is (= (- -1 (complex -3 -7)) (complex 2 7)))
  (is (= (- (complex -3 -7) -1) (complex -2 -7)))
  (is (= (- -1 (imaginary -2)) (complex -1 2)))
  (is (= (- (imaginary -2) -1) (complex 1 -2)))
  (is (= (- -1 (imaginary 5)) (complex -1 -5)))
  (is (= (- (imaginary 5) -1) (complex 1 5)))
  (is (= (- (imaginary -2) (complex 1 2)) (complex -1 -4)))
  (is (= (- (complex 1 2) (imaginary -2)) (complex 1 4)))
  (is (= (- (imaginary -2) (complex -3 -7)) (complex 3 5)))
  (is (= (- (complex -3 -7) (imaginary -2)) (complex -3 -5)))
  (is (= (- (imaginary -2) 3) (complex -3 -2)))
  (is (= (- 3 (imaginary -2)) (complex 3 2)))
  (is (= (- (imaginary -2) -1) (complex 1 -2)))
  (is (= (- -1 (imaginary -2)) (complex -1 2)))
  (is (= (- (imaginary -2) (imaginary -2)) 0))
  (is (= (- (imaginary -2) (imaginary 5)) (imaginary -7)))
  (is (= (- (imaginary 5) (imaginary -2)) (imaginary 7)))
  (is (= (- (imaginary 5) (complex 1 2)) (complex -1 3)))
  (is (= (- (complex 1 2) (imaginary 5)) (complex 1 -3)))
  (is (= (- (imaginary 5) (complex -3 -7)) (complex 3 12)))
  (is (= (- (complex -3 -7) (imaginary 5)) (complex -3 -12)))
  (is (= (- (imaginary 5) 3) (complex -3 5)))
  (is (= (- 3 (imaginary 5)) (complex 3 -5)))
  (is (= (- (imaginary 5) -1) (complex 1 5)))
  (is (= (- -1 (imaginary 5)) (complex -1 -5)))
  (is (= (- (imaginary 5) (imaginary -2)) (imaginary 7)))
  (is (= (- (imaginary -2) (imaginary 5)) (imaginary -7)))
  (is (= (- (imaginary 5) (imaginary 5)) 0)))

(deftest complex-multiplication
  (is (= (* (complex 1 2) (complex 1 2)) (complex -3 4)))
  (is (= (* (complex 1 2) (complex -3 -7)) (complex 11 -13)))
  (is (= (* (complex -3 -7) (complex 1 2)) (complex 11 -13)))
  (is (= (* (complex 1 2) 3) (complex 3 6)))
  (is (= (* 3 (complex 1 2)) (complex 3 6)))
  (is (= (* (complex 1 2) -1) (complex -1 -2)))
  (is (= (* -1 (complex 1 2)) (complex -1 -2)))
  (is (= (* (complex 1 2) (imaginary -2)) (complex 4 -2)))
  (is (= (* (imaginary -2) (complex 1 2)) (complex 4 -2)))
  (is (= (* (complex 1 2) (imaginary 5)) (complex -10 5)))
  (is (= (* (imaginary 5) (complex 1 2)) (complex -10 5)))
  (is (= (* (complex -3 -7) (complex 1 2)) (complex 11 -13)))
  (is (= (* (complex 1 2) (complex -3 -7)) (complex 11 -13)))
  (is (= (* (complex -3 -7) (complex -3 -7)) (complex -40 42)))
  (is (= (* (complex -3 -7) 3) (complex -9 -21)))
  (is (= (* 3 (complex -3 -7)) (complex -9 -21)))
  (is (= (* (complex -3 -7) -1) (complex 3 7)))
  (is (= (* -1 (complex -3 -7)) (complex 3 7)))
  (is (= (* (complex -3 -7) (imaginary -2)) (complex -14 6)))
  (is (= (* (imaginary -2) (complex -3 -7)) (complex -14 6)))
  (is (= (* (complex -3 -7) (imaginary 5)) (complex 35 -15)))
  (is (= (* (imaginary 5) (complex -3 -7)) (complex 35 -15)))
  (is (= (* 3 (complex 1 2)) (complex 3 6)))
  (is (= (* (complex 1 2) 3) (complex 3 6)))
  (is (= (* 3 (complex -3 -7)) (complex -9 -21)))
  (is (= (* (complex -3 -7) 3) (complex -9 -21)))
  (is (= (* 3 (imaginary -2)) (imaginary -6)))
  (is (= (* (imaginary -2) 3) (imaginary -6)))
  (is (= (* 3 (imaginary 5)) (imaginary 15)))
  (is (= (* (imaginary 5) 3) (imaginary 15)))
  (is (= (* -1 (complex 1 2)) (complex -1 -2)))
  (is (= (* (complex 1 2) -1) (complex -1 -2)))
  (is (= (* -1 (complex -3 -7)) (complex 3 7)))
  (is (= (* (complex -3 -7) -1) (complex 3 7)))
  (is (= (* -1 (imaginary -2)) (imaginary 2)))
  (is (= (* (imaginary -2) -1) (imaginary 2)))
  (is (= (* -1 (imaginary 5)) (imaginary -5)))
  (is (= (* (imaginary 5) -1) (imaginary -5)))
  (is (= (* (imaginary -2) (complex 1 2)) (complex 4 -2)))
  (is (= (* (complex 1 2) (imaginary -2)) (complex 4 -2)))
  (is (= (* (imaginary -2) (complex -3 -7)) (complex -14 6)))
  (is (= (* (complex -3 -7) (imaginary -2)) (complex -14 6)))
  (is (= (* (imaginary -2) 3) (imaginary -6)))
  (is (= (* 3 (imaginary -2)) (imaginary -6)))
  (is (= (* (imaginary -2) -1) (imaginary 2)))
  (is (= (* -1 (imaginary -2)) (imaginary 2)))
  (is (= (* (imaginary -2) (imaginary -2)) -4))
  (is (= (* (imaginary -2) (imaginary 5)) 10))
  (is (= (* (imaginary 5) (imaginary -2)) 10))
  (is (= (* (imaginary 5) (complex 1 2)) (complex -10 5)))
  (is (= (* (complex 1 2) (imaginary 5)) (complex -10 5)))
  (is (= (* (imaginary 5) (complex -3 -7)) (complex 35 -15)))
  (is (= (* (complex -3 -7) (imaginary 5)) (complex 35 -15)))
  (is (= (* (imaginary 5) 3) (imaginary 15)))
  (is (= (* 3 (imaginary 5)) (imaginary 15)))
  (is (= (* (imaginary 5) -1) (imaginary -5)))
  (is (= (* -1 (imaginary 5)) (imaginary -5)))
  (is (= (* (imaginary 5) (imaginary -2)) 10))
  (is (= (* (imaginary -2) (imaginary 5)) 10))
  (is (= (* (imaginary 5) (imaginary 5)) -25)))

(deftest complex-division
  (is (= (/ (complex 1 2) (complex 1 2)) 1))
  (is (= (/ (complex 1 2) (complex -3 -7)) (complex -17/58 1/58)))
  (is (= (/ (complex -3 -7) (complex 1 2)) (complex -17/5 -1/5)))
  (is (= (/ (complex 1 2) 3) (complex 1/3 2/3)))
  (is (= (/ 3 (complex 1 2)) (complex 3/5 -6/5)))
  (is (= (/ (complex 1 2) -1) (complex -1 -2)))
  (is (= (/ -1 (complex 1 2)) (complex -1/5 2/5)))
  (is (= (/ (complex 1 2) (imaginary -2)) (complex -1 1/2)))
  (is (= (/ (imaginary -2) (complex 1 2)) (complex -4/5 -2/5)))
  (is (= (/ (complex 1 2) (imaginary 5)) (complex 2/5 -1/5)))
  (is (= (/ (imaginary 5) (complex 1 2)) (complex 2 1)))
  (is (= (/ (complex -3 -7) (complex 1 2)) (complex -17/5 -1/5)))
  (is (= (/ (complex 1 2) (complex -3 -7)) (complex -17/58 1/58)))
  (is (= (/ (complex -3 -7) (complex -3 -7)) 1))
  (is (= (/ (complex -3 -7) 3) (complex -1 -7/3)))
  (is (= (/ 3 (complex -3 -7)) (complex -9/58 21/58)))
  (is (= (/ (complex -3 -7) -1) (complex 3 7)))
  (is (= (/ -1 (complex -3 -7)) (complex 3/58 -7/58)))
  (is (= (/ (complex -3 -7) (imaginary -2)) (complex 7/2 -3/2)))
  (is (= (/ (imaginary -2) (complex -3 -7)) (complex 7/29 3/29)))
  (is (= (/ (complex -3 -7) (imaginary 5)) (complex -7/5 3/5)))
  (is (= (/ (imaginary 5) (complex -3 -7)) (complex -35/58 -15/58)))
  (is (= (/ 3 (complex 1 2)) (complex 3/5 -6/5)))
  (is (= (/ (complex 1 2) 3) (complex 1/3 2/3)))
  (is (= (/ 3 (complex -3 -7)) (complex -9/58 21/58)))
  (is (= (/ (complex -3 -7) 3) (complex -1 -7/3)))
  (is (= (/ 3 (imaginary -2)) (imaginary 1.5)))
  (is (= (/ (imaginary -2) 3) (imaginary -2/3)))
  (is (= (/ 3 (imaginary 5)) (imaginary -3/5)))
  (is (= (/ (imaginary 5) 3) (imaginary 5/3)))
  (is (= (/ -1 (complex 1 2)) (complex -1/5 2/5)))
  (is (= (/ (complex 1 2) -1) (complex -1 -2)))
  (is (= (/ -1 (complex -3 -7)) (complex 3/58 -7/58)))
  (is (= (/ (complex -3 -7) -1) (complex 3 7)))
  (is (= (/ -1 (imaginary -2)) (imaginary -1/2)))
  (is (= (/ (imaginary -2) -1) (imaginary 2)))
  (is (= (/ -1 (imaginary 5)) (imaginary 1/5)))
  (is (= (/ (imaginary 5) -1) (imaginary -5)))
  (is (= (/ (imaginary -2) (complex 1 2)) (complex -4/5 -2/5)))
  (is (= (/ (complex 1 2) (imaginary -2)) (complex -1 1/2)))
  (is (= (/ (imaginary -2) (complex -3 -7)) (complex 7/29 3/29)))
  (is (= (/ (complex -3 -7) (imaginary -2)) (complex 7/2 -3/2)))
  (is (= (/ (imaginary -2) 3) (imaginary -2/3)))
  (is (= (/ 3 (imaginary -2)) (imaginary 3/2)))
  (is (= (/ (imaginary -2) -1) (imaginary 2)))
  (is (= (/ -1 (imaginary -2)) (imaginary -1/2)))
  (is (= (/ (imaginary -2) (imaginary -2)) 1))
  (is (= (/ (imaginary -2) (imaginary 5)) -2/5))
  (is (= (/ (imaginary 5) (imaginary -2)) -5/2))
  (is (= (/ (imaginary 5) (complex 1 2)) (complex 2 1)))
  (is (= (/ (complex 1 2) (imaginary 5)) (complex 2/5 -1/5)))
  (is (= (/ (imaginary 5) (complex -3 -7)) (complex -35/58 -15/58)))
  (is (= (/ (complex -3 -7) (imaginary 5)) (complex -7/5 3/5)))
  (is (= (/ (imaginary 5) 3) (imaginary 5/3)))
  (is (= (/ 3 (imaginary 5)) (imaginary -3/5)))
  (is (= (/ (imaginary 5) -1) (imaginary -5)))
  (is (= (/ -1 (imaginary 5)) (imaginary 1/5)))
  (is (= (/ (imaginary 5) (imaginary -2)) -5/2))
  (is (= (/ (imaginary -2) (imaginary 5)) -2/5))
  (is (= (/ (imaginary 5) (imaginary 5)) 1)))

(deftest complex-conjugate
  (is (= (conjugate (complex 1 2)) (complex 1 -2)))
  (is (= (conjugate (complex -3 -7)) (complex -3 7)))
  (is (= (conjugate (imaginary -2)) (imaginary 2)))
  (is (= (conjugate (imaginary 5)) (imaginary -5))))

(deftest complex-abs
  (doseq [c [(complex 1 2) (complex -2 3) (complex 4 -2)
	     (complex -3 -7) (imaginary -2) (imaginary 5)]]
    (is (approx= (* c (conjugate c))
		 (sqr (abs c))
		 1e-14))))

(deftest complex-sqrt
  (doseq [c [(complex 1 2) (complex -2 3) (complex 4 -2)
	     (complex -3 -7) (imaginary -2) (imaginary 5)]]
    (let [r (sqrt c)]
      (is (approx= c (sqr r) 1e-14))
      (is (>= (real r) 0)))))

(deftest complex-exp
  (is (approx= (exp (complex 1 2))
	       (complex -1.1312043837568135 2.4717266720048188)
	       1e-14))
  (is (approx= (exp (complex 2 3))
	       (complex -7.3151100949011028 1.0427436562359045)
	       1e-14))
  (is (approx= (exp (complex 4 -2))
	       (complex -22.720847417619233 -49.645957334580565)
	       1e-14))
  (is (approx= (exp (complex 3 -7))
	       (complex 15.142531566086868 -13.195928586605717)
	       1e-14))
  (is (approx= (exp (imaginary -2))
	       (complex -0.41614683654714241 -0.90929742682568171)
	       1e-14))
  (is (approx= (exp (imaginary 5))
	       (complex 0.2836621854632263 -0.95892427466313845)
	       1e-14)))
