/*
 *   This file is part of Clinica.
 *
 *   Clinica is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   Clinica is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with Clinica.  If not, see <http://www.gnu.org/licenses/>.
 *
 *   Authors: Leonardo Robol <leo@robol.it>
 *            Gianmarco Brocchi <brocchi@poisson.phc.unipi.it>
 */

using Gtk;

namespace Clinica {

    public class PatientListStore : ListStore {
    
    	enum Field {
    		PATIENT,
    		GIVEN_NAME,
    		SURNAME,
    		COMPLETE_NAME,
    	}
    
        public signal void error (string message);
        
		private ResourceManager resource_manager;
        
        /**
         * @brief Create a new PatientListStore starting
         * from the passed ResourceManager.
         */
        public PatientListStore (ResourceManager resources) {
            resource_manager = resources;
            /* Keep a reference to the resource manager and connect
             * callback error */
            error.connect ((t,l) => resources.error_callback(t,l));
        
            /* Init new list, with type Patient, name, surname and
             * Complete name */
            Type [] column_headers = { typeof (Patient) , typeof (string), 
                                       typeof (string) , typeof (string)};
            set_column_types (column_headers);
            
            /* Asynchronous loading of patients */
            Idle.add(() => {
		        foreach (Patient p in Patient.all (resource_manager)) {
		        	add_patient (p);
		        }
		        
		        /* We don't need to reiterate this function any more */
		        return false;
		    });
        }
        
        /**
         * @brief Append the given patient to the liststore
         */
        public void add_patient (Patient p) {
        	TreeIter it;
            append (out it);
            set_value (it, Field.PATIENT,    p);
            set_value (it, Field.GIVEN_NAME, p.given_name);
            set_value (it, Field.SURNAME,    p.surname);
            set_value (it, Field.COMPLETE_NAME, p.get_complete_name ());
	    }
	    
	    /**
	     * @brief Reload information about patient p that
	     * should be already present in the database
	     */
	    public void reload_patient (Patient p) {
	    	TreeIter iter;
	    	Value patient;
	    	
	    	if (!get_iter_first (out iter)) {
				error(_("Patients database seems corrupted. This is likely to be a bug in the application"));
	    	}
	    	
	    	do {
	    		get_value (iter, Field.PATIENT, out patient);
	    		if ((patient as Patient).get_id () == p.get_id ()) {
	    			set_value(iter, Field.PATIENT, p);
	    			set_value(iter, Field.GIVEN_NAME, p.given_name);
	    			set_value(iter, Field.SURNAME, p.surname);
	    			set_value(iter, Field.COMPLETE_NAME, p.get_complete_name ());
	    			return;
	    		}
	    	} while (iter_next (ref iter));
	    	
	    	// We hope to find the patient :)
	    	assert_not_reached ();
	    }
	    
	    /**
	     * @brief Return a list of treeiters pointing to the patients
	     * of the given doctor. 
	     */
	    public List<TreeIter?> get_patients_of (Doctor doc) {
	    	Value value;
	    	var iters = new List<TreeIter?> ();
	    	TreeIter it;
	    	Patient p;
	    	
	    	if (!get_iter_first (out it)) {
	    		return iters;
	    	}
	    	do {
	    		get_value (it, Field.PATIENT, out value);
	    		p = value as Patient;
	    		if (p.doctor.get_id () == doc.get_id ()) {
	    			iters.append (it);
	    		} 
	    	} while (iter_next(ref it));
	    	
	    	return iters;
	    }
	    
	    public TreeIter patient_to_iter (Patient p) {
	    	TreeIter it;
	    	Value patient;
	    	if (!get_iter_first (out it))
				error(_("Patients database seems corrupted. This is likely to be a bug in the application"));
	    	do {
	    		get_value (it, Field.PATIENT, out patient);
	    		if ((patient as Patient).get_id () == p.get_id ()) {
	    			return it;
	    		}
	    	} while (iter_next (ref it));
	    	
	    	assert_not_reached ();
	    }
	    
	    public void remove_patient (Patient p) {
	    	TreeIter p_iter = patient_to_iter (p);
	    	remove_patient_from_iter (p_iter);
	    }
	    
	    public void remove_patient_from_iter (TreeIter it) {
	    	Value patient;
	    	get_value (it, Field.PATIENT, out patient);
	    	
	    	var visits = resource_manager.visit_list_store.get_visits_of (patient as Patient);
	    	
	    	foreach (TreeIter v_it in visits) {
	    		resource_manager.visit_list_store.remove_visit_from_iter (v_it);
	    	}
	    	
	    	/* Actually remove the patient */
	    	(patient as Patient).remove ();
	    	remove (it);
	    }
	}
}
