/*  CamStream: a collection of GUI webcam tools
    Copyright (C) 2002-2005 Nemosoft Unv.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    For questions, remarks, patches, etc. for this program, the author can be
    reached at camstream@smcc.demon.nl.
*/

/**
  \class CVideoFrame

  CVideoFrame holds a single image from a video stream. It can hold
  the RGB or YUV image (or both). It contains sequence number and timestamp
  fields.

  The classes uses shared data, so passing copies of the class around does
  not incur a penalty. However, changes to the image data will be reflected
  by all shared objects!

*/

#include "VideoFrame.h"

CVideoFrame::CVideoFrame(uint number, const QImage *rgb, const QImage *y, const QImage *u, const QImage *v)
{
   data = new VideoDataShared;
   if (data == 0)
     return;

   data->RGB = rgb;
   data->Y = y;
   data->U = u;
   data->V = v;
   data->sequence = 0;
   data->time_stamp = 0;
   m_Number = number;
}

CVideoFrame::CVideoFrame(const CVideoFrame &f)
{
   data = f.data;
   if (data != 0)
   {
     data->ref();
   }
}


CVideoFrame::~CVideoFrame()
{
   if (data && data->deref()) {
     delete data;
     data = 0;
   }
}

// private

uint CVideoFrame::GetRefCount() const
{
   if (data != 0)
     return 0;
   return data->count;
}

// public

CVideoFrame &CVideoFrame::operator =(const CVideoFrame &f)
{
   if (f.data == 0)
     return *this;
   f.data->ref();
   if (data && data->deref()) {
     delete data;
     data = 0;
   }
   data = f.data;
   return *this;
}

const QImage *CVideoFrame::GetRGB() const
{
   return data->RGB;
}

const QImage *CVideoFrame::GetY() const
{
   return data->Y;
}

const QImage *CVideoFrame::GetU() const
{
   return data->U;
}

const QImage *CVideoFrame::GetV() const
{
   return data->V;
}


uint CVideoFrame::GetNumber() const
{
   return m_Number;
}

void CVideoFrame::SetSequence(unsigned long seq)
{
   data->sequence = seq;
}

/**
  \brief Get sequence number

  VideoFrames are sequentially numbered as they are fetches from videodevice
  or file. The sequence number is a monotonous increasing number.

  Note: when fetching frames from a video device, sequence numbers are not
  contiguous in case frames are dropped by the capture process.

*/
unsigned long CVideoFrame::GetSequence() const
{
   return data->sequence;
}

void CVideoFrame::SetTimeStamp(unsigned long stamp)
{
   data->time_stamp = stamp;
}

/**
  \brief Get time stamp
  \return Time in milliseconds

  VideoFrames are time stamped, to make it easier to order/reference them. The
  timestamp is measured in milliseconds.

  Note: The first video frame of a video sequence is not guaranteed to have a
  timestamp of 0! The only guarantee you have is that the timestamp is a monotonous
  increasing number.
*/
unsigned long CVideoFrame::GetTimeStamp() const
{
   return data->time_stamp;
}

