/** \file libassogiate/mime-directory.cc */
/*
 * This file is part of assoGiate,
 * an editor of the file types database for GNOME.
 *
 * Copyright (C) 2007 Kevin Daughtridge <kevin@kdau.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "private.hh"
#include "mime-directory.hh"

#include <cerrno>
#include <glibmm/fileutils.h>
#include <glibmm/miscutils.h>
#include <glibmm/stringutils.h>
#include <libgnomevfsmm/init.h>

/******************************************************************************/
namespace assoGiate {
/******************************************************************************/

/******************************************************************************/
/* class MimeDirectory                                                        */
/******************************************************************************/

MimeDirectory*
MimeDirectory::create(const std::string& data_dir, Location type)
	throw(MimeDatabaseLoadError, Glib::FileError)
{
	ustring mime_dir = Glib::build_filename(data_dir, "mime"),
		pkgs_dir = Glib::build_filename(mime_dir, "packages");

	if (!Glib::file_test(pkgs_dir, Glib::FILE_TEST_IS_DIR))
		if (g_mkdir_with_parents(pkgs_dir.data(), 0755) != 0)
			throw Glib::FileError(Glib::FileError::FAILED,
				Glib::strerror(errno));
	
	return new MimeDirectory(data_dir, type);
}

MimeDirectory*
MimeDirectory::load(const std::string& data_dir, Location type)
	throw(MimeDatabaseLoadError)
{
	ustring mime_dir = Glib::build_filename(data_dir, "mime"),
		pkgs_dir = Glib::build_filename(mime_dir, "packages");

	if (!Glib::file_test(pkgs_dir, Glib::FILE_TEST_IS_DIR))
		return NULL;

	return new MimeDirectory(data_dir, type);
}

MimeDirectory::MimeDirectory(const std::string& data_dir, Location type)
	throw(MimeDatabaseLoadError)
:	m_data_dir(data_dir),
	m_mime_dir(Glib::build_filename(m_data_dir, "mime")),
	m_packages_dir(Glib::build_filename(m_mime_dir, "packages")),
	m_type(type), m_packages(),
	m_monitor(), s_changed()
{
	reload();
	
	Gnome::Vfs::init();
	m_monitor.add(m_packages_dir, Gnome::Vfs::MONITOR_DIRECTORY,
		sigc::mem_fun(*this, &MimeDirectory::on_monitor_event));
}

MimeDirectory::~MimeDirectory()
{	clear(); }

std::string
MimeDirectory::get_data_dir() const throw()
{	return m_data_dir; }

std::string
MimeDirectory::get_mime_dir() const throw()
{	return m_mime_dir; }

std::string
MimeDirectory::get_packages_dir() const throw()
{	return m_packages_dir; }

void
MimeDirectory::extend_node_map(MimeNodeMap& node_map) throw()
{
	FOREACH(std::list<MimePackage*>, m_packages, i)
		(*i)->extend_node_map(node_map);
}

void
MimeDirectory::reload() throw(MimeDatabaseLoadError)
{
	clear();
	
	try {
		Glib::Dir pkgs_dir(m_packages_dir);
		
		FOREACH_BASE(Glib::DirIterator, pkgs_dir, i) {
			bool is_override = (*i == "Override.xml");
			Location type = m_type;

			switch (m_type) {
			case SYSTEM_STANDARD:
			case USER_STANDARD:
				if (is_override) continue;
				break;
			case SYSTEM_OVERRIDE:
				if (!is_override) type = SYSTEM_STANDARD;
				break;
			case USER_OVERRIDE:
				if (!is_override) type = USER_STANDARD;
				break;
			case NO_LOCATION:
				continue;
			}

			MimePackage *pkg = new MimePackage
				(Glib::build_filename(m_packages_dir, *i), type);
			m_packages.push_back(pkg);
		}
	} catch (Glib::Exception& e) {
		throw MimeDatabaseLoadError(e.what());
	}
	
	try {
		s_changed.emit();
	} catch (...) {}
}

sigc::signal<void>
MimeDirectory::signal_changed() throw()
{	return s_changed; }

void
MimeDirectory::clear() throw()
{
	FOREACH(std::list<MimePackage*>, m_packages, i)
		delete *i;
	m_packages.clear();
}

void
MimeDirectory::on_monitor_event(const Gnome::Vfs::MonitorHandle&,
	const ustring&, const ustring& info_uri, Gnome::Vfs::MonitorEventType type)
	throw()
{
	if (m_type != SYSTEM_OVERRIDE && m_type != USER_OVERRIDE &&
		Glib::str_has_suffix(info_uri, "Override.xml"))
		return;

	switch (type) {
	case Gnome::Vfs::MONITOR_EVENT_CREATED:
	case Gnome::Vfs::MONITOR_EVENT_DELETED:
	case Gnome::Vfs::MONITOR_EVENT_CHANGED:
		try {
			reload();
		} catch (MimeDatabaseLoadError& e) {
			e.unhandled();
		}
	default: break;
	}
}

} /* namespace assoGiate */
