------------------------------------------------------------------------------
--                                                                          --
--                          GNATCHECK COMPONENTS                            --
--                                                                          --
--       A S I S _ U L . G L O B A L _ S T A T E . U T I L I T I E S        --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                       Copyright (C) 2007, AdaCore                        --
--                                                                          --
-- Asis Utility Library (ASIS UL) is free software; you can redistribute it --
-- and/or  modify  it  under  terms  of  the  GNU General Public License as --
-- published by the Free Software Foundation; either version 2, or (at your --
-- option)  any later version.  ASIS UL  is distributed in the hope that it --
-- will  be  useful,  but  WITHOUT  ANY  WARRANTY; without even the implied --
-- warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the --
-- GNU  General Public License for more details. You should have received a --
-- copy of the  GNU General Public License  distributed with GNAT; see file --
-- COPYING. If not,  write  to the  Free Software Foundation,  51 Franklin  --
-- Street, Fifth Floor, Boston, MA 02110-1301, USA.                         --
--                                                                          --
-- ASIS UL is maintained by AdaCore (http://www.adacore.com).               --
--                                                                          --
------------------------------------------------------------------------------

with Asis.Declarations;          use Asis.Declarations;
with Asis.Elements;              use Asis.Elements;
with Asis.Expressions;           use Asis.Expressions;
with Asis.Extensions;            use Asis.Extensions;
with Asis.Extensions.Flat_Kinds; use Asis.Extensions.Flat_Kinds;
with Asis.Statements;            use Asis.Statements;

with A4G.A_Sem;                  use A4G.A_Sem;
with Asis.Set_Get;               use Asis.Set_Get;

with Atree;                      use Atree;
with Einfo;                      use Einfo;
with Sinfo;                      use Sinfo;

package body ASIS_UL.Global_State.Utilities is

   ----------------------
   -- Can_Create_Tasks --
   ----------------------

   function Can_Create_Tasks (El : Asis.Element) return Boolean is
   begin
      return
        Declaration_Kind (El) in
          A_Variable_Declaration .. A_Constant_Declaration
       or else
        Expression_Kind (El) = An_Allocation_From_Subtype;
   end Can_Create_Tasks;

   ---------------------------
   -- Corresponding_Element --
   ---------------------------

   function Corresponding_Element (El : Asis.Element) return Asis.Element is
      Result  : Asis.Element := El;
      Res_Old : Asis.Element := El;
   begin

      case Flat_Element_Kind (Result) is
         when A_Task_Body_Declaration =>

            if Is_Subunit (Result) then
               Result := Corresponding_Body_Stub (Result);
            end if;

            Result := Corresponding_Declaration (Result);

         when A_Procedure_Declaration |
              A_Function_Declaration  =>

            if Is_Part_Of_Inherited (Result) then
               Result := Corresponding_Declaration (Result);
               Result := Corresponding_Element (Result);
            end if;

            if Is_Part_Of_Instance (Result) then
               Res_Old := Result;
               Result  := Enclosing_Element (Result);

               if Declaration_Kind (Result) not in
                    A_Procedure_Instantiation .. A_Function_Instantiation
               then
                  --  Not an expanded spec, so undo the step up:
                  Result := Res_Old;
               end if;

            end if;

         when An_Entry_Body_Declaration =>
            Result := Corresponding_Declaration (Result);
         when A_Procedure_Body_Declaration |
              A_Function_Body_Declaration  |
              A_Procedure_Body_Stub        |
              A_Function_Body_Stub         =>

            Result := Corresponding_Declaration (Result);

            if Is_Nil (Result) then

               if Is_Subunit (El) then
                  Result :=
                    Corresponding_Element (Corresponding_Body_Stub (El));
               else
                  --  No explicit spec
                  Result := El;

               end if;

            end if;

         when A_Procedure_Renaming_Declaration |
              A_Function_Renaming_Declaration =>
            Result := Get_Renamed_Subprogram (El);

         when An_Accept_Statement =>
            Result :=
              Corresponding_Name_Declaration
                (Accept_Entry_Direct_Name (Result));

         when others =>
            null;
      end case;

      return Result;
   end Corresponding_Element;

   ---------------------
   -- Enclosing_Scope --
   ---------------------

   function Enclosing_Scope (El : Asis.Element) return Asis.Element is
      Result : Asis.Element := El;
   begin

      while not Is_Nil (Result) loop

         if Is_Subunit (Result) then
            Result := Corresponding_Body_Stub (Result);
         else
            Result := Enclosing_Element (Result);
         end if;

         exit when Is_Scope (Result);
      end loop;

      return Result;
   end Enclosing_Scope;

   ------------------------
   -- Get_Called_Element --
   ------------------------

   function Get_Called_Element (Call : Asis.Element) return Asis.Element is
   begin

      if Expression_Kind (Call) = A_Function_Call then
         return Corresponding_Called_Function (Call);
      else
         return Corresponding_Called_Entity (Call);
      end if;

   end Get_Called_Element;

   ----------------------------
   -- Get_Renamed_Subprogram --
   ----------------------------

   function Get_Renamed_Subprogram (El : Asis.Element) return Asis.Element is
      Result : Asis.Element;
   begin
      Result := Corresponding_Base_Entity (El);

      if Expression_Kind (Result) = A_Selected_Component then
         Result := Selector (Result);
      end if;

      case Expression_Kind (Result) is
         when An_Identifier |
              An_Operator_Symbol =>
            Result := Corresponding_Name_Declaration (Result);

            if Declaration_Kind (Result) in A_Procedure_Renaming_Declaration ..
                 A_Function_Renaming_Declaration
            then
               Result := Get_Renamed_Subprogram (Result);
            end if;

         when An_Attribute_Reference |
              An_Enumeration_Literal =>
            null;
         when others =>
            Result := Nil_Element;
      end case;

      return Result;
   end Get_Renamed_Subprogram;

   -------------
   -- Is_Call --
   -------------

   function Is_Call (El : Asis.Element) return Boolean is
      Result : Boolean := False;
   begin
      case Flat_Element_Kind (El) is
         when A_Function_Call            |
              A_Procedure_Call_Statement =>

            Result := True;
         when An_Entry_Call_Statement =>
            Result := True;
         when others =>
            null;
      end case;

      return Result;
   end Is_Call;

   -------------------------------------
   -- Is_Call_To_Predefined_Operation --
   -------------------------------------

   function Is_Call_To_Predefined_Operation
     (Call : Asis.Element)
      return Boolean
   is
      Result    : Boolean := False;
      Pref_Node : Node_Id;

      function Is_Call_To_Predefined_Op_Of_User_Type
        (N    : Node_Id)
         return Boolean;
      --  This function covers the cases not covered by
      --  A4G.A_Sem.Defined_In_Standard. For example, a predefined
      --  concatenation for a user-defined one-dimentioal array type

      function Is_Call_To_Predefined_Op_Of_User_Type
        (N    : Node_Id)
         return Boolean
      is
         N_Entity : Node_Id := Empty;
         Result   : Boolean := False;
      begin

         if Nkind (N) in N_Has_Entity then
            N_Entity := Entity (N);
         elsif Nkind (N) in Sinfo.N_Entity then
            N_Entity := N;
         end if;

         Result :=
           Present (N_Entity)
          and then
           not Comes_From_Source (N_Entity)
          and then
           No (Parent (N_Entity))
          and then
           Is_Intrinsic_Subprogram (N_Entity);

         return Result;

      end Is_Call_To_Predefined_Op_Of_User_Type;

   begin

      if Is_Static (Call) then
         Result := True;
      elsif Expression_Kind (Call) = A_Function_Call
        and then
         Function_Call_Parameters (Call)'Length in 1 .. 2
      then
         --  We use the direct access into the GNAT tree
         Pref_Node := R_Node (Call);

         if Nkind (Pref_Node) not in N_Op then
            Pref_Node := Node (Call);
         end if;

         if Nkind (Pref_Node) in N_Op
          and then
            (Defined_In_Standard (Pref_Node)
            or else
             Is_Call_To_Predefined_Op_Of_User_Type (Pref_Node))
         then
            Result := True;
         end if;

      end if;

      return Result;

   end Is_Call_To_Predefined_Operation;

   ---------------------------------------
   -- Is_Declaration_Of_Callable_Entity --
   ---------------------------------------

   function Is_Declaration_Of_Callable_Entity
     (El   : Asis.Element)
      return Boolean
   is
      Result : Boolean := False;
   begin

      case Declaration_Kind (El) is
         when A_Procedure_Instantiation |
              A_Function_Instantiation  |
              A_Task_Type_Declaration   |
              A_Single_Task_Declaration |
              An_Entry_Declaration      =>
            Result := True;

         when A_Procedure_Declaration |
              A_Function_Declaration  =>
            Result := Trait_Kind (El) /= An_Abstract_Trait;

         when A_Procedure_Body_Stub |
              A_Function_Body_Stub  =>

            if Declaration_Kind (Corresponding_Declaration (El)) not in
                 A_Generic_Declaration
            then
               Result := True;
            end if;

         when others =>
            null;
      end case;

      return Result;
   end Is_Declaration_Of_Callable_Entity;

   --------------------------------------
   -- Is_Predefined_Operation_Renaming --
   --------------------------------------

   function Is_Predefined_Operation_Renaming
     (Ren  : Asis.Element)
      return Boolean
   is
      Op_Entity : Entity_Id;
      Result    : Boolean := False;
   begin

      if Declaration_Kind (Ren) = A_Function_Renaming_Declaration then
         Op_Entity := Defining_Unit_Name (Specification (Node (Ren)));

         if Nkind (Op_Entity) /= N_Defining_Program_Unit_Name
           and then
            Ekind (Op_Entity) = E_Function
         then

            while Present (Alias (Op_Entity)) loop
               Op_Entity := Alias (Op_Entity);
            end loop;

            Result := Defined_In_Standard (Op_Entity);

         end if;

      end if;

      return Result;
   end Is_Predefined_Operation_Renaming;

   --------------
   -- Is_Scope --
   --------------

   function Is_Scope (El : Asis.Element) return Boolean is
      Result : Boolean := False;
   begin

      case Flat_Element_Kind (El) is
         when A_Procedure_Body_Declaration |
              A_Function_Body_Declaration  |
              A_Task_Body_Declaration      |
              An_Entry_Body_Declaration    |
              An_Accept_Statement          |
              A_Single_Task_Declaration    |
              A_Task_Type_Declaration      =>
            Result := True;
         when others =>
            null;
      end case;

      return Result;
   end Is_Scope;

end ASIS_UL.Global_State.Utilities;
