// ---------------------------------------------------------------------------
// - Cipher.hpp                                                              -
// - afnix cryptography - base cipher class definition                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2007 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  AFNIX_CIPHER_HPP
#define  AFNIX_CIPHER_HPP

#ifndef  AFNIX_KEY_HPP
#include "Key.hpp"
#endif

#ifndef  AFNIX_INPUT_HPP
#include "Input.hpp"
#endif

#ifndef  AFNIX_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace afnix {

  /// The Cipher class is a base class that is used to implement a symmetric
  /// block cipher. A reverse flag controls the cipher operation. When false, 
  /// the cipher operates in encrypt mode. When true, the cipher operates in 
  /// decrypt mode (i.e reverse mode).
  /// @author amaury darsch

  class Cipher : public Object {
  protected:
    /// the cipher name
    String  d_name;
    /// the cipher key
    Key     d_ckey;
    /// the cipher block size
    long    d_cbsz;
    /// the reverse flag
    bool    d_rflg;

  public:
    /// create a cipher by name
    /// @param name the cipher name
    Cipher (const String& name);

    /// create a cipher by name and key
    /// @param name the cipher name
    /// @param key  the cipher key
    Cipher (const String& name, const Key& key);

    /// @return the class name
    String repr (void) const;

    /// reset this cipher
    virtual void reset (void);

    /// @return the cipher name
    virtual String getname (void) const;

    /// set the cipher key
    /// @param key the key to set
    virtual void setkey (const Key& key);

    /// @return the cipher key
    virtual Key getkey (void) const;

    /// set the cipher reverse flag
    /// @param rflg the flag to set
    virtual void setrflg (const bool rflg);

    /// @return the reverse flag
    virtual bool getrflg (void) const;
    
    /// @return the cipher block size
    virtual long getcbsz (void) const;

    /// get the normal waist from a file size
    /// @param size the size to normalize
    virtual t_long waist (const t_long size) const;

    /// process an input buffer into another buffer
    /// @param bo the output buffer
    /// @param bi the input buffer
    virtual void process (t_byte* bo, const t_byte* bi) =0;

    /// stream an input stream into an output stream
    /// @param os the output stream to write
    /// @param is the input stream to read
    /// @return the number of processed bytes
    virtual t_long stream (Output& os , Input& is);

  private:
    // make the copy constructor private
    Cipher (const Cipher&);
    // make the assignment operator private
    Cipher& operator = (const Cipher&);

  public:
    /// @return true if the given quark is defined
    bool isquark (const long quark, const bool hflg) const;

    /// apply thsi object with a set of arguments and a quark
    /// @param robj  the current runnable
    /// @param nset  the current nameset    
    /// @param quark the quark to apply these arguments
    /// @param argv  the arguments to apply
    Object* apply (Runnable* robj, Nameset* nset, const long quark,
		   Vector* argv);
  };
}

#endif
