# Global ruby extensions preloaded before execution.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

module Kernel
  private
  def subclass_responsibility
    raise NotImplementedError, self.class.name
  end

  def print_elapsed_time(io)
    tms = Process.times
    io.print "  %.3fs user, %.3fs system, " % [tms.utime, tms.stime]
    total = tms.utime + tms.stime
    h = total / 3600
    m = total / 60 % 60
    s = (total % 60).floor
    io.puts "%02d:%02d:%02d.%02d total" % [h, m, s, ((total % 60) - s) * 100]
  end

  def verbose=(verbose)
    # NOTE: The meaning of this verbose flag is not same as one of $VERBOSE.
    @@verbose = verbose
  end

  def verbose?
    @@verbose
  end
end

class Module
  # NOTE: Module.private_constant is added in Ruby 1.9.3-p0.
  unless public_instance_methods.include?(:private_constant)
    def private_constant(*name) end
  end
end

class Encoding
  def self.include_name?(enc_name)
    if enc_name && !enc_name.empty?
      Encoding.find(enc_name)
    end
    true
  rescue
    false
  end
end

class String
  def to_default_external
    encode(Encoding.default_external, :invalid => :replace, :undef => :replace)
  end
end

class LazyInstantiation
  def initialize(klass, *args, &block)
    @klass = klass
    @args = args
    @block = block
  end

  undef_method(*(Object.new.public_methods - [:__id__, :object_id, :__send__]))

  def method_missing(id, *args)
    @receiver ||= @klass.new(*@args, &@block)
    @receiver.__send__(id, *args)
  end
end

class Class
  def lazy_new(*args, &block)
    LazyInstantiation.new(self, *args, &block)
  end
end

class Integer
  # NOTE: To restrict the bit-shift width.
  SHIFT_MAX_BITS = 64

  def logical_right_shift(rhs)
    return self if rhs < 0
    bits = to_s(2).split("")
    shift_width = [rhs, SHIFT_MAX_BITS].min
    bits.length < shift_width ? 0 : bits[-1..(-shift_width - 1)].join.to_i(2)
  end

  def arithmetic_right_shift(rhs)
    rhs < 0 ? self : self >> [rhs, SHIFT_MAX_BITS].min
  end

  def left_shift(rhs)
    rhs < 0 ? self : self << [rhs, SHIFT_MAX_BITS].min
  end
end

class Symbol
  INVERSION_TABLE = {
    :== => :!=, :!= => :==, :< => :>=, :> => :<=, :<= => :>, :>= => :<
  }

  def invert
    if result = INVERSION_TABLE[self]
      result
    else
      self
    end
  end
end

class Pathname
  def components
    self.each_filename.to_a
  end

  def strip(num = 0)
    comps = self.components
    if num >= 0 && num < comps.size
      comps = comps.slice(num..-1)
    end

    Pathname.new(comps.reduce { |stripped, comp| File.join(stripped, comp) })
  end

  def add_ext(ext_str)
    Pathname.new(self.to_s + ext_str)
  end
end

class String
  # === DESCRIPTION
  # Finds the longest common substrings in two strings.
  #
  # Fast and small memory footprint clone detection algorithm developed by
  # Yutaka Yanoh.
  # This algorithm is based on "suffix array with height" data structure.
  #
  # === PARAMETER
  # _rhs_:: String -- A String comparing to the receiver.
  #
  # === RETURN VALUE
  # Array< SubstringPair > -- The longest common substrings.
  def longest_common_substrings_with(rhs)
    suffix_array = SuffixArray.new(LeftSuffix.of(self) + RightSuffix.of(rhs))
    suffix_array.longest_common_substrings
  end
end

class Substring < String
  def initialize(str, index, length)
    @range = index...(index + length)
    super(str.slice(@range))
  end

  attr_reader :range
end

class SubstringPair < Array
  def initialize(lhs_suffix, rhs_suffix, length)
    super([lhs_suffix.prefix(length), rhs_suffix.prefix(length)])
  end

  def lhs
    self.first
  end

  def rhs
    self.last
  end
end

class Suffix
  include Comparable

  def self.of(str)
    str.length.times.map { |index| new(str, index) }
  end

  def initialize(owner, index)
    @owner = owner
    @index = index
  end
  private_class_method :new

  def lhs?
    subclass_responsibility
  end

  def rhs?
    !lhs?
  end

  def same_owner?(rhs)
    @owner.equal?(rhs.owner)
  end

  def common_prefix_length(rhs)
    to_s.chars.zip(rhs.to_s.chars).take_while { |lch, rch| lch == rch }.size
  end

  def prefix(length)
    Substring.new(@owner, @index, length)
  end

  def <=>(rhs)
    self.to_s <=> rhs.to_s
  end

  def to_s
    @owner.slice(@index..-1)
  end

  protected
  attr_reader :owner
end

class LeftSuffix < Suffix
  def lhs?
    true
  end
end

class RightSuffix < Suffix
  def lhs?
    false
  end
end

class SuffixArray < Array
  def initialize(suffixes)
    super(suffixes.sort.map { |suffix| [suffix, 0] })
    update_height_of_each_suffixes!
  end

  def longest_common_substrings
    length = self.longest_common_prefix_length

    return [] if length == 0

    self.each_index.reduce([]) { |result, index|
      self[index][1] == length ?
        result + create_substring_pairs(index, length) : result
    }
  end

  def longest_common_prefix_length
    self.map { |suffix, height| height }.max
  end

  private
  def update_height_of_each_suffixes!
    (1...self.size).each do |index|
      self[index][1] = self[index - 1][0].common_prefix_length(self[index][0])
    end
  end

  def create_substring_pairs(index, length)
    base_suffix = self[index][0]

    create_entry = base_suffix.lhs? ?
      lambda { |lhs, rhs, len| SubstringPair.new(lhs, rhs, len) } :
      lambda { |rhs, lhs, len| SubstringPair.new(lhs, rhs, len) }

    result = []

    (0...index).reverse_each do |i|
      break unless self[i + 1][1] == length

      unless base_suffix.same_owner?(self[i][0])
        result.push(create_entry[base_suffix, self[i][0], length])
      end
    end

    result
  end
end

if $0 == __FILE__
  h1 = Hash.lazy_new(1)
  p h1[0]
  h2 = Hash.lazy_new { |h, k| h[k] = 1 }
  p h2[0]

  substrs = "foohogebargeho".longest_common_substrings_with("hogehoge")
  p substrs # => [["geho", "geho"], ["hoge", "hoge"], ["hoge", "hoge"]]
  p substrs.map { |l, r| l.range } # => [10...14, 3...7, 3...7]
  p substrs.map { |l, r| r.range } # => [2...6, 4...8, 0...4]
end
