# AdLint specific runtime error classes.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"

module AdLint #:nodoc:

  # == DESCRIPTION
  # Error base class for AdLint specific errors.
  class Error < StandardError
    # === DESCRIPTION
    # Constructs an error object.
    #
    # === PARAMETER
    # _message_:: String -- Error message.
    # _location_:: Location -- Location where an error occured.
    # _parts_:: Array< Object > -- Message formatting values.
    def initialize(message, location = nil, *parts)
      super(message)
      @location = location
      @parts = parts
    end

    # === VALUE
    # Location -- The location where this error occured.
    attr_reader :location

    # === VALUE
    # Array -- Message formatting values.
    attr_reader :parts

    # === DESCRIPTION
    # Reads the message ID of this error.
    #
    # Subclasses must implement this method.
    #
    # === RETURN VALUE
    # Symbol -- Message ID.
    def id
      subclass_responsibility
    end
  end

  # == DESCRIPTION
  # \AdLint fatal internal error.
  class InternalError < Error
    # === DESCRIPTION
    # Constructs an AdLint specific fatal internal error object.
    #
    # === PARAMETER
    # _cause_:: Exception -- Cause exception object.
    # _location_:: Location -- Location where an error occured.
    def initialize(cause, location)
      @cause = cause
      super(cause.message, location, "#{cause.class} : #{cause.message}")
    end

    # === DESCRIPTION
    # Reads the message ID of this error.
    #
    # === RETURN VALUE
    # Symbol -- Message ID.
    def id
      :X0001
    end

    # === DESCRIPTION
    # Reads the message of this error.
    #
    # === RETURN VALUE
    # String -- Error message.
    def message
      @cause.message
    end

    # === DESCRIPTION
    # Reads the backtrace information of this error.
    #
    # === RETURN VALUE
    # Array -- Backtrace information.
    def backtrace
      @cause.backtrace
    end
  end

  # == DESCRIPTION
  # Internal error which indicates invalid message ID used.
  class InvalidMessageIdError < Error
    # === DESCRIPTION
    # Constructs a invalid message ID error.
    #
    # === PARAMETER
    # _id_:: Symbol -- Invalid message ID.
    # _location_:: Location -- Location where the error occured.
    def initialize(id, location = nil)
      super(id, location, id)
    end

    # === DESCRIPTION
    # Reads the message ID of this error.
    #
    # === RETURN VALUE
    # String -- Message ID.
    def id
      :X0002
    end
  end

  class InvalidMessageFormatError < Error
    def initialize(id, location = nil)
      super(id, location, id)
    end

    def id
      :X0004
    end
  end

  class FatalError < Error
    def id
      :X0003
    end
  end

  class ParseError < FatalError
    def initialize(location, msg_fpath, log_fpath)
      super("failed to parse preprocessed file.", location,
            msg_fpath, log_fpath)
    end
  end

  class MissingUserHeaderError < FatalError
    def initialize(location, basename, msg_fpath, log_fpath)
      super("cannot open \"#{basename}\".", location, msg_fpath, log_fpath)
    end
  end

  class MissingSystemHeaderError < FatalError
    def initialize(location, basename, msg_fpath, log_fpath)
      super("cannot open <#{basename}>.", location, msg_fpath, log_fpath)
    end
  end

  class UnterminatedCommentError < FatalError
    def initialize(location, msg_fpath, log_fpath)
      super("unterminated comment block found.",
            location, msg_fpath, log_fpath)
    end
  end

end
