/*
 * This file is part of ACGVision, SaaS system-monitoring software
 * Copyright (C) 2009 ACGCenter
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package com.acgvision.agent.action;

import com.acgvision.core.ws.Period;
import com.acgvision.core.ws.Production;
import com.acgvision.core.ws.Service;
import com.acgvision.core.ws.Week;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Iterator;
import org.apache.log4j.Logger;

/**
 *
 * @author Rémi Debay <remi.debay@acgcenter.com>
 */
public class ServiceActions {

    private static final Logger logger = Logger.getLogger(HostActions.class);

    public static boolean validate(Service service, final Date date) {
        boolean result = validate(service.getProduction(), date);
        Iterator<Service> it = service.getFathers().iterator();
        while (it.hasNext() && result) {
            result = validate(it.next(), date);
        }
        return result;
    }

    /**
     * Checks if the date is valid to control
     * @param production
     * @param date
     * @return true if the control is allowed
     */
    public static boolean validate(Production production, final Date date) {
        if (date == null) {
            logger.error("Date not set.");
            return false;
        }
        if (production == null) {
            return true;
        }
        GregorianCalendar asked = new GregorianCalendar();
        asked.setTime(date);
        if(!validate(production.getWeek(), asked.get(GregorianCalendar.DAY_OF_WEEK))) {
            logger.info("Paused weekday.");
            return false;
        }

        Period start = production.getStarting();
        Period end = production.getEndding();
        boolean p = compare(start, end);// p = start <end
        boolean f = compare(end, asked.getTime());//f = end<asked
        boolean d = compare(start, asked.getTime());//d = start<asked
        return ((p && !f && d) || (!p && !f && !d) || !p && f && d);
    }

    /**
     * Checks if the control can occur according to the day of the week.
     * @param w
     * @param day
     * @return true if the day is valid or if week is not set
     */
    private static boolean validate(final Week w,final int day) {
        if (w ==null )return true;
        if (day == GregorianCalendar.MONDAY) {
            return w.isMonday();
        }
        if (day == GregorianCalendar.TUESDAY) {
            return w.isTuesday();
        }
        if (day == GregorianCalendar.WEDNESDAY) {
            return w.isWednesday();
        }
        if (day == GregorianCalendar.THURSDAY) {
            return w.isThursday();
        }
        if (day == GregorianCalendar.FRIDAY) {
            return w.isFriday();
        }
        if (day == GregorianCalendar.SATURDAY) {
            return w.isSaturday();
        }
        if (day == GregorianCalendar.SUNDAY) {
            return w.isSunday();
        }
        logger.error("Invalid day code:" + day);
        return false;
    }

    public static boolean compare(Period p1, Date date) {
        if(p1 ==null ) return false;
        if(date==null) return false;
        GregorianCalendar d = new GregorianCalendar();
        d.setTime(date);
        Period p2 = new Period();
        p2.setYear(d.get(GregorianCalendar.YEAR));
        p2.setMonth(d.get(GregorianCalendar.MONTH));
        p2.setDay(d.get(GregorianCalendar.DAY_OF_MONTH));
        p2.setHour(d.get(GregorianCalendar.HOUR_OF_DAY));
        p2.setMinute(d.get(GregorianCalendar.MINUTE));
        return compare(p1, p2);
    }

    /**
     * compares two period and says which one is before the other one.
     * @param p1
     * @param p2
     * @return true if (p1<p2 ) false if p1>=p2
     */
    public static boolean compare(Period p1, Period p2) {
        if (p1 == null) {
            return false;
        }
        if (p2 == null) {
            return false;
        }
        //comparaison des années
        if (p1.getYear() != -1 && p2.getYear() != -1) {
            if (p1.getYear() < p2.getYear()) {
                return true;
            }
            if (p1.getYear() > p2.getYear()) {
                return false;
            }
        }

        if (p1.getMonth() != -1 && p2.getMonth() != -1) {//comparaison des mois
            if (p1.getMonth() < p2.getMonth()) {
                return true;
            } else if (p1.getMonth() > p2.getMonth()) {
                return false;
            }
        }

        if (p1.getDay() != -1 && p2.getDay() != -1) {//comparaisons des jours
            if (p1.getDay() < p2.getDay()) {
                return true;
            } else if (p1.getDay() > p2.getDay()) {
                return false;
            }
        }
        if (p1.getHour() != -1 && p2.getHour() != -1) {//comparaisons des heures
            if (p1.getHour() < p2.getHour()) {
                return true;
            } else if (p1.getHour() > p2.getHour()) {
                return false;
            }
        }
        if (p1.getMinute() != -1 && p2.getMinute() != -1) {//comparaison des minutes
            if (p1.getMinute() < p2.getMinute()) {
                return true;
            } else if (p1.getMinute() > p2.getMinute()) {
                return false;
            }
        }
        //finnalement c'est égal, on retourne faux
        return false;
    }

    /**
     * Gets next time to do the job starting from now.
     * @param p
     * @return time before next checking
     */
    public static long getNextTime(final Period p) {
        if (p == null) {
            return -1;
        }
        return getNextTime(p, new GregorianCalendar());
    }

    /**
     * Gets next time to do the job starting from now.
     * @param p periodicity of the control
     * @param lastControl occurency
     * @return -1 on errror or date in Milli seconds
     */
    public static long getNextTime(final Period p, final GregorianCalendar lastControl) {
        if (p == null) {
            return -1;
        }
        if (lastControl == null) {
            return -1;
        }

        GregorianCalendar gc = new GregorianCalendar();
        if (p.getYear() > 2000) {//une date <2000 n'a plus de sens !!!!
            //C'est une date précise !            
            gc.set(p.getYear(), p.getMonth(), p.getDay(), p.getHour(), p.getMinute());
            return gc.getTimeInMillis();
        }
        gc = (GregorianCalendar) lastControl.clone();
        //on génère la prochaine date
        //D'abord on applique le masque, si c'est dans le passé, alors on ajoute la périodicité
        //Exemple aujourdhui 2009/03/13 17:42
        //Tous les 30 du mois à minuit
        //on applique le masque 2009/03/30 00:00 => ok c'est le futur
        //Tous les 01 du mois à minuit
        //on applique le masque 2009/03/01 00:00 => Nok c'est le passé
        //+1 mois 2009/04/01 00:00
        if (p.getYear() > 0) {//le controle se fait tous les ans à une date précise

            gc.set(GregorianCalendar.MONTH, p.getMonth());
            gc.set(GregorianCalendar.DAY_OF_MONTH, p.getDay());
            gc.set(GregorianCalendar.HOUR_OF_DAY, p.getHour());
            gc.set(GregorianCalendar.MINUTE, p.getMinute());
            while (gc.before(lastControl)) {
                gc.add(GregorianCalendar.YEAR, p.getYear());
            }
            //on recherche la prochaine date dans le futur modulo la périodicité
            while (!gc.after(lastControl)) {
                gc.add(GregorianCalendar.YEAR, 1);
            }
        } else if (p.getMonth() > 0) {//lecontrole se fait tous n les mois à une date précise
            gc.set(GregorianCalendar.DAY_OF_MONTH, p.getDay());
            gc.set(GregorianCalendar.HOUR_OF_DAY, p.getHour());
            gc.set(GregorianCalendar.MINUTE, p.getMinute());
            while (!gc.after(lastControl)) {
                gc.add(GregorianCalendar.MONTH, p.getMonth());
            }
        } else if (p.getDay() > 0) {//le controle se fait tous n les jours
            gc.set(GregorianCalendar.HOUR_OF_DAY, p.getHour());
            gc.set(GregorianCalendar.MINUTE, p.getMinute());
            while (!gc.after(lastControl)) {
                gc.add(GregorianCalendar.DAY_OF_MONTH, p.getDay());
            }
        } else if (p.getHour() > 0) {//le controle se fait toutes n les heures
            gc.set(GregorianCalendar.MINUTE, p.getMinute());
            while (!gc.after(lastControl)) {
                gc.add(GregorianCalendar.HOUR_OF_DAY, p.getHour());
            }
        } else if (p.getMinute() > 0) {//le contrôle se fait toutes n les minutes
            gc.add(GregorianCalendar.MINUTE, p.getMinute());
        } else {
            return -1;//il y a erreur
        }
        return gc.getTimeInMillis();
    }
}
