/* Logging.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.logging;

import be.ugent.caagt.swirl.logging.JListLogCellRenderer;
import be.ugent.caagt.swirl.logging.JListLogHandler;

import java.awt.Frame;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogManager;
import java.util.logging.LogRecord;
import javax.swing.Icon;
import javax.swing.JDialog;
import javax.swing.JList;
import javax.swing.JScrollPane;

import org.grinvin.gui.icons.SvgIconManager;
import org.grinvin.help.HelpManager;

/**
 * Installs the main log handler for the application.
 */
public final class Logging {
    
    // not to be instantiated
    private Logging() {}
    
    //
    private static LoggingWindow loggingWindow;
    
    /**
     * The main logging window of this application, or null
     * when the logging handler was not yet initialized.
     */
    public static LoggingWindow getLoggingWindow() {
        return loggingWindow;
    }
    
    /**
     * Initializes the logging handler for this application
     * if this was not already done before.
     * @param windowTitle Title to be used for the logging window.
     */
    public static void installHandler(String windowTitle) {
        if (loggingWindow == null) {
            loggingWindow = new LoggingWindow((Frame)null, windowTitle, false);
            JList list = new JList();
            Handler handler = new JListLogHandler(list);
            LogManager.getLogManager().getLogger("").addHandler(handler);
            loggingWindow.add(new JScrollPane(list));
            loggingWindow.pack();
            loggingWindow.setSize(500, 300); // reasonable default size
            list.setCellRenderer(new Renderer(handler.getFormatter()));
            HelpManager.setHelpIDString(loggingWindow, "org.grinvin.window.logging");
            HelpManager.enableHelpKey(loggingWindow.getRootPane(), "org.grinvin.window.logging");
            HelpManager.enableHelpKey(list, "org.grinvin.window.logging");
        }
    }
    
    /**
     * Private cell renderer which uses icons for some of the severity levels.
     */
    private static class Renderer extends JListLogCellRenderer {
        
        private static Icon severeIcon
                = SvgIconManager.getInstance().getIcon("/org/grinvin/logging/severe.svg", 16);
        
        private static Icon warningIcon
                = SvgIconManager.getInstance().getIcon("/org/grinvin/logging/warn.svg", 16);
        
        private static Icon infoIcon
                = SvgIconManager.getInstance().getIcon("/org/grinvin/logging/info.svg", 16);
        
        public Renderer(Formatter formatter) {
            super(formatter);
        }
        
        protected Icon toIcon(LogRecord record) {
            int val = record.getLevel().intValue();
            if (val < Level.INFO.intValue())
                return null;
            else if (val < Level.WARNING.intValue())
                return infoIcon;
            else if (val < Level.SEVERE.intValue())
                return warningIcon;
            else
                return severeIcon;
        }
        
    }
    
}
