/* GraphPropertiesWindow.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.gui.windows;

import java.awt.BorderLayout;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.util.ResourceBundle;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;

import org.grinvin.graphs.GraphURI;
import org.grinvin.gui.components.GraphPropertiesTableComponent;
import org.grinvin.gui.components.NamedGraphComponent;
import org.grinvin.worksheet.actions.InvalidateGraphBundleCacheAction;
import org.grinvin.worksheet.actions.RenameGraphListElementAction;
import org.grinvin.invariants.Invariant;
import org.grinvin.list.graphs.GraphListElement;
import org.grinvin.list.graphs.GraphListElementListener;
import org.grinvin.list.GraphPropertiesTableModel;
import org.grinvin.list.actions.DisposeWindowAction;

/**
 * Window that displays a drawing of the graph as well as its known invariants.
 */
public class GraphPropertiesWindow extends JFrame implements GraphListElementListener {
    
    //
    private final GraphListElement element;
    
    //
    private JSplitPane sp;
    
    //
    private JLabel uri;
    
    public GraphPropertiesWindow(GraphListElement element) {
        super (element.getName());
        this.element = element;
        
        //menu
        ResourceBundle bundle = ResourceBundle.getBundle("org.grinvin.list.actions.resources");
        JMenu file = new JMenu(bundle.getString("menu.file.text"));
        file.setMnemonic(bundle.getString("menu.file.mnemonic").charAt(0));
        file.add(new RenameGraphListElementAction(element));
        file.add(new InvalidateGraphBundleCacheAction(element));
        file.add(new DisposeWindowAction(this));
        JMenuBar mBar = new JMenuBar();
        mBar.add(file);
        setJMenuBar(mBar);
        
        JScrollPane propertiestable = new JScrollPane(new GraphPropertiesTableComponent(new GraphPropertiesTableModel(element)));
       
        NamedGraphComponent graph = new NamedGraphComponent(element);
        
        sp = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT, graph, propertiestable);
        sp.setOpaque(true);
        setBackground();
        getContentPane().add(sp, BorderLayout.CENTER);
        
        uri = new JLabel();
        uri.setHorizontalAlignment(SwingConstants.CENTER);
        setURI();
        getContentPane().add(uri, BorderLayout.NORTH);
        
        setDefaultCloseOperation(DISPOSE_ON_CLOSE);
        // make sure listeners are removed when window is 
        addWindowListener(new WindowAdapter() {
            // should guarantee that the listener is added again when the window is opened after
            // being dispose. (I know, we do not really intend to do this, but anyway...)
            public void windowOpened (WindowEvent ev) {
                GraphPropertiesWindow.this.element.addGraphListElementListener(GraphPropertiesWindow.this);
            }
            public void windowClosed (WindowEvent ev) {
                GraphPropertiesWindow.this.element.removeGraphListElementListener(GraphPropertiesWindow.this);
            }
        });
        pack();
    }
    
    private void setURI() {
        if (element.getURI() != null)
            uri.setText(element.getURI().toString());
    }
    
    private void setBackground() {
        sp.setBackground(GraphURI.getType(element.getURI()).getIconBackgroundColor());
    }
    
    public GraphListElement getModel() {
        return element;
    }
    
    public void graphListElementURIChanged(GraphListElement element) {
        setURI();
        setBackground();
    }

    public void graphListElementNameChanged(GraphListElement element) {
        setTitle(element.getName());
    }

    public void graphListElementInvariantComputed(GraphListElement element, Invariant invariant) {
        //ignore
    }

    public void graphListElementGraphChanged(GraphListElement element) {
        //ignore
    }
    
}
