/* BooleanValueGraphFilter.java
 * =========================================================================
 * This file is part of the GrInvIn project - http://www.grinvin.org
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

package org.grinvin.conjecture.filter;

import org.grinvin.graphs.GraphBundle;
import org.grinvin.graphs.GraphBundleView;
import org.grinvin.invariants.Invariant;
import org.grinvin.invariants.InvariantManager;
import org.grinvin.invariants.UnknownInvariantException;
import org.grinvin.invariants.values.BooleanValue;
import org.jdom.Element;

/**
 * {@link GraphFilter} that filters graphs based on the value of an invariant that returns a {@link BooleanValue}.
 */
public class BooleanValueGraphFilter extends AbstractGraphFilter {
    
    private Invariant invariant;
    
    /** Creates a new instance of BooleanValueGraphFilter */
    BooleanValueGraphFilter() {
        // to be used by GraphFilterManager
    }
    
    public BooleanValueGraphFilter(Invariant invariant) {
        this(invariant, false);
    }
    
    public BooleanValueGraphFilter(Invariant invariant, boolean negate) {
        super(negate);
        if(invariant != null && invariant.getType().equals(BooleanValue.class))
            this.invariant = invariant;
    }
    
    @Override
    public boolean accepts(GraphBundleView gb) {
        if(invariant==null)
            return true;
        else
            return super.accepts(gb);
    }
    
    @Override
    public String getDescription() {
        if(invariant==null)
            return "";
        else
            return super.getDescription();
    }
    
    public Element toElement() {
        String invariantID = invariant == null ? "null" : invariant.getId();
        return new Element("BooleanFilter").setAttribute("invariant", invariantID).setAttribute("negate", Boolean.toString(isNegated()));
    }
    
    public void fromElement(Element element) {
        if(element.getName().equalsIgnoreCase("BooleanFilter")){
            if(element.getAttributeValue("invariant").equals("null"))
                invariant = null;
            else
                try {
                    invariant = InvariantManager.getInstance().getInvariant(element.getAttributeValue("invariant"));
                } catch (UnknownInvariantException ex) {
                    invariant = null;
                }
            negate(Boolean.parseBoolean(element.getAttributeValue("negate")));
        }
    }

    public boolean accepts_impl(GraphBundleView gb) {
            try {
                return ((BooleanValue)InvariantManager.getInstance().getInvariantComputerFor(invariant).compute(gb)).getValue();
            } catch (Exception ex) {
                return true; //when error: accept graph (is this correct behaviour?)
            }
    }

    public String getDescription_impl() {
        return invariant.getName();
    }
    
}
